/*
 * simple quicktime movie player, needs libquicktime (or quicktime4linux).
 *
 *  (c) 2002 Gerd Knorr <kraxel@bytesex.org>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <strings.h>
#include <errno.h>
#include <fcntl.h>

#include <sys/types.h>
#include <sys/time.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>

#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Simple.h>
#include <X11/extensions/XShm.h>
#include <X11/extensions/Xv.h>
#include <X11/extensions/Xvlib.h>

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>

#include <avilib.h>
#include <qtime.h>

#include "csp.h"
#include "vcodec.h"
#include "acodec.h"

#define FOURCC_YV12 0x32315659  /* YV12   YUV420P */
#define FOURCC_YUV2 0x32595559  /* YUV2   YUV422 */
#define FOURCC_I420 0x30323449  /* I420   IntelIndeo 4 */

/* ------------------------------------------------------------------------ */
/* X11 code                                                                 */

static XtAppContext app_context;
static Widget       app_shell;
static Display      *dpy;
static Visual       *visual;
static XVisualInfo  vinfo,*vinfo_list;
static XPixmapFormatValues *pf;

static Widget simple;
static Dimension swidth,sheight;

static int xv_port      = 0;
static int xv_have_YUY2 = 0;
static int xv_have_I420 = 0;
static int xv_have_YV12 = 0;

static int no_mitshm    = 0;
static int pixmap_bytes = 0;
static int x11_byteswap = 0;
static int use_gl       = 0;

static unsigned long   lut_red[256];
static unsigned long   lut_green[256];
static unsigned long   lut_blue[256];

#define SWAP2(x) (((x>>8) & 0x00ff) |\
                  ((x<<8) & 0xff00))

#define SWAP4(x) (((x>>24) & 0x000000ff) |\
                  ((x>>8)  & 0x0000ff00) |\
                  ((x<<8)  & 0x00ff0000) |\
                  ((x<<24) & 0xff000000))

static void
x11_lut(unsigned long red_mask, unsigned long green_mask,
	unsigned long blue_mask, int bytes, int swap)
{
    int             rgb_red_bits = 0;
    int             rgb_red_shift = 0;
    int             rgb_green_bits = 0;
    int             rgb_green_shift = 0;
    int             rgb_blue_bits = 0;
    int             rgb_blue_shift = 0;
    unsigned int    i;
    unsigned int    mask;

    for (i = 0; i < 32; i++) {
        mask = (1 << i);
        if (red_mask & mask)
            rgb_red_bits++;
        else if (!rgb_red_bits)
            rgb_red_shift++;
        if (green_mask & mask)
            rgb_green_bits++;
        else if (!rgb_green_bits)
            rgb_green_shift++;
        if (blue_mask & mask)
            rgb_blue_bits++;
        else if (!rgb_blue_bits)
            rgb_blue_shift++;
    }
    if (rgb_red_bits > 8)
	for (i = 0; i < 256; i++)
	    lut_red[i] = (i << (rgb_red_bits + rgb_red_shift - 8));
    else
	for (i = 0; i < 256; i++)
	    lut_red[i] = (i >> (8 - rgb_red_bits)) << rgb_red_shift;
    if (rgb_green_bits > 8)
	for (i = 0; i < 256; i++)
	    lut_green[i] = (i << (rgb_green_bits + rgb_green_shift - 8));
    else
	for (i = 0; i < 256; i++)
	    lut_green[i] = (i >> (8 - rgb_green_bits)) << rgb_green_shift;
    if (rgb_blue_bits > 8)
	for (i = 0; i < 256; i++)
	    lut_blue[i] = (i << (rgb_blue_bits + rgb_blue_shift - 8));
    else
	for (i = 0; i < 256; i++)
	    lut_blue[i] = (i >> (8 - rgb_blue_bits)) << rgb_blue_shift;

    if (2 == bytes && swap) {
	for (i = 0; i < 256; i++) {
	    lut_red[i] = SWAP2(lut_red[i]);
	    lut_green[i] = SWAP2(lut_green[i]);
	    lut_blue[i] = SWAP2(lut_blue[i]);
	}
    }
    if (4 == bytes && swap) {
	for (i = 0; i < 256; i++) {
	    lut_red[i] = SWAP4(lut_red[i]);
	    lut_green[i] = SWAP4(lut_green[i]);
	    lut_blue[i] = SWAP4(lut_blue[i]);
	}
    }
}

static void
rgb_to_lut2(unsigned char *dest, unsigned char *src, int p)
{
    unsigned short *d = (unsigned short*)dest;

    while (p-- > 0) {
	*(d++) = lut_red[src[2]] | lut_green[src[1]] | lut_blue[src[0]];
	src += 3;
    }
}

static void
rgb_to_lut4(unsigned char *dest, unsigned char *src, int p)
{
    unsigned int *d = (unsigned int*)dest;

    while (p-- > 0) {
      *(d++) = *((unsigned int*)src);
      src += 3;
    }
#if 0
    while (p-- > 0) {
	*(d++) = lut_red[src[0]] | lut_green[src[1]] | lut_blue[src[2]];
	src += 3;
    }
#endif
}

static void x11_init(void)
{
    int i,n;
    
    /* get visual info */
    visual = DefaultVisualOfScreen(XtScreen(app_shell));
    vinfo.visualid = XVisualIDFromVisual(visual);
    vinfo_list = XGetVisualInfo(dpy, VisualIDMask, &vinfo, &n);
    vinfo = vinfo_list[0];
    if (vinfo.class != TrueColor || vinfo.depth < 15) {
	fprintf(stderr,"can't handle visuals != TrueColor, sorry\n");
	exit(1);
    }

    /* look for default pixmap format */
    pf = XListPixmapFormats(dpy,&n);
    for (i = 0; i < n; i++)
	if (pf[i].depth == vinfo.depth)
	    pixmap_bytes = pf[i].bits_per_pixel/8;

    /* byteswapping needed ??? */
    if (ImageByteOrder(dpy)==LSBFirst && BYTE_ORDER!=LITTLE_ENDIAN)
	x11_byteswap=1;
    if (ImageByteOrder(dpy)==MSBFirst && BYTE_ORDER!=BIG_ENDIAN)
	x11_byteswap=1;

    /* init lookup tables */
    x11_lut(vinfo.red_mask, vinfo.green_mask, vinfo.blue_mask,
	    pixmap_bytes,x11_byteswap);
}

static void xv_init(void)
{
    int ver, rel, req, ev, err, i;
    int adaptors, formats;
    XvAdaptorInfo        *ai;
    XvImageFormatValues  *fo;
    
    if (Success != XvQueryExtension(dpy,&ver,&rel,&req,&ev,&err))
	return;

    /* find + lock port */
    if (Success != XvQueryAdaptors(dpy,DefaultRootWindow(dpy),&adaptors,&ai))
	return;
    for (i = 0; i < adaptors; i++) {
	if ((ai[i].type & XvInputMask) && (ai[i].type & XvImageMask)) {
	    if (Success != XvGrabPort(dpy,ai[i].base_id,CurrentTime)) {
		fprintf(stderr,"INFO: Xvideo port %ld: is busy, skipping\n",
			ai[i].base_id);
		continue;
	    }
	    xv_port = ai[i].base_id;
	    break;
	}
    }
    if (0 == xv_port)
	return;

    /* check image formats */
    fo = XvListImageFormats(dpy, xv_port, &formats);
    for(i = 0; i < formats; i++) {
	fprintf(stderr, "INFO: Xvideo port %d: 0x%x (%c%c%c%c) %s",
		xv_port,
		fo[i].id,
		(fo[i].id)       & 0xff,
		(fo[i].id >>  8) & 0xff,
		(fo[i].id >> 16) & 0xff,
		(fo[i].id >> 24) & 0xff,
		(fo[i].format == XvPacked) ? "packed" : "planar");
	if (FOURCC_YUV2 == fo[i].id) {
	    fprintf(stderr," [BC_YUV422]");
	    xv_have_YUY2 = 1;
	}
        if (FOURCC_YV12 == fo[i].id) {
            fprintf(stderr," [BC_YUV420P]");
            xv_have_YV12 = 1;
        }
	if (FOURCC_I420 == fo[i].id) {
	    fprintf(stderr," [BC_YUV420P]");
            xv_have_I420 = 1;
	}
	fprintf(stderr,"\n");
    }
}

static int
catch_no_mitshm(Display * dpy, XErrorEvent * event)
{
    no_mitshm++;
    return 0;
}

static XImage*
x11_create_ximage(Display *dpy, int width, int height)
{
    XImage          *ximage = NULL;
    unsigned char   *ximage_data;
    XShmSegmentInfo *shminfo = NULL;
    void            *old_handler;
    
    if (no_mitshm)
	goto no_mitshm;
    
    old_handler = XSetErrorHandler(catch_no_mitshm);
    shminfo = malloc(sizeof(XShmSegmentInfo));
    memset(shminfo, 0, sizeof(XShmSegmentInfo));
    ximage = XShmCreateImage(dpy,vinfo.visual,vinfo.depth,
			     ZPixmap, NULL,
			     shminfo, width, height);
    if (NULL == ximage)
	goto shm_error;
    shminfo->shmid = shmget(IPC_PRIVATE,
			    ximage->bytes_per_line * ximage->height,
			    IPC_CREAT | 0777);
    if (-1 == shminfo->shmid) {
	perror("shmget");
	goto shm_error;
    }
    shminfo->shmaddr = (char *) shmat(shminfo->shmid, 0, 0);
    if ((void *)-1 == shminfo->shmaddr) {
	perror("shmat");
	goto shm_error;
    }
    ximage->data = shminfo->shmaddr;
    shminfo->readOnly = False;
    
    XShmAttach(dpy, shminfo);
    XSync(dpy, False);
    if (no_mitshm)
	goto shm_error;
    shmctl(shminfo->shmid, IPC_RMID, 0);
    XSetErrorHandler(old_handler);
    return ximage;

shm_error:
    if (ximage) {
	XDestroyImage(ximage);
	ximage = NULL;
    }
    if ((void *)-1 != shminfo->shmaddr  &&  NULL != shminfo->shmaddr)
	shmdt(shminfo->shmaddr);
    free(shminfo);
    XSetErrorHandler(old_handler);
    no_mitshm = 1;

 no_mitshm:
    if (NULL == (ximage_data = malloc(width * height * pixmap_bytes))) {
	fprintf(stderr,"out of memory\n");
	exit(1);
    }
    ximage = XCreateImage(dpy, vinfo.visual, vinfo.depth,
			  ZPixmap, 0, ximage_data,
			  width, height,
			  8, 0);
    memset(ximage->data, 0, ximage->bytes_per_line * ximage->height);
    return ximage;
}

static XvImage*
xv_create_ximage(Display *dpy, int width, int height, int port, int format)
{
    XvImage         *xvimage = NULL;
    unsigned char   *ximage_data;
    XShmSegmentInfo *shminfo = NULL;
    void            *old_handler;
    
    if (no_mitshm)
	goto no_mitshm;
    
    old_handler = XSetErrorHandler(catch_no_mitshm);
    shminfo = malloc(sizeof(XShmSegmentInfo));
    memset(shminfo, 0, sizeof(XShmSegmentInfo));
    xvimage = XvShmCreateImage(dpy, port, format, 0,
			       width, height, shminfo);
    if (NULL == xvimage)
	goto shm_error;
    shminfo->shmid = shmget(IPC_PRIVATE, xvimage->data_size,
			    IPC_CREAT | 0777);
    if (-1 == shminfo->shmid) {
	perror("shmget");
	goto shm_error;
    }
    shminfo->shmaddr = (char *) shmat(shminfo->shmid, 0, 0);
    if ((void *)-1 == shminfo->shmaddr) {
	perror("shmat");
	goto shm_error;
    }
    xvimage->data = shminfo->shmaddr;
    shminfo->readOnly = False;
    
    XShmAttach(dpy, shminfo);
    XSync(dpy, False);
    if (no_mitshm)
	goto shm_error;
    shmctl(shminfo->shmid, IPC_RMID, 0);
    XSetErrorHandler(old_handler);
    return xvimage;

shm_error:
    if (xvimage) {
	XFree(xvimage);
	xvimage = NULL;
    }
    if ((void *)-1 != shminfo->shmaddr  &&  NULL != shminfo->shmaddr)
	shmdt(shminfo->shmaddr);
    free(shminfo);
    XSetErrorHandler(old_handler);
    no_mitshm = 1;

 no_mitshm:
    if (NULL == (ximage_data = malloc(width * height * 2))) {
	fprintf(stderr,"out of memory\n");
	exit(1);
    }
    xvimage = XvCreateImage(dpy, port, format, ximage_data,
			    width, height);
    return xvimage;
}

static void x11_blit(Window win, GC gc, XImage *xi, int width, int height)
{
    if (no_mitshm)
	XPutImage(dpy,win,gc,xi, 0,0,0,0, width,height);
    else
//	XShmPutImage(dpy,win,gc,xi, 0,0,0,0, width,height, True);
	XShmPutImage(dpy,win,gc,xi, 0,0,0,0, width,height, False);
}

static void xv_blit(Window win, GC gc, XvImage *xi,
		    int iw, int ih, int ww, int wh)
{
    if (no_mitshm)
	XvPutImage(dpy,xv_port,win,gc,xi, 0,0,iw,ih, 0,0,ww,wh);
    else
//	XvShmPutImage(dpy,xv_port,win,gc,xi, 0,0,iw,ih, 0,0,ww,wh, True);
	XvShmPutImage(dpy,xv_port,win,gc,xi, 0,0,iw,ih, 0,0,ww,wh, False);
}

/* ------------------------------------------------------------------------ */
/* OpenGL code                                                              */

static int tw,th;
static GLint tex;
static int gl_attrib[] = { GLX_RGBA,
			   GLX_RED_SIZE, 1,
			   GLX_GREEN_SIZE, 1,
			   GLX_BLUE_SIZE, 1,
			   GLX_DOUBLEBUFFER,
			   None };

static void gl_resize(Widget widget, int width, int height)
{
    glViewport(0, 0, width, height);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluOrtho2D(0.0, width, 0.0, height);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

}

static void gl_blit(Widget widget, char *rgbbuf,
		    int iw, int ih, int ww, int wh)
{
    char *dummy;
    float x,y;

    if (0 == tex) {
	glGenTextures(1,&tex);
	glBindTexture(GL_TEXTURE_2D,tex);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	dummy = malloc(tw*th*3);
	memset(dummy,128,tw*th*3);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB,tw,th,0,
		     GL_RGB,GL_UNSIGNED_BYTE,dummy);
	free(dummy);
    }
    glTexSubImage2D(GL_TEXTURE_2D, 0, 0,0,iw,ih,
		    GL_RGB,GL_UNSIGNED_BYTE,rgbbuf);
    x = (float)iw/tw;
    y = (float)ih/th;

    glEnable(GL_TEXTURE_2D);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
    glBegin(GL_QUADS);
    glTexCoord2f(0,y);  glVertex3f(0,0,0);
    glTexCoord2f(0,0);  glVertex3f(0,wh,0);
    glTexCoord2f(x,0);  glVertex3f(ww,wh,0);
    glTexCoord2f(x,y);  glVertex3f(ww,0,0);
    glEnd();
    glXSwapBuffers(XtDisplay(widget), XtWindow(widget));
    glDisable(GL_TEXTURE_2D);
}

static void gl_init(Widget widget, int iw, int ih)
{
    XVisualInfo *visinfo;
    GLXContext ctx;
    int i = 0;

    visinfo = glXChooseVisual(XtDisplay(widget),
			      DefaultScreen(XtDisplay(widget)),
			      gl_attrib);
    if (!visinfo) {
	fprintf(stderr,"WARNING: gl: can't get visual (rgb,db)\n");
	return;
    }
    ctx = glXCreateContext(dpy, visinfo, NULL, True);
    glXMakeCurrent(XtDisplay(widget),XtWindow(widget),ctx);
    fprintf(stderr, "INFO: gl: DRI=%s\n",
	    glXIsDirect(dpy, ctx) ? "Yes" : "No");
    if (!glXIsDirect(dpy, ctx))
	return;
    
    /* check against max size */
    glGetIntegerv(GL_MAX_TEXTURE_SIZE,&i);
    fprintf(stderr,"INFO: gl: texture max size: %d\n",i);
    if (iw > i)
	return;
    if (ih > i)
	return;
    
    /* textures have power-of-two x,y dimensions */
    for (i = 0; iw >= (1 << i); i++)
	;
    tw = (1 << i);
    for (i = 0; ih >= (1 << i); i++)
	;
    th = (1 << i);
    fprintf(stderr,"INFO: gl: frame=%dx%d, texture=%dx%d\n",iw,ih,tw,th);

    glClearColor (0.0, 0.0, 0.0, 0.0);
    glShadeModel(GL_FLAT);
    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
    use_gl = 1;
}

/* ------------------------------------------------------------------------ */
/* oss code                                                                 */

#ifndef AFMT_S16_NE
# if BYTE_ORDER == BIG_ENDIAN
#  define AFMT_S16_NE AFMT_S16_BE
# else
#  define AFMT_S16_NE AFMT_S16_LE
# endif
#endif

static int oss_fd = -1;
static int oss_sr,oss_hr;
static int rate, channels, bits;
static int oss_out_delay;
static int oss_fragments, oss_fragsize, oss_fragstotal;

static int
oss_setformat(int chan, int rate)
{
    int hw_afmt = AFMT_S16_NE;
    int hw_chan = chan;
    int hw_rate = rate;
    audio_buf_info abinfo;

    ioctl(oss_fd, SNDCTL_DSP_SETFMT, &hw_afmt);
    if (AFMT_S16_LE != hw_afmt) {
	fprintf(stderr,"ERROR: can't set sound format\n");
	exit(1);
    }
    ioctl(oss_fd, SNDCTL_DSP_CHANNELS, &hw_chan);
    if (chan != hw_chan) {
	fprintf(stderr,"ERROR: can't set sound channels, %d\n", chan);
	exit(1);
    }
    ioctl(oss_fd, SNDCTL_DSP_SPEED, &hw_rate);
    if (rate != hw_rate) {
	oss_sr = rate;
	oss_hr = hw_rate;
	fprintf(stderr,"WARNING: sample rate mismatch (need %d, got %d)\n",
		rate,hw_rate);
    }
    if (ioctl(oss_fd, SNDCTL_DSP_GETODELAY, &oss_out_delay) < 0) {
      perror("SNDCTL_DSP_GETODELAY");
      exit(1);
    }
    if (ioctl(oss_fd, SNDCTL_DSP_GETOSPACE, &abinfo) < 0) {
      perror("SNDCTL_DSP_GETOSPACE");
      exit(1);
    }
    oss_fragments = abinfo.fragments;
    oss_fragsize = abinfo.fragsize;
    oss_fragstotal = abinfo.fragstotal;
    printf("oss_out_delay %d\n", oss_out_delay);
    return 0;
}

static int oss_init(char *dev, int chan, int r)
{
    int trigger;

    oss_fd = open(dev,O_WRONLY | O_NONBLOCK);
    if (-1 == oss_fd) {
	fprintf(stderr,"WARNING: open %s: %s\n",dev,strerror(errno));
	return -1;
    }
    oss_setformat(chan,r);
    trigger = PCM_ENABLE_OUTPUT;
    ioctl(oss_fd,SNDCTL_DSP_SETTRIGGER,&trigger);
    return 0;
}

/* ------------------------------------------------------------------------ */
/* qtime code                                                           */

static avi_t *avi;
static int hasaudio, hasvideo;
static qtime_t *qt = NULL;
static qtime_track_t *vqtrk = NULL;
static qtime_track_t *aqtrk = NULL;
#define FILE_FORMAT_UNKNOWN 0
#define FILE_FORMAT_AVI     1
#define FILE_FORMAT_MOV     2
static int file_format;

static XImage *ximage;
static XvImage *xvimage;
static GC gc;
static int width = 320, height = 32;
static int drop = 0, droptotal = 0;
static double fps = 0;
static unsigned char *frame_buf;
static unsigned char *read_buf;
static unsigned char *pic_data;
static uint32_t vcodec;
static int csp;
static int max_frame_num;
static int frame_num;

static uint32_t acodec = ACODEC_UNKNOWN;
//static int audio_frame_samples = 0;
static int bytespspl = 0;
//static int frame_size = 0;
static short *pcm_buf[2];
static unsigned char *audio_read_buf;
static unsigned char *audio_buf;
static int audio_buf_pos = 0;
static int audio_read_bytes = 0;
static int audio_samples = 0;
static int audio_gap = 0;
//static int avsync_step = 3;
//static int avsync_next_second = 0;
static struct timeval start;

int
check_file_format_avi(const char *name)
{
  int fd;
  int format;
  unsigned char head_buf1[8];
  unsigned char head_buf2[8];

  format = FILE_FORMAT_UNKNOWN;

  if ((fd = open(name, O_RDONLY)) < 0) {
    fprintf(stderr, "check_file_format: cannot open %s.\n", name);
    exit(1);
//    return FILE_FORMAT_UNKNOWN;
  }
  if (read(fd, head_buf1, 8) == 8) {
    if (strncasecmp(head_buf1, "RIFF", 4) == 0) {
      if (read(fd, head_buf2, 8) == 8) {
        if (strncasecmp(head_buf2, "AVI ", 4) == 0) {
          format = FILE_FORMAT_AVI;
        }
      }
    }
  }
  close(fd);

  return format;
}

static void file_init(FILE *fp, char *filename)
{
  const char *comp, *acomp;
  int audio_bytes;

  file_format = check_file_format_avi(filename);

  if (file_format == FILE_FORMAT_AVI) {
    avi = AVI_open_input_file(filename, 1);
    if (NULL == avi) {
      fprintf(stderr,"ERROR: can't open file: %s\n",filename);
      exit(1);
    }
    max_frame_num = AVI_video_frames(avi);
    if (max_frame_num > 0) {
      hasvideo = 1;
      width = AVI_video_width(avi);
      height = AVI_video_height(avi);
      fps = AVI_frame_rate(avi);
      comp = AVI_video_compressor(avi);
      vcodec = vcodec_fourcc_str_to_codec(comp);
    } else {
      hasvideo = 0;
      width = 0;
      height = 0;
      fps = 0;
      comp = NULL;
      vcodec = VCODEC_UNKNOWN;
      max_frame_num = 0;
    }

    audio_bytes = AVI_audio_bytes(avi);
    if (audio_bytes > 0) {
      hasaudio = 1;
      rate = AVI_audio_rate(avi);
      channels = AVI_audio_channels(avi);
      bits = AVI_audio_bits(avi);
      bytespspl = ((bits+7)/8) * channels;
      acodec = acodec_code_to_codec(AVI_audio_format(avi));
      switch (acodec) {
	case ACODEC_PCM:
	case ACODEC_PCM_LE:
	  audio_read_bytes = 576 * bytespspl;
	  break;
	case ACODEC_MP3:
	  audio_read_bytes = 288;
	  break;
      }
//      audio_frame_samples = 1152 * 4 * bytespspl;
    } else {
      hasaudio = 0;
      rate = 0;
      channels = 0;
      bits = 0;
      acodec = ACODEC_UNKNOWN;
      audio_read_bytes = 0;
//      audio_frame_samples = 0;
    }
  } else {
    qt = qtime_open_read(filename);
    if (NULL == qt) {
	fprintf(stderr,"ERROR: can't open file: %s\n",filename);
	exit(1);
    }
    file_format = FILE_FORMAT_MOV;

    vqtrk = qtime_get_next_video_track(qt, NULL);
    aqtrk = qtime_get_next_sound_track(qt, NULL);

    if (vqtrk) {
      hasvideo = 1;
      width  = qtime_track_get_video_width(vqtrk);
      height = qtime_track_get_video_height(vqtrk);
      fps = qtime_track_get_video_fps(vqtrk);
      comp = qtime_track_get_compressor(vqtrk);
      max_frame_num = qtime_track_get_video_frames(vqtrk);
      vcodec = vcodec_fourcc_str_to_codec(comp);
    } else {
      hasvideo = 0;
      width = 0;
      height = 0;
      fps = 0;
      comp = NULL;
      vcodec = VCODEC_UNKNOWN;
      max_frame_num = 0;
    }

    if (aqtrk) {
      hasaudio = 1;
      rate = qtime_track_get_audio_rate(aqtrk);
      channels = qtime_track_get_audio_channels(aqtrk);
      bits = qtime_track_get_audio_bits(aqtrk);
      acomp = qtime_track_get_compressor(aqtrk);
      acodec = acodec_fourcc_to_codec(acomp);
      bytespspl = ((bits+7)/8) * channels;
      switch (acodec) {
	case ACODEC_PCM:
	case ACODEC_PCM_LE:
	case ACODEC_PCM_BE:
	  audio_read_bytes = 576 * bytespspl;
	  break;
	case ACODEC_MP3:
	  audio_read_bytes = 288;
	  break;
      }
    } else {
      hasaudio = 0;
      rate = 0;
      channels = 0;
      bits = 0;
      acomp = NULL;
      acodec = ACODEC_UNKNOWN;
      bytespspl = 0;
//      audio_frame_samples = 0;
    }
  }

  if (hasvideo) {
    if (vcodec == VCODEC_UNKNOWN) {
      fprintf(fp, "fourcc '%.4s' is unsupported.\n", comp);
      exit(1);
    }
    if (vcodec_decode_set_codec(vcodec) == VCODEC_FAIL) {
      fprintf(fp, "fourcc '%.4s' is unsupported.\n", comp);
      exit(1);
    }
    csp = vcodec_decode_csp_cap(CSP_UNKNOWN);

    fprintf(fp,"INFO: playing %s\n",filename);
    /* print video info */
    fprintf(fp, "  video:\n");
    fprintf(fp, "      width : %d\n"
		"      height: %d\n"
		"      rate  : %.2f fps\n"
		"      codec : %s\n",
		width, height, fps, comp);
  }

  if (hasaudio) {
    if (acodec_decode_set_codec(acodec) != ACODEC_OK) {
      fprintf(stderr,"WARNING: unsupported audio codec\n");
      hasaudio = 0;
      return;
    }
    if (acodec_get_cap(acodec, ACODEC_CAP_ANALYSIS) != ACODEC_CAP_NONE) {
      int ret, size = 0;
      int r, c, b;
      uint8_t rbuf[640*1024];
      int num_samples;

      ret = ACODEC_NEED_MORE_DATA;
      while (ret == ACODEC_NEED_MORE_DATA) {
	if (file_format == FILE_FORMAT_AVI)
	  size = AVI_read_audio(avi, rbuf, audio_read_bytes);
	else if (file_format == FILE_FORMAT_MOV) {
	  if (acodec == ACODEC_MP3)
	    num_samples = 1;
	  else if (acodec == ACODEC_PCM || acodec == ACODEC_PCM_LE ||
	           acodec == ACODEC_PCM_BE)
	    num_samples = audio_read_bytes / bytespspl;
          size = qtime_track_read_audio(aqtrk, rbuf, &num_samples);
	}
        ret = acodec_decode_analyze_stream(rbuf, size, &r, &c, &b);
        if (ret == ACODEC_FAIL) {
          fprintf(stderr, "acodec_decode_analyze_stream() error.\n");
	  hasaudio = 0;
          return;
        }
      }
      if (bits != 16) {
        fprintf(stderr, "unsupported audio bits.\n");
	hasaudio = 0;
        return;
      }

      rate = r;
      channels = c;
      bits = b;
      bytespspl = ((bits+7)/8) * channels;

      if (file_format == FILE_FORMAT_AVI)
	AVI_set_audio_position(avi, 0);
      else if (file_format == FILE_FORMAT_MOV)
        qtime_track_set_audio_position(aqtrk, 0);
    }

    if (channels > 2) {
      fprintf(stderr, "unsupported channels %d.\n", channels);
      hasaudio = 0;
      return;
    }

    pcm_buf[0] = (short*)malloc(rate * sizeof(short) * 2);
    pcm_buf[1] = (short*)malloc(rate * sizeof(short) * 2);
    audio_buf = (unsigned char*)malloc(rate * bytespspl * 2);
    audio_read_buf = (unsigned char*)malloc(rate * bytespspl * 2);
    if (!pcm_buf[0] || !pcm_buf[1] || !audio_buf || !audio_read_buf * 2) {
      fprintf(stderr, "memory allocate error.\n");
      exit(1);
    }

    if (acodec_decode_init(acodec, rate, channels, bits) != ACODEC_OK) {
      fprintf(stderr, "acodec_decode_init error.\n");
      hasaudio = 0;
      return;
    }
    if (-1 == oss_init("/dev/dsp", channels, rate)) {
      hasaudio = 0;
      return;
    }

    /* print audio info */
    fprintf(fp,"  audio:\n");
    fprintf(fp, "      rate  : %d Hz\n"
		"      bits  : %d\n"
		"      chans : %d\n"
		"      codec : %s\n",
		rate, bits, channels, acodec_codec_to_fourcc(acodec));
  }

  if (!hasvideo && !hasaudio) {
    fprintf(stderr,"ERROR: no playable stream found\n");
    exit(1);
  }
}

static int frame_blit(void)
{
    int i;
    int keyframe, size=0, flag;
    
    if (!hasvideo)
      return 0;

    if (0 == frame_num) {
	/* init */
        if (vcodec_decode_csp_cap(csp) != csp) {
	  fprintf(stderr, "cannot decode %s\n", csp_to_str(csp));
	  exit(1);
	}
	frame_buf = malloc(width * height * 4);
	read_buf = malloc(width * height * 4);
	gc = XCreateGC(dpy,XtWindow(simple),0,NULL);
	switch (csp) {
	case CSP_RGB24:
	    fprintf(stderr,"INFO: using BC_RGB888 + %s\n",
		    use_gl ? "OpenGL" : "plain X11");
	    ximage = x11_create_ximage(dpy,width,height);
	    pic_data = frame_buf;
	    break;
	case CSP_YUV422:
	    fprintf(stderr,"INFO: using BC_YUV422 + Xvideo extention\n");
	    xvimage = xv_create_ximage(dpy,width,height,
					  xv_port,FOURCC_YUV2);
	    pic_data = frame_buf;
	    break;
	case CSP_YV12:
            if(xv_have_YV12)
              {
              fprintf(stderr,
                      "INFO: using BC_YUV420P + Xvideo extention (YV12)\n");
              xvimage = xv_create_ximage(dpy,width,height,
              xv_port,FOURCC_YV12);
	      pic_data = xvimage->data;
              }
	    break;
	case CSP_YUV420P:
	    if(xv_have_I420)
              {
              fprintf(stderr,
                      "INFO: using BC_YUV420P + Xvideo extention (I420)\n");
              xvimage = xv_create_ximage(dpy,width,height,
	                                    xv_port,FOURCC_I420);
	      pic_data = xvimage->data;
	      }
              break;
	default:
	    fprintf(stderr,"ERROR: internal error at %s:%d\n",
		    __FILE__,__LINE__);
	    exit(1);
	}
	if (vcodec_decode_init(vcodec, width, height, csp) !=
	    VCODEC_OK) {
	  fprintf(stderr,"ERROR: video codec '%s' initialize failed.\n", vcodec_get_codec_name(vcodec));
	  exit(1);
	}
    }

    if (drop) {
	droptotal += drop;
	fprintf(stderr,"dropped %d frame(s)\r",droptotal);
	for (i = 0; i < drop; i++) {
	  flag = VCODEC_NULL_DECODE;
	  if (file_format == FILE_FORMAT_AVI)
            size = AVI_read_frame(avi, read_buf, &keyframe);
	  else if (file_format == FILE_FORMAT_MOV)
            size = qtime_track_read_video(vqtrk, read_buf, &keyframe);
	  if (size < 0)
	    return -1;
	  else if (size > 0)
            vcodec_decode(pic_data, read_buf, size, &flag);
	  frame_num++;
	}
	drop = 0;
    }
    if (file_format == FILE_FORMAT_AVI)
      size = AVI_read_frame(avi, read_buf, &keyframe);
    else if (file_format == FILE_FORMAT_MOV)
      size = qtime_track_read_video(vqtrk, read_buf, &keyframe);
    else
      size = 0;
    flag = 0;
    if (size < 0)
      return -1;
    else if (size > 0)
      vcodec_decode(pic_data, read_buf, size, &flag);
    frame_num++;

    switch (csp) {
    case CSP_RGB24:
	if (use_gl) {
	    gl_blit(simple,frame_buf,width,height,swidth,sheight);
	} else {
	    switch (pixmap_bytes) {
	    case 2:
		rgb_to_lut2(ximage->data,frame_buf,width*height);
		break;
	    case 4:
		rgb_to_lut4(ximage->data,frame_buf,width*height);
		break;
	    }
	    x11_blit(XtWindow(simple),gc,ximage,width,height);
	}
	break;
    case CSP_YUV422:
    case CSP_YUV420P:
    case CSP_YV12:
	xv_blit(XtWindow(simple),gc,xvimage,
		width,height,swidth,sheight);
	break;
    default:
	fprintf(stderr,"ERROR: internal error at %s:%d\n",
		__FILE__,__LINE__);
	exit(1);
    }

    if (frame_num > max_frame_num)
	return -1;
    return 0;
}

#if 0
static int audio_write(void)
{
    int rc, i, c;
    int num_spl, spls;
    int size, count;
    int max_count;
    int put_bytes = 0;

    if (!hasaudio)
      return 0;

#if 0
    max_count = (64*1024) / (audio_frame_samples*bytespspl);
    if (max_count <= 0)
      max_count = 2;
#else
    max_count = 20;
#endif
    for (count = 0; count < max_count; count++) {
      count++;

      if (audio_buf_pos <= 0) {
        if (file_format == FILE_FORMAT_AVI)
          size = AVI_read_audio(avi, audio_read_buf, audio_read_bytes);
        else
          size = qtime_track_read_audio_frame(aqtrk, audio_read_buf, &num_spl);
        if (size < 0)
          return -1;

        spls = acodec_decode((void**)pcm_buf, audio_read_buf, size);
        if (spls == 0)
	  continue;
        for (c = 0; c < channels; c++) {
          short *sp = (short*)audio_buf;
          sp += c;
          for (i = 0; i < spls; i++) {
            *sp = pcm_buf[c][i];
	    sp += channels;
          }
        }
	audio_buf_pos = spls * bytespspl;
      }

      put_bytes = audio_buf_pos;
      rc = write(oss_fd, audio_buf, audio_buf_pos);
      if (rc >= 0) {
       	if (rc != audio_buf_pos) {
	  fprintf(stderr, "write_audio: put_bytes %d, rc %d.\n", put_bytes, rc);
	  if (rc > 0) {
	    memmove(audio_buf, audio_buf + rc, audio_buf_pos - rc);
	    audio_buf_pos -= rc;
	    if (audio_buf_pos < 0) {
	      fprintf(stderr, "write_audio: audio_buf_pos negative %d.\n", audio_buf_pos);
	      audio_buf_pos = 0;
	    }
	  }
	  break;
	} else {
	  audio_buf_pos = 0;
	}
      } else if (errno == EAGAIN) {
	break;
      }

      switch (rc) {
      case -1:
	  perror("write dsp");
	  close(oss_fd);
	  oss_fd = -1;
	  aqtrk = NULL;
	  break;
      case 0:
	  fprintf(stderr,"write dsp: Huh? no data written?\n");
	  close(oss_fd);
	  oss_fd = -1;
	  aqtrk = NULL;
	  break;
      default:
	  break;
      }
    }

    printf("count %d\n", count);
    return 0;
}
#endif
static int audio_write(void)
{
  int rc, i, c;
  int num_spl, spls;
  int size;
  int put_bytes = 0;
  struct timeval now;

  if (!hasaudio)
    return 0;

  while (audio_buf_pos <= 0) {
    if (file_format == FILE_FORMAT_AVI)
      size = AVI_read_audio(avi, audio_read_buf, audio_read_bytes);
    else {
      num_spl = audio_read_bytes / bytespspl;
      size = qtime_track_read_audio(aqtrk, audio_read_buf, &num_spl);
    }
    if (size <= 0)
      return -1;

    spls = acodec_decode((void**)pcm_buf, audio_read_buf, size);
    if (spls == 0)
      continue;
    for (c = 0; c < channels; c++) {
      short *sp = (short*)audio_buf;
      sp += c;
      for (i = 0; i < spls; i++) {
        *sp = pcm_buf[c][i];
        sp += channels;
      }
    }
    audio_buf_pos = spls * bytespspl;
  }

  put_bytes = audio_buf_pos;
  rc = write(oss_fd, audio_buf, audio_buf_pos);
  if (rc >= 0) {
    if (rc != audio_buf_pos) {
      fprintf(stderr, "write_audio: put_bytes %d, rc %d.\n", put_bytes, rc);
      if (rc > 0) {
        memmove(audio_buf, audio_buf + rc, audio_buf_pos - rc);
        audio_buf_pos -= rc;
        if (audio_buf_pos < 0) {
          fprintf(stderr, "write_audio: audio_buf_pos negative %d.\n", audio_buf_pos);
          audio_buf_pos = 0;
        }
	return 0;
      }
    } else {
      audio_buf_pos = 0;
    }
    audio_samples += rc / bytespspl;
  } else if (errno == EAGAIN) {
    return 0;
  }

  switch (rc) {
    case -1:
      perror("write dsp");
      close(oss_fd);
      oss_fd = -1;
      aqtrk = NULL;
      break;
    case 0:
      fprintf(stderr,"write dsp: Huh? no data written?\n");
      close(oss_fd);
      oss_fd = -1;
      aqtrk = NULL;
      break;
    default:
      break;
  }

  {
  long msec, amsec;
  int out_delay;

  out_delay = (int)((double)(oss_fragsize*oss_fragstotal/bytespspl) / (double)rate * 1000);

  gettimeofday(&now,NULL);
  msec  = (start.tv_sec  - now.tv_sec)  * 1000;
  msec += (start.tv_usec - now.tv_usec) / 1000;
//  if (now.tv_sec > avsync_next_second) {
    amsec = (long)((double)(audio_samples) / (double)rate * 1000.0);
    amsec -= out_delay;
    amsec += msec;
    amsec += audio_gap;
    if (amsec < -150)
      audio_gap +=  -amsec / 4;
    else if (amsec > 150)
      audio_gap -= amsec / 4;
   // avsync_next_second = now.tv_sec + avsync_step;
//    avsync_next_second = now.tv_sec + 1;
//    printf("audio_gap %d.\n", audio_gap);
//    printf("amsec = %d, droptotal %d.\n", amsec, droptotal);
//    printf("out_delay %d\n", out_delay);
//  }
  }

  return 0;
}

static void frame_delay(struct timeval *start, struct timeval *wait)
{
    struct timeval now;
    long msec;

    gettimeofday(&now,NULL);
    msec  = (start->tv_sec  - now.tv_sec)  * 1000;
    msec += (start->tv_usec - now.tv_usec) / 1000;
    msec += frame_num * 1000 / fps;
    msec += audio_gap;
    if (msec < 0) {
	drop = -msec * fps / 1000;
	wait->tv_sec  = 0;
	wait->tv_usec = 0;
    } else {
	wait->tv_sec  = msec / 1000;
	wait->tv_usec = (msec % 1000) * 1000;
    }
}

/* ------------------------------------------------------------------------ */
/* main                                                                     */

struct ARGS {
    int  xv;
    int  gl;
} args;

XtResource args_desc[] = {
    /* name, class, type, size, offset, default_type, default_addr */
    {
	/* Integer */
	"xv",
	XtCValue, XtRInt, sizeof(int),
	XtOffset(struct ARGS*,xv),
	XtRString, "1"
    },{
	"gl",
	XtCValue, XtRInt, sizeof(int),
	XtOffset(struct ARGS*,gl),
	XtRString, "1"
    }
};
const int args_count = XtNumber(args_desc);

XrmOptionDescRec opt_desc[] = {
    { "-noxv",  "xv", XrmoptionNoArg,  "0" },
    { "-nogl",  "gl", XrmoptionNoArg,  "0" },
};
const int opt_count = (sizeof(opt_desc)/sizeof(XrmOptionDescRec));

static void usage(FILE *fp, char *prog)
{
    char *p;

    p = strrchr(prog,'/');
    if (p) prog = p+1;
    fprintf(fp,
	    "\n"
	    "Very simple quicktime movie player for X11.  Just playes\n"
	    "the movie and nothing else.  No fancy gui, no controls.\n"
	    "\n"
	    "You can quit with 'Q' and 'ESC' keys.\n"
	    "\n"
	    "usage: %s [ options ] <file>\n"
	    "options:\n"
	    "  -noxv   don't use the Xvideo extention\n"
	    "  -nogl   don't use OpenGL\n"
	    "\n",
	    prog);
}

static void quit_ac(Widget widget, XEvent *event,
		    String *params, Cardinal *num_params)
{
    exit(0);
}

static void resize_ev(Widget widget, XtPointer client_data,
		      XEvent *event, Boolean *d)
{
    switch(event->type) {
    case MapNotify:
    case ConfigureNotify:
	XtVaGetValues(widget,XtNheight,&sheight,XtNwidth,&swidth,NULL);
	fprintf(stderr,"INFO: window size is %dx%d\n",swidth,sheight);
	if (use_gl)
	    gl_resize(widget,swidth,sheight);
	break;
    }
}

static XtActionsRec action_table[] = {
    { "Quit",  quit_ac },
};

static String res[] = {
    "ndeplay.playback.translations:  #override \\n"
    "        <Key>Q:                 Quit()    \\n"
    "        <Key>Escape:            Quit()",
    "ndeplay.playback.background:    black",
    NULL
};

int main(int argc, char *argv[])
{
    struct timeval wait;
    int video_blit = 0;
    
    app_shell = XtVaAppInitialize(&app_context, "play",
				  opt_desc, opt_count,
				  &argc, argv,
				  res, NULL);
    XtGetApplicationResources(app_shell,&args,
			      args_desc,args_count,
			      NULL,0);

    /* open file */
    if (argc < 2) {
	usage(stderr,argv[0]);
	exit(1);
    }
    file_init(stdout,argv[1]);
    frame_num = 0;

    /* init x11 stuff */
    dpy = XtDisplay(app_shell);
    XtAppAddActions(app_context,action_table,
		    sizeof(action_table)/sizeof(XtActionsRec));
    XtVaSetValues(app_shell, XtNtitle,argv[1],NULL);
    simple = XtVaCreateManagedWidget("playback",simpleWidgetClass,app_shell,
				     XtNwidth,  width,
				     XtNheight, height,
				     NULL);
    XtAddEventHandler(simple,StructureNotifyMask, True, resize_ev, NULL);
    x11_init();
    if (args.xv && hasvideo)
	xv_init();

    /* use Xvideo? */
    csp = CSP_UNKNOWN;
    if (xv_have_YUY2 && vcodec_decode_csp_cap(CSP_YUV422) == CSP_YUV422)
      csp = CSP_YUV422;
    if (xv_have_I420 && vcodec_decode_csp_cap(CSP_YUV420P) == CSP_YUV420P)
      csp = CSP_YUV420P;
    if (xv_have_YV12 && vcodec_decode_csp_cap(CSP_YV12) == CSP_YV12)
      csp = CSP_YV12;

    if (csp == CSP_UNKNOWN)
      csp = CSP_RGB24;

    /* use OpenGL? */
    XtRealizeWidget(app_shell);
    if (CSP_RGB24 == csp && args.gl && hasvideo)
	gl_init(simple,width,height);

    /* enter main loop */
    gettimeofday(&start,NULL);
    for (;;) {
	int rc,max;
	fd_set rd,wr;
	XEvent event;

	if (True == XCheckMaskEvent(dpy, ~0, &event)) {
	    XtDispatchEvent(&event);
	} else {
	    XFlush(dpy);
	    FD_ZERO(&rd);
	    FD_ZERO(&wr);
	    FD_SET(ConnectionNumber(dpy),&rd);
	    max = ConnectionNumber(dpy);
	    if (hasaudio) {
		FD_SET(oss_fd,&wr);
		if (oss_fd > max)
		    max = oss_fd;
	    }
	    if (hasvideo) {
		frame_delay(&start,&wait);
		if (wait.tv_usec < 1000)
		  video_blit = 1;
	    } else {
		wait.tv_sec  = 1;
		wait.tv_usec = 0;
	    }
	    rc = select(max+1,&rd,&wr,NULL,&wait);
	    if (hasaudio && FD_ISSET(oss_fd,&wr))
	      if (0 != audio_write()) {
		hasaudio = 0;
//		printf("audio break\n");
//	        break;
	      }
//	    if (hasvideo && 0 == rc)
	    if (hasvideo)
	      if (0 == rc || video_blit) {
                if (0 != frame_blit()) {
		  hasvideo = 0;
//		  printf("video break\n");
//	          break;
		}
		video_blit = 0;
	      }
	}
	if (!hasvideo && !hasaudio)
	  break;
    }
    return 0;
}

