/* ifile.c */

#include <stdio.h>
#include <stdlib.h>

#include "video.h"
#include "audio.h"
#include "parseopt.h"
#include "ifile.h"
#include "ofile.h"
#include "util.h"
#include "deinterlace.h"
#include "util.h"
#include "vutil.h"
#include "vcodec.h"
#include "acodec.h"
#include "zoom.h"

#ifdef BUILD_LIBQTIME
#include "qtime.h"
#endif /* BUILD_LIBQTIME */

static IN_FILE **in_file = NULL;
static int in_file_num = 0;
static int crop_left = 0;
static int crop_right = 0;
static int crop_top = 0;
static int crop_bottom = 0;
static CUT_BLOCK *cut_blocks = NULL;
static int cut_blocks_num = 0;
static int interlaced_src_flag = 0;
static int deinterlace_flag = 0;
static int deinterlace_type = 0;
static int field_order = TOP_FEILD_FIRST;
//static int frame_shift = 0;
static int flicker_level = 0;
static int flicker_func_num = 2;
static int flicker_eval_per = 10;
//static int deinterlace_y_only_flag = 0;
//static int deinterlace_uv_only_flag = 0;

static int median_filt_flag = 0;

static int total_frames = 0;
static int total_audio_bytes = 0;

static int denoise_flag = 0;
static int denoise_radius = 0;
static int denoise_threshold = 0;
static int denoise_ppthreshold = 0;
static int denoise_delay = 0;
static int denoise_sharpen = 0;
static int denoise_uv_flag = 0;
static int denoise_uvthreshold = 0;
static const char *denoise_test = NULL;

static int bright = 0;
static int hue = 0;
static int contrast = 0;
static int color = 0;

static int max_in_file_num = MAX_IN_FILE_NUM;

#define MAX_INT (2147483647)

static int
set_cut_frames (int start, int end)
{
  CUT_BLOCK *blk;
  int tmp;

  if (start < 0)
    start = 0;
  if (end < 0)
    end = 0;

  if (end < start) {
    tmp = start;
    start = end;
    end = tmp;
  }

  blk = (CUT_BLOCK*)realloc(cut_blocks, sizeof(CUT_BLOCK) * (cut_blocks_num+2));
  if (blk == NULL) {
    fprintf (stderr, "set_cut_frames(): realloc error.\n");
    exit (1);
  }
  cut_blocks = blk;

  cut_blocks[cut_blocks_num].start = start;
  cut_blocks[cut_blocks_num].end = end;

  cut_blocks_num++;

  return 0;
}

static int
opt_frame_cut (const char *arg)
{
  char *p;
  char *ep;
  int start;
  int end;
  int tmp;

  p = (char*)arg;
  ep = p;
  while (*p != '\0') {
    if (*p == ':')
      start = 0;
    else {
      start = strtol(p, &ep, 10);
      if (ep == NULL) {
        fprintf (stderr, "option error. -cut %s\n", arg);
        return -1;
      }
    }
    if (ep != NULL && *ep != '\0')
      p = ++ep;
    else
      p = ep;
    if (*p == ',' || *p == '\0')
      end = MAX_INT;
    else
      end = strtol(p, &ep, 10);
    if (ep != NULL && *ep != '\0')
      p = ++ep;
    else
      p = ep;
    if (start > end) {
      tmp = start;
      start = end;
      end = tmp;
    }
    start--;
    end--;
    set_cut_frames (start, end);
  }
  return 0;
}

static int
opt_frame_crop (const char *arg)
{
  char *p;
  char *ep;
  int base = 10;
  int left, right, top, bottom;
  int error1 = 0;

  if (!arg) {
    fprintf (stderr, "crop required augument\n");
    exit (1);
  }

  left = right = top = bottom = 0;

  p = (char*)arg;
  left = strtol(p, &ep, base);
  if (ep != NULL && *ep == ':' && left >= 0)
    p = ++ep;
  else
    goto fail;

  right = strtol(p, &ep, base);
  if (ep != NULL && *ep == ':' && right >= 0)
    p = ++ep;
  else
    goto fail;

  top = strtol(p, &ep, base);
  if (ep != NULL && *ep == ':' && top >= 0)
    p = ++ep;
  else
    goto fail;

  bottom = strtol(p, &ep, base);
  if (bottom < 0)
    goto fail;

  if ((left % 2) || (right % 2) || (top % 2) || (bottom % 2)) {
    error1 = 1;
    goto fail;
  }

  crop_left = left;
  crop_right = right;
  crop_top = top;
  crop_bottom = bottom;

  return 0;

fail:
  fprintf (stderr, "option -crop error. arg '%s'\n", arg);
  if (error1)
    fprintf (stderr, "left, right, top, bottom must be multiple of 2.\n");
  exit (1);
  return -1;
}

void
free_in_files(void)
{
  int i;

  if (in_file == NULL)
    return;

  for (i = 0; in_file[i] != NULL; i++) {
    if (in_file[i]->file_format == FILE_FORMAT_AVI) {
      if (in_file[i]->avifile) {
        AVI_close(in_file[i]->avifile);
        in_file[i]->avifile = NULL;
      }
    } else if (in_file[i]->file_format == FILE_FORMAT_MOV) {
      if (in_file[i]->qt) {
	qtime_close(in_file[i]->qt);
	in_file[i]->qt = NULL;
	in_file[i]->vqtrk = NULL;
	in_file[i]->aqtrk = NULL;
      }
    } else {
      fprintf(stderr, "free_in_files: unknown file format.\n");
      exit(1);
    }
    if (in_file[i]->name)
      free(in_file[i]->name);
    if (in_file[i]->cut_blocks)
      free(in_file[i]->cut_blocks);
    free(in_file[i]);
  }
  free(in_file);
  in_file = NULL;
  in_file_num = 0;
}

int
set_in_filename (const char *name)
{
  int len;
  IN_FILE **ifp;

  if (in_file_num > max_in_file_num) {
    fprintf(stderr, "set_in_filename: input files over maximum %d\n",
	max_in_file_num);
    exit(1);
  }

  ifp = (IN_FILE**) realloc(in_file, sizeof(IN_FILE*) * (in_file_num+2));
  if (ifp == NULL) {
    mem_alloc_fail("opt_in_filename", NO_EXIT);
    quit_ifile();
    exit(1);
  }
  in_file = ifp;
  in_file[in_file_num] = NULL;
  in_file[in_file_num+1] = NULL;

  in_file = ifp;

  in_file[in_file_num] = (IN_FILE*) malloc(sizeof(IN_FILE));
  if (in_file[in_file_num] == NULL) {
    mem_alloc_fail("opt_in_filename", NO_EXIT);
    quit_ifile();
    exit(1);
  }

  memset(in_file[in_file_num], 0, sizeof(IN_FILE));

  len = strlen(name);
  if (len > MAX_FILENAME_LENGTH-1)
    len = MAX_FILENAME_LENGTH-1;
  in_file[in_file_num]->name = (char*) malloc(len+1);
  if (in_file[in_file_num]->name == NULL) {
    mem_alloc_fail("opt_in_filename", NO_EXIT);
    quit_ifile();
    exit(1);
  }
  memcpy(in_file[in_file_num]->name, name, len);
  in_file[in_file_num]->name[len] = '\0';

  in_file[in_file_num]->crop_left = crop_left;
  in_file[in_file_num]->crop_right = crop_right;
  in_file[in_file_num]->crop_top = crop_top;
  in_file[in_file_num]->crop_bottom = crop_bottom;

  crop_left = crop_right = crop_top = crop_bottom = 0;

  in_file[in_file_num]->interlaced_src_flag = interlaced_src_flag;
  parseopt_set_default("interlaced_source", NULL);

  in_file[in_file_num]->deinterlace_flag = deinterlace_flag;
  parseopt_set_default("deinterlace", NULL);
  in_file[in_file_num]->deinterlace_type = deinterlace_type;
  parseopt_set_default("deinterlace_type", NULL);
//  in_file[in_file_num]->deinterlace_y_only_flag = deinterlace_y_only_flag;
//  deinterlace_y_only_flag = 0;
//  in_file[in_file_num]->deinterlace_uv_only_flag = deinterlace_uv_only_flag;
//  deinterlace_uv_only_flag = 0;


  in_file[in_file_num]->cut_blocks = cut_blocks;
  in_file[in_file_num]->cut_blocks_num = cut_blocks_num;
  cut_blocks = NULL;
  cut_blocks_num = 0;

  in_file[in_file_num]->field_order = field_order;
  parseopt_set_default("field_order", NULL);

//  in_file[in_file_num]->frame_shift = frame_shift;
//  parseopt_set_default("frame_shift", NULL);


  in_file[in_file_num]->flicker_level = flicker_level;
//  parseopt_set_default("flicker_level", NULL);
  in_file[in_file_num]->flicker_func = flicker_func_num;
//  parseopt_set_default("flicker_func", NULL);
  in_file[in_file_num]->flicker_eval_per = flicker_eval_per;


  in_file[in_file_num]->median_filt_flag = median_filt_flag;
  parseopt_set_default("median_filter", NULL);

  in_file[in_file_num]->denoise_flag = denoise_flag;
  parseopt_set_default("denoise", NULL);

  in_file[in_file_num]->denoise_radius = denoise_radius;
  parseopt_set_default("denoise_radius", NULL);

  in_file[in_file_num]->denoise_threshold = denoise_threshold;
  parseopt_set_default("denoise_threshold", NULL);

  in_file[in_file_num]->denoise_ppthreshold = denoise_ppthreshold;
  parseopt_set_default("denoise_ppthreshold", NULL);

  in_file[in_file_num]->denoise_delay = denoise_delay;
  parseopt_set_default("denoise_delay", NULL);

  in_file[in_file_num]->denoise_sharpen = denoise_sharpen;
  parseopt_set_default("denoise_sharpen", NULL);

  in_file[in_file_num]->denoise_uv_flag = denoise_uv_flag;
  parseopt_set_default("denoise_uv", NULL);

  in_file[in_file_num]->denoise_uvthreshold = denoise_uvthreshold;
  parseopt_set_default("denoise_uvthreshold", NULL);

  in_file[in_file_num]->denoise_test = denoise_test;
  parseopt_set_default("denoise_test", NULL);

  in_file[in_file_num]->bright = bright;
  parseopt_set_default("bright", NULL);

  in_file[in_file_num]->hue = hue;
  parseopt_set_default("hue", NULL);

  in_file[in_file_num]->contrast = contrast;
  parseopt_set_default("contrast", NULL);

  in_file[in_file_num]->color= color;
  parseopt_set_default("color", NULL);

  in_file[in_file_num]->refer_count = 0;

  in_file_num++;

  return 0;
}

int
opt_flicker_func(const char *arg)
{
  flicker_func_num = atoi(arg);
  set_flicker_func(flicker_func_num);
  return 0;
}

#if 0
int
opt_filter_type(const char *arg)
{
  zoom_set_xfilt(arg);
  zoom_set_yfilt(arg);
  return 0;
}

int
opt_window_type(const char *arg)
{
  zoom_set_xwindow(arg);
  zoom_set_ywindow(arg);
  return 0;
}
#endif

static SEL_COMPO deinterlaceerlace_type[] = {
  { "linear_blend", DEINTERLACE_LINEAR_BLEND },
  { "bilinear", DEINTERLACE_BILINEAR },
  { "linedoubling", DEINTERLACE_LINEDOUBLING },
  { "yuvdenoise", DEINTERLACE_DENOISE },
};

static SEL_COMPO field_order_sel[] = {
  { "top", TOP_FEILD_FIRST },
  { "bottom", BOTTOM_FEILD_FIRST },
};

OptionDef in_file_param[] = {
    { "crop", NULL, HAS_ARG|OPT_FUNC, {(void*)opt_frame_crop}, {(int)"0:0:0:0"}, {0}, 0, "set input frame crop (left:right:top:bottom)", "l:r:t:b"},
    { "cut", NULL, HAS_ARG|OPT_FUNC, {(void*)opt_frame_cut}, {0}, {0}, 0, "set cut frame (start_frame:end_frame[,start_frame:end_frame,...])", "xxxx:xxxx"},
    { "deinterlace", "DEINTERLACE", OPT_BOOL, {(void*)&deinterlace_flag}, {0}, {0}, 1, "deinterlaced image", NULL},
    { "deinterlace-type", "DEINTERLACE_TYPE", HAS_ARG|OPT_SEL, {(void*)&deinterlace_type}, {DEINTERLACE_LINEAR_BLEND}, {(int)deinterlaceerlace_type}, sizeof(deinterlaceerlace_type)/sizeof(SEL_COMPO), "deinterlace type", "type"},
    { "interlaced-source", "INTERLACED_SOURCE", OPT_BOOL, {(void*)&interlaced_src_flag}, {0}, {0}, 1, "field of a front and the back is evaluated.", NULL},
    { "flicker-eval", "FLICKER_EVAL", HAS_ARG, {(void*)&flicker_eval_per}, {10}, {1}, 100, "flicker evaluation blocks in percent.", "d"},
    { "field-order", "FEILD_ORDER", HAS_ARG|OPT_SEL, {(void*)&field_order}, {TOP_FEILD_FIRST}, {(int)field_order_sel}, sizeof(field_order_sel)/sizeof(SEL_COMPO), "set input file field order", "order"},
    { "median-filter", "MEDIAN_FILTER", OPT_BOOL, {(void*)&median_filt_flag}, {0}, {0}, 1, "median filter", NULL},
    { "bright", "BRIGHT", HAS_ARG, {(void*)&bright}, {0}, {-255}, 255, "increase or decrease brightness", "<decimal>"},
//    { "hue", "HUE", HAS_ARG, {(void*)&hue}, {0}, {-360}, 360, "hue translate", "<decimal>"},
    { "contrast", "CONTRAST", HAS_ARG, {(void*)&contrast}, {0}, {-1000}, 1000, "contrast translate", "<decimal>"},
    { "color", "COLOR", HAS_ARG, {(void*)&color}, {0}, {-1000}, 1000, "color translate", "<decimal>"},
    { "denoise", "DENOISE", OPT_BOOL, {(void*)&denoise_flag}, {0}, {0}, 1, "denoise video frames", NULL},
    { "denoise-radius", "DENOISE_RADIUS", HAS_ARG, {(void*)&denoise_radius}, {8}, {1}, 16, "denoise search radius", "radius"},
    { "denoise-threshold", "DENOISE_THRESHOLD", HAS_ARG, {(void*)&denoise_threshold}, {5}, {1}, 127, "denoise threshold", "threshold"},
    { "denoise-ppthreshold", "DENOISE_PPTHRESHOLD", HAS_ARG, {(void*)&denoise_ppthreshold}, {4}, {1}, 127, "denoise post process threshold", "threshold"},
    { "denoise-delay", "DENOISE_DELAY", HAS_ARG, {(void*)&denoise_delay}, {3}, {1}, 126, "denoise average delay", "delay"},
    { "denoise-sharpen", "DENOISE_SHARPEN", HAS_ARG, {(void*)&denoise_sharpen}, {125}, {1}, 255, "denoise sharpen", "sharpen"},
    { "denoise-uv", "DENOISE_UV", OPT_BOOL, {(void*)&denoise_uv_flag}, {1}, {0}, 1, "denoise uv", NULL},
    { "denoise-uvthreshold", "DENOISE_UVTHRESHOLD", HAS_ARG, {(void*)&denoise_uvthreshold}, {10}, {1}, 127, "denoise uv threshold", "threshold"},
    { "denoise-test", NULL, HAS_ARG|OPT_STR, {(void*)&denoise_test}, {(int)NULL}, {0}, 0, "denoise test", "string"},
    { "i", NULL, HAS_ARG|OPT_FUNC, {(void*)&set_in_filename}, {0}, {0}, 0, "input file name", "name"},
};

int in_file_param_num = sizeof(in_file_param) / sizeof(OptionDef);

static int
calc_frames (int actual_frames, CUT_BLOCK *cut_blocks, int cut_blocks_num)
{
  CUT_BLOCK tmp_block;
  CUT_BLOCK *tmp_block_p;
  int i,j;
  int tmp_start;
  int total_cut_frames;

  if (cut_blocks_num <= 0)
    return actual_frames;

  cut_blocks[cut_blocks_num].start = MAX_INT;
  cut_blocks[cut_blocks_num].end = MAX_INT;

  for (i = 1; i < cut_blocks_num; i++) {
    tmp_block = cut_blocks[i];
    tmp_start = cut_blocks[i].start;
    for (j = i - 1; j >= 0 && cut_blocks[j].start > tmp_start; j--)
      cut_blocks[j + 1] = cut_blocks[j];
    cut_blocks[j + 1] = tmp_block;
  }

  for (i = 0; i < cut_blocks_num; i++) {
    if (cut_blocks[i].start >= actual_frames) {
      cut_blocks_num = i;
      break;
    }
    if (cut_blocks[i].end >= cut_blocks[i+1].start) {
      cut_blocks[i].end = cut_blocks[i+1].end;
      for (j = i+1; j < cut_blocks_num; j++) {
        cut_blocks[j] = cut_blocks[j+1];
      }
      cut_blocks_num--;
    }
    if (cut_blocks[i].end > actual_frames)
      cut_blocks[i].end = actual_frames;
  }

  tmp_block_p=(CUT_BLOCK*)realloc(cut_blocks, sizeof(CUT_BLOCK)*cut_blocks_num);
  if (tmp_block_p == NULL) {
    fprintf(stderr, "calc_frames: realloc error.\n");
    exit(1);
  }
  cut_blocks = tmp_block_p;

  total_cut_frames = 0;
  for (i = 0; i < cut_blocks_num; i++) {
    total_cut_frames += cut_blocks[i].end - cut_blocks[i].start + 1;
//    printf ("cut_frames(): start %d, end %d\n", cut_blocks[i].start+1, cut_blocks[i].end+1);
  }
  return actual_frames - total_cut_frames;
}

int
check_file_format_avi(const char *name)
{
  int fd;
  int format;
  unsigned char head_buf1[8];
  unsigned char head_buf2[8];

  format = FILE_FORMAT_UNKNOWN;

  if ((fd = open(name, O_RDONLY)) < 0) {
    fprintf(stderr, "check_file_format: cannot open %s.\n", name);
    exit(1);
//    return FILE_FORMAT_UNKNOWN;
  }
  if (read(fd, head_buf1, 8) == 8) {
    if (strncasecmp(head_buf1, "RIFF", 4) == 0) {
      if (read(fd, head_buf2, 8) == 8) {
        if (strncasecmp(head_buf2, "AVI ", 4) == 0) {
          format = FILE_FORMAT_AVI;
	}
      }
    }
  }
  close(fd);

  return format;
}

void
close_in_file(IN_FILE *in_file)
{
  in_file->refer_count--;

  if (in_file->refer_count > 0) {
    return;
  } else if (in_file->refer_count < 0) {
    in_file->refer_count = 0;
    return;
  }

  if (in_file->file_format == FILE_FORMAT_AVI) {
    AVI_close(in_file->avifile);
    in_file->avifile = NULL;
  } else if (in_file->file_format == FILE_FORMAT_MOV) {
    qtime_close(in_file->qt);
    in_file->qt = NULL;
    in_file->vqtrk = NULL;
    in_file->aqtrk = NULL;
  } else {
    fprintf(stderr, "close_in_file: unknown file format.\n");
  }
}

int
open_in_file(IN_FILE *in_file)
{
  if (in_file == NULL) {
    fprintf(stderr, "open_in_files: no input file.\n");
    exit(1);
  }

  in_file->refer_count++;

  if (in_file->refer_count > 1)
    return in_file->file_format;

  in_file->avifile = NULL;
  in_file->qt = NULL;
  in_file->vqtrk = NULL;
  in_file->aqtrk = NULL;

  in_file->file_format = check_file_format_avi(in_file->name);
  if (in_file->file_format == FILE_FORMAT_AVI) {
    in_file->avifile = AVI_open_input_file(in_file->name, 1);
    if (in_file->avifile == NULL) {
      AVI_print_error(in_file->name);
      return FILE_FORMAT_UNKNOWN;
    }

    in_file->actual_frames = AVI_video_frames(in_file->avifile);
    in_file->actual_width = AVI_video_width(in_file->avifile);
    in_file->actual_height = AVI_video_height(in_file->avifile);
    in_file->fps = AVI_frame_rate(in_file->avifile);
    in_file->compressor = AVI_video_compressor(in_file->avifile);
  
    in_file->vcodec = vcodec_fourcc_str_to_codec(in_file->compressor);
    if (in_file->vcodec == VCODEC_UNKNOWN) {
      AVI_close(in_file->avifile);
      in_file->avifile = NULL;
      fprintf(stderr, "check_in_files: %s unsupported video codec '%.4s'\n", in_file->name, in_file->compressor);
      return FILE_FORMAT_UNKNOWN;
    }

#ifndef NDEVIEW_SOURCE
    in_file->audio_bytes = AVI_audio_bytes(in_file->avifile);
    if (in_file->audio_bytes <= 0) {
      in_file->acodec = ACODEC_UNKNOWN;
      in_file->audio_samples = 0;
    } else {
      in_file->channels = AVI_audio_channels(in_file->avifile);
      in_file->bits = AVI_audio_bits(in_file->avifile);
      in_file->rate = AVI_audio_rate(in_file->avifile);
      in_file->bitrate = AVI_audio_mp3rate(in_file->avifile);
      in_file->format = AVI_audio_format(in_file->avifile);
      in_file->bytespspl = ((in_file->bits+7)/8) * in_file->channels;

      in_file->acodec = acodec_code_to_codec(in_file->format);
      if (in_file->acodec == ACODEC_UNKNOWN) {
        fprintf(stderr, "check_in_files: %s unsupported audio codec found.\n", in_file->name);
	in_file->format = 0;
      }

      switch (in_file->format) {
	case 0x55:
          in_file->audio_samples = -1;
          in_file->audio_frame_samples = 576;
          in_file->audio_frames = 0;
	  break;
	case 0x1:
          in_file->audio_samples = in_file->audio_bytes / (((in_file->bits+7)/8) * in_file->channels);
          in_file->audio_frame_samples = 576;
          in_file->audio_frames = 0;
          in_file->audio_frames = (in_file->audio_samples + (in_file->audio_frame_samples-1)) / in_file->audio_frame_samples;
	  break;
	default:
	  in_file->acodec = ACODEC_UNKNOWN;
      }
      check_audio(in_file);
    }
#endif /* NDEVIEW_SOURCE */
  } else if ((in_file->qt = qtime_open_read(in_file->name)) != NULL) {
    in_file->file_format = FILE_FORMAT_MOV;
    in_file->vqtrk = qtime_get_next_video_track(in_file->qt, NULL);
    in_file->aqtrk = qtime_get_next_sound_track(in_file->qt, NULL);

    in_file->actual_frames = qtime_track_get_video_frames(in_file->vqtrk);
    in_file->actual_width = qtime_track_get_video_width(in_file->vqtrk);
    in_file->actual_height = qtime_track_get_video_height(in_file->vqtrk);
    in_file->fps = qtime_track_get_video_fps(in_file->vqtrk);
    in_file->compressor = qtime_track_get_compressor(in_file->vqtrk);
  
    in_file->vcodec = vcodec_fourcc_str_to_codec(in_file->compressor);
    if (in_file->vcodec == VCODEC_UNKNOWN) {
      fprintf(stderr, "check_in_files: %s unsupported video codec '%.4s'\n", in_file->name, in_file->compressor);
      if (in_file->qt) {
        qtime_close(in_file->qt);
        in_file->qt = NULL;
        in_file->vqtrk = NULL;
        in_file->aqtrk = NULL;
      }
      return FILE_FORMAT_UNKNOWN;
    }

#ifndef NDEVIEW_SOURCE
    if (!in_file->aqtrk) {
      in_file->audio_bytes = 0;
      in_file->acodec = ACODEC_UNKNOWN;
      in_file->audio_samples = 0;
    } else {
      in_file->channels = qtime_track_get_audio_channels(in_file->aqtrk);
      in_file->channels = qtime_track_get_audio_channels(in_file->aqtrk);
      in_file->bits = qtime_track_get_audio_bits(in_file->aqtrk);
      in_file->rate = qtime_track_get_audio_rate(in_file->aqtrk);
      in_file->compressor = qtime_track_get_compressor(in_file->aqtrk);
      in_file->bytespspl = ((in_file->bits+7)/8) * in_file->channels;
      in_file->bitrate = -1;

      in_file->acodec = acodec_fourcc_to_codec(in_file->compressor);
      if (in_file->acodec == ACODEC_UNKNOWN) {
        fprintf(stderr, "check_in_files: %s unsupported audio codec found '%.4s'.\n", in_file->name, in_file->compressor);
      }

      switch (in_file->acodec) {
	case ACODEC_MP3:
          in_file->audio_samples = qtime_track_get_audio_samples(in_file->aqtrk);
          in_file->format = 0x55;
	  in_file->audio_bytes = 1;
//          in_file->audio_frames = qtime_track_get_audio_frames(in_file->aqtrk);
//          in_file->audio_frame_samples = qtime_track_get_audio_frame_samples(in_file->aqtrk);
	  break;
	case ACODEC_PCM:
	case ACODEC_PCM_LE:
	  in_file->audio_be = 0;
	case ACODEC_PCM_BE:
          in_file->audio_samples = qtime_track_get_audio_samples(in_file->aqtrk);
          in_file->format = 0x1;
	  in_file->audio_bytes = in_file->audio_samples * (((in_file->bits+7)/8)*in_file->channels);
//          in_file->audio_frames = qtime_track_get_audio_frames(in_file->aqtrk);
//          in_file->audio_frame_samples = qtime_track_get_audio_frame_samples(in_file->aqtrk);
	  in_file->audio_be = 1;
	  break;
	default:
          in_file->acodec = ACODEC_UNKNOWN;
          in_file->audio_samples = 0;
          in_file->format = 0x0;
	  in_file->audio_bytes = 0;
          in_file->audio_frames = 0;
          in_file->audio_frame_samples = 0;
	break;
      }
      check_audio(in_file);
    }
#endif /* NDEVIEW_SOURCE */
  } else {
    in_file->file_format = FILE_FORMAT_UNKNOWN;
  }

  return in_file->file_format;
}

int
check_in_files(void)
{
  static int done = 0;
  int i;

  if (done)
    return 0;

  if (in_file == NULL) {
    fprintf(stderr, "check_in_files: no input file.\n");
    exit(1);
  }

  for (i = 0; in_file[i] != NULL; i++) {
    if (open_in_file(in_file[i]) == FILE_FORMAT_UNKNOWN) {
      fprintf(stderr, "check_in_file: cannot open %s.\n", in_file[i]->name);
      quit_ifile();
      quit_ofile();
      exit (1);
        //return -1;
    }

    in_file[i]->frames = calc_frames(in_file[i]->actual_frames, in_file[i]->cut_blocks, in_file[i]->cut_blocks_num);
    in_file[i]->width = in_file[i]->actual_width - in_file[i]->crop_left - in_file[i]->crop_right;
    in_file[i]->height = in_file[i]->actual_height - in_file[i]->crop_top - in_file[i]->crop_bottom;
    if (in_file[i]->width <= 0 || in_file[i]->height <= 0) {
      fprintf(stderr, "check_in_files: %s invalid dimension width %d, height %d\n", in_file[i]->name, in_file[i]->width, in_file[i]->height);
      //in_file[i]->avifile = NULL;
      quit_ifile();
      quit_ofile();
      exit (1);
      //return -1;
    }

    close_in_file(in_file[i]);
  }

  done = 1;
  return 0;
}

IN_FILE*
ifile_get_in_file(int index)
{
  if (index >= in_file_num || index < 0)
    return NULL;
  return in_file[index];
}

int
ifile_calc_total_frames(double out_fps)
{
  int i;
  total_frames = 0;
  for (i = 0; in_file[i] != NULL; i++) {
    double fps_ratio = out_fps / in_file[i]->fps;
    in_file[i]->read_frames =
      (int)((double)(in_file[i]->frames) * fps_ratio + 1e-5);
    total_frames += in_file[i]->read_frames;
  }
  return total_frames;
}

int
ifile_calc_total_audio_bytes(void)
{
  int i;
  total_audio_bytes = 0;
  for (i = 0; in_file[i] != NULL; i++) {
    total_audio_bytes += in_file[i]->audio_bytes;
  }
  return total_audio_bytes;
}

int
init_ifile(void)
{
  static int done = 0;

  if (done)
    return 0;

  if (check_in_files() < 0) {
    fprintf(stderr, "init_ifile: check_in_files error.\n");
    quit_ifile();
    quit_ofile();
    exit(1);
  }

  done = 1;
  return 0;
}

void
quit_ifile(void)
{
  int i;

  if (in_file) {
    for (i = 0; i < in_file_num; i++) {
      close_in_file(in_file[i]);
    }
    free_in_files();
  }
}

int
get_in_file_num(void)
{
  return in_file_num;
}

IN_FILE *
set_in_file_video(int index)
{
  if (index < 0 || index >= in_file_num)
    return NULL;

  if (in_file[index]->refer_count > 0) {
    return in_file[index];
  }

  if (open_in_file(in_file[index]) == FILE_FORMAT_UNKNOWN) {
    fprintf(stderr,"set_in_file_video: cannot open %s.\n",in_file[index]->name);
    quit_ifile();
    quit_ofile();
    return NULL;
  }
  return in_file[index];
}

IN_FILE *
set_in_file_audio(int index)
{
  if (index < 0 || index >= in_file_num)
    return NULL;

  if (in_file[index]->refer_count > 0) {
    return in_file[index];
  }

  if (open_in_file(in_file[index]) == FILE_FORMAT_UNKNOWN) {
    fprintf(stderr,"set_in_file_audio: cannot open %s.\n",in_file[index]->name);
    quit_ifile();
    quit_ofile();
    return NULL;
  }
  return in_file[index];
}

void
unset_in_file_video(int index)
{
  close_in_file(in_file[index]);
}

void
unset_in_file_audio(int index)
{
  close_in_file(in_file[index]);
}

int
set_in_file_video_frame_position(IN_FILE *i_file, int position)
{
  if (i_file->file_format == FILE_FORMAT_AVI) {
    return AVI_set_video_position(i_file->avifile, position);
  } else if (i_file->file_format == FILE_FORMAT_MOV) {
    return qtime_track_set_video_position(i_file->vqtrk, position);
  }
  return -1;
}

int
read_video_data(IN_FILE *i_file, unsigned char *read_buf, int *keyframe)
{
  if (i_file->file_format == FILE_FORMAT_AVI) {
    return AVI_read_frame(i_file->avifile, read_buf, keyframe);
  } else if (i_file->file_format == FILE_FORMAT_MOV) {
    return qtime_track_read_video(i_file->vqtrk, read_buf, keyframe);
  }
  return -1;
}

int
prev_key_frame (IN_FILE *i_file, int pos)
{
  if (pos >= i_file->actual_frames) {
    pos = i_file->actual_frames;
  } else if (pos < 0)
    return 0;
    
  if (i_file->file_format == FILE_FORMAT_AVI) {
    avi_t *avi = i_file->avifile;
    while (avi->video_index[pos].key != 0x10) {
      pos--;
      if (pos < 0) {
        pos = 0;
        break;
      }
    }
  } else if (i_file->file_format == FILE_FORMAT_MOV) {
    while (!qtime_track_video_is_keyframe(i_file->vqtrk, pos)) {
      pos--;
      if (pos < 0) {
        pos = 0;
        break;
      }
    }
  } else {
    pos = -1;
  }

  return pos;
}

int
next_key_frame (IN_FILE *i_file, int pos)
{
  if (pos >= i_file->actual_frames) {
    pos = i_file->actual_frames;
  } else if (pos < 0)
    return 0;
    
  if (i_file->file_format == FILE_FORMAT_AVI) {
    avi_t *avi = i_file->avifile;
    while (avi->video_index[pos].key != 0x10) {
      pos++;
      if (pos >= avi->video_frames) {
        pos = prev_key_frame (i_file, avi->video_frames);
        break;
      }
    }
  } else if (i_file->file_format == FILE_FORMAT_MOV) {
    while (!qtime_track_video_is_keyframe(i_file->vqtrk, pos)) {
      pos++;
      if (pos >= i_file->actual_frames) {
        pos = prev_key_frame (i_file, i_file->actual_frames);
	break;
      }
    }
  } else {
    pos = -1;
  }

  return pos;
}

int
read_audio_data(IN_FILE *i_file, unsigned char *buf, int *num_samples)
{
  int size;
  if (i_file->file_format == FILE_FORMAT_AVI) {
    if (i_file->acodec == ACODEC_PCM || i_file->acodec == ACODEC_PCM_LE)
      size = i_file->audio_frame_samples * i_file->bytespspl;
    else
      size = i_file->audio_frame_samples;
    size = AVI_read_audio(i_file->avifile, buf, size);
    if (i_file->acodec == ACODEC_PCM || i_file->acodec == ACODEC_PCM_LE)
      *num_samples = size / i_file->bytespspl;
    else
      *num_samples = -1;
    return size;
  } else if (i_file->file_format == FILE_FORMAT_MOV) {
    if ((i_file->acodec == ACODEC_PCM || i_file->acodec == ACODEC_PCM_LE ||
	i_file->acodec == ACODEC_PCM_BE) && *num_samples <= 0)
      *num_samples = i_file->audio_frame_samples;
    return qtime_track_read_audio(i_file->aqtrk, buf, num_samples);
  }
  return -1;
}

int
set_audio_position(IN_FILE *i_file, int pos)
{
  return AVI_set_audio_position(i_file->avifile, pos);
}

int
set_audio_position_start(IN_FILE *i_file)
{
  if (i_file->file_format == FILE_FORMAT_AVI) {
    return AVI_set_audio_position(i_file->avifile, 0);
  } else if (i_file->file_format == FILE_FORMAT_MOV) {
    return qtime_track_set_audio_position(i_file->aqtrk, 0);
  }
  return -1;
}

