/* mgr.c */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>

#include "state.h"
#include "vgrab.h"
#include "venc.h"
#include "vframe_mgr.h"
#include "ffmt.h"
#include "vcodec.h"
#include "parseopt.h"
#include "csp.h"
#include "aread.h"
#include "acodec.h"
#include "aenc.h"
#include "v4l.h"
#include "oss.h"
#include "preset.h"
//#include "vc_xvid.h"

#include "mgr.h"

#ifndef NDEBUG
#define debug(str) fprintf(stderr, str);fflush(stderr)
#else
#define debug(str)
#endif /* NDEBUG */

static MGR_CONFIG mgr_config;

static SEL_COMPO video_codecs[] = {
#ifdef HAVE_LIBXVIDCORE
  { "XVID",  (int)VCODEC_XVID },
#endif /* HAVE_LIBXVIDCORE */
  { "RGB24", (int)VCODEC_RGB24 },
  { "YV12",  (int)VCODEC_YV12 },
#ifdef HAVE_RTJPEG
  { "RTJPEG",  (int)VCODEC_RTJPEG },
#endif /* HAVE_RTJPEG */
#ifdef HAVE_LIBAVCODEC
  { "MPEG1",  (int)VCODEC_MPEG1 },
  { "H263",  (int)VCODEC_H263 },
  { "H263P",  (int)VCODEC_H263P },
  { "MJPEG",  (int)VCODEC_MJPEG },
  { "MPEG4",  (int)VCODEC_MPEG4 },
  { "MSMPEG4V1",  (int)VCODEC_MSMPEG4V1 },
  { "MSMPEG4V2",  (int)VCODEC_MSMPEG4V2 },
  { "MSMPEG4V3",  (int)VCODEC_MSMPEG4V3 },
  { "WMV1",  (int)VCODEC_WMV1 },
  { "WMV2",  (int)VCODEC_WMV2 },
  { "HUFFYUV",  (int)VCODEC_HUFFYUV },
#endif /* HAVE_LIBAVCODEC */
};

#define NTVREC_PRESET_FILENAME "~/.ntvrec/ntvrec.preset"

static int
opt_set_preset(const char *arg)
{
  if (preset_init(NTVREC_PRESET_FILENAME) != OK)
    return OK;
//    return FAIL;
  if (preset_set(arg) != OK)
    return FAIL;
  if (preset_quit() != OK)
    return FAIL;
  return OK;
}

static int
opt_set_fps (const char *arg)
{
  mgr_config.fps = strtod (arg, 0);
  return 0;
}

static int
opt_set_size (const char *arg)
{
  char *p;
  char *ep;
  int width;
  int height;

  p = (char*)arg;
  width = strtol(p, &ep, 10);
  if (ep == NULL) {
    fprintf (stderr, "option frame size error. '%s'\n", arg);
    return -1;
  }
  p = ++ep;
  height = strtol(p, &ep, 10);
  if (width <= 0 || height <= 0) {
    fprintf(stderr, "Incorrect frame size\n");
    return -1;
  }
  if ((width % 16) != 0) {
    width = width - (width % 16);
    fprintf(stderr, "Frame width must be a multiple of 16, %d\n", width);
    fprintf(stderr, "force width %d\n", width);
  }
  if ((height % 16) != 0) {
    height = height - (height % 16);
    fprintf(stderr, "Frame height must be a multiple of 16, %d\n", height);
    fprintf(stderr, "force height %d\n", height);
  }
  mgr_config.width = width;
  mgr_config.height = height;
  return 0;
}

char *use_fourcc = NULL;
const char *fourcc_str = NULL;

int
opt_set_use_fourcc (const char *arg)
{
  if (use_fourcc)
    free(use_fourcc);
  use_fourcc = (char*)strndup(arg, 4);
  return 0;
}

static SEL_COMPO grab_csps[] = {
  { "I420",  CSP_I420 },
  { "RGB24", CSP_RGB24 },
  { "RGB32", CSP_RGB32 },
  { "YV12",  CSP_YV12 },
//  { "YV12_INTERLACE",  CSP_YV12_INTERLACE},
};

OptionDef mgr_param[] = {
#ifdef HAVE_LIBXVIDCORE
  {"vcodec", "VIDEO_CODEC", HAS_ARG|OPT_SEL, {(void*)&mgr_config.vcodec}, {VCODEC_XVID}, {(int)video_codecs}, sizeof(video_codecs)/sizeof(SEL_COMPO), "Output video codec", "codec"},
#else
  {"vcodec", "VIDEO_CODEC", HAS_ARG|OPT_SEL, {(void*)&mgr_config.vcodec}, {VCODEC_YV12}, {(int)video_codecs}, sizeof(video_codecs)/sizeof(SEL_COMPO), "Output video codec", "codec"},
#endif /* HAVE_LIBXVIDCORE */
  {"fps", "VIDEO_FPS", HAS_ARG|OPT_FUNC, {(void*)&opt_set_fps}, {0}, {0}, 0, "Video frames/second", "r"},
  {"s", "VIDEO_SIZE", HAS_ARG|OPT_FUNC, {(void*)&opt_set_size}, {(int)"320x240"}, {0}, 0, "frame size (WxH)", "WxH"},
  {"max-buffers", "VIDEO_MAX_BUFFERS", HAS_ARG, {(void*)&mgr_config.max_buffers}, {32}, {4}, 64, "Max number of buffers", "b"},
  {"use-fourcc", "USE_FOURCC", HAS_ARG|OPT_FUNC, {(void*)&opt_set_use_fourcc}, {0}, {0}, 5, "use fourcc", "f"},
  {"grab-csp", "GRAB_CSP", HAS_ARG|OPT_SEL, {(void*)&mgr_config.grab_csp}, {CSP_UNKNOWN}, {(int)grab_csps}, sizeof(grab_csps)/sizeof(SEL_COMPO), "color space specified when grabing", "b"},
  {"preset", NULL, HAS_ARG|OPT_FUNC|OPT_BEFORE, {(void*)&opt_set_preset}, {0}, {0}, 0, "read preset from preset file.", "name"},
};

int mgr_param_num = sizeof(mgr_param) / sizeof(OptionDef);

static SEL_COMPO vd_norm[] = {
  { "PAL",  NORM_PAL },
  { "NTSC",  NORM_NTSC },
  { "SECAM",  NORM_SECAM },
  { "AUTO",  NORM_AUTO },
};

OptionDef vdev_param[] = {
  {"vd", "VIDEO_DEVICE", HAS_ARG|OPT_STR, {(void*)&mgr_config.vdev_name}, {(int)"/dev/video0"}, {0}, 1024, "video device", "name"},
  {"vd-source", "VIDEO_DEVICE_SOURCE", HAS_ARG|OPT_INT, {(void*)&mgr_config.vdev_source_channel}, {0}, {0}, 4, "video device source [0,1,...]", "num"},
  {"vd-tuner", "VIDEO_DEVICE_TUNER", HAS_ARG|OPT_INT, {(void*)&mgr_config.vdev_tuner}, {0}, {0}, 4, "video device tuner [0,1,...]", "num"},
  {"vd-norm", "VIDEO_DEVICE_NORM", HAS_ARG|OPT_SEL, {(void*)&mgr_config.vdev_norm}, {NORM_NTSC}, {(int)vd_norm}, sizeof(vd_norm)/sizeof(SEL_COMPO), "video device norm", "norm"},
  {"vd-audio", "VIDEO_DEVICE_AUDIO", HAS_ARG|OPT_INT, {(void*)&mgr_config.vdev_audio}, {0}, {0}, 4, "video device audio [0,1,...]", "num"},
  {"vd-brightness", "VIDEO_DEVICE_BRIGHTNESS", HAS_ARG|OPT_INT, {(void*)&mgr_config.vdev_brightness}, {-1}, {-1}, 100, "video device brightness", "num"},
  {"vd-hue", "VIDEO_DEVICE_HUE", HAS_ARG|OPT_INT, {(void*)&mgr_config.vdev_hue}, {-1}, {-1}, 100, "video device hue", "num"},
  {"vd-color", "VIDEO_DEVICE_COLOUR", HAS_ARG|OPT_INT, {(void*)&mgr_config.vdev_colour}, {-1}, {-1}, 100, "video device color", "num"},
  {"vd-contrast", "VIDEO_DEVICE_CONTRAST", HAS_ARG|OPT_INT, {(void*)&mgr_config.vdev_contrast}, {-1}, {-1}, 100, "video device contrast", "num"},
  {"vd-whiteness", "VIDEO_DEVICE_WHITENESS", HAS_ARG|OPT_INT, {(void*)&mgr_config.vdev_whiteness}, {-1}, {-1}, 100, "video device whiteness", "num"},
  {"vd-freq", "VIDEO_DEVICE_FREQ", HAS_ARG|OPT_INT, {(void*)&mgr_config.vdev_freq}, {-1}, {-1}, INT_MAX, "video device tuner frequency", "num"},
};
int vdev_param_num = sizeof(vdev_param) / sizeof(OptionDef);

static SEL_COMPO audio_codecs[] = {
#ifdef HAVE_LIBMP3LAME
  {"MP3", ACODEC_MP3},
#endif /* HAVE_LIBMP3LAME */
  {"PCM", ACODEC_PCM},
#ifdef HAVE_OGGVORBIS
  {"oggvorbis", ACODEC_OGGVORBIS},
#endif /* HAVE_OGGVORBIS */
};

#if 0
static SEL_COMPO audio_formats[] = {
  {"U8", AUDIO_FMT_U8},
  {"S16LE", AUDIO_FMT_S16_LE}
};
#endif

OptionDef audio_param[] = {
  {"ad", "AUDIO_DEVICE", HAS_ARG|OPT_STR, {(void*)&mgr_config.adev_name}, {(int)"/dev/dsp"}, {0}, 1024, "audio device", "name"},
  {"ad-mixer", "AUDIO_MIXER_DEVICE", HAS_ARG|OPT_STR, {(void*)&mgr_config.adev_mixer_name}, {(int)"/dev/mixer"}, {0}, 1024, "audio mixer device", "name"},
  {"ad-source", "AUDIO_DEVICE_SOURCE", HAS_ARG|OPT_STR, {(void*)&mgr_config.adev_source}, {(int)NULL}, {0}, 32, "audio record source", "name"},

#ifdef HAVE_LIBMP3LAME
  {"acodec", "AUDIO_CODEC", HAS_ARG|OPT_SEL, {(void*)&mgr_config.audio_codec}, {ACODEC_MP3}, {(int)audio_codecs}, sizeof(audio_codecs)/sizeof(SEL_COMPO), "force audio codec ", "c"},
#else
  {"acodec", "AUDIO_CODEC", HAS_ARG|OPT_SEL, {(void*)&mgr_config.audio_codec}, {ACODEC_PCM}, {(int)audio_codecs}, sizeof(audio_codecs)/sizeof(SEL_COMPO), "force audio codec ", "c"},
#endif /* HAVE_LIBMP3LAME */
  {"audio-rate", "AUDIO_RATE", HAS_ARG, {(void*)&mgr_config.audio_rate}, {22050}, {0}, 96000, "set audio sampling rate [in Hz]", "r"},
  {"audio-channels", "AUDIO_CHANNEL", HAS_ARG, {(void*)&mgr_config.audio_channels}, {2}, {0}, 2, "set number of audio channels", "c"},
//  {"audio_format", "AUDIO_FORMAT", HAS_ARG|OPT_SEL, {(void*)&mgr_config.audio_format}, {AUDIO_FMT_S16_LE}, {(int)audio_formats}, sizeof(audio_formats)/sizeof(SEL_COMPO), "set audio format ", "c"},
};

int audio_param_num = sizeof(audio_param) / sizeof(OptionDef);

static SEL_COMPO fmts[] = {
  { "AVI",  FFMT_FORMAT_AVI },
  { "RAW",  FFMT_FORMAT_RAW },
  { "MOV",  FFMT_FORMAT_MOV},
};

int
mgr_set_output_file_name (const char *arg)
{
  mgr_config.output_fname = (char*)strndup(arg, MAX_FILE_NAME_LENGTH);
  return 0;
}

OptionDef ffmt_param[] = {
  {"file-format", "FILE_FORMAT", HAS_ARG|OPT_SEL, {(void*)&mgr_config.file_format}, {FFMT_FORMAT_MOV}, {(int)fmts}, sizeof(fmts)/sizeof(SEL_COMPO), "Output file format", "sel"},
  {"o", NULL, HAS_ARG|OPT_FUNC, {(void*)&mgr_set_output_file_name}, {0}, {0}, MAX_FILE_NAME_LENGTH, "Output file name", "name"},
};

int ffmt_param_num = sizeof(ffmt_param) / sizeof(OptionDef);

int
mgr_state_start(void)
{
  MGR_CONFIG *mgr_conf = &mgr_config;
  int retval = MGR_FAIL;
  if (mgr_conf->state == STATE_START)
    return MGR_OK;
  if (vgrab_state_start() != VGRAB_OK) {
    fprintf(stderr, "mgr_state_start: vgrab_state_start failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_start: vgrab_state_start done\n");
  if (aread_state_start() != AREAD_OK) {
    fprintf(stderr, "mgr_state_start: aread_state_start failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_start: aread_state_start done\n");
  if (venc_state_start() != VENC_OK) {
    fprintf(stderr, "mgr_state_start: venc_state_start failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_start: venc_state_start done\n");
  if (aenc_state_start() != AENC_OK) {
    fprintf(stderr, "mgr_state_start: aenc_state_start failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_start: aend_state_start done\n");
  if (ffmt_state_start() != FFMT_OK) {
    fprintf(stderr, "mgr_state_start: ffmt_state_start failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_start: ffmt_state_start done\n");
  mgr_conf->state = STATE_START;
//  debug("mgr_state_start: done\n");
  return MGR_OK;
}

int
mgr_state_pause(void)
{
  MGR_CONFIG *mgr_conf = &mgr_config;
  int retval = MGR_FAIL;
  if (mgr_conf->state != STATE_START)
    return MGR_OK;
  if (vgrab_state_pause() != VGRAB_OK) {
    fprintf(stderr, "mgr_state_pause: vgrab_state_pause failed.\n");
    mgr_error_quit();
    return retval;
  }
  if (aread_state_pause() != AREAD_OK) {
    fprintf(stderr, "mgr_state_pause: aread_state_pause failed.\n");
    mgr_error_quit();
    return retval;
  }
  if (venc_state_pause() != VENC_OK) {
    fprintf(stderr, "mgr_state_pause: venc_state_pause failed.\n");
    mgr_error_quit();
    return retval;
  }
  if (aenc_state_pause() != AENC_OK) {
    fprintf(stderr, "mgr_state_pause: aenc_state_pause failed.\n");
    mgr_error_quit();
    return retval;
  }
  if (ffmt_state_pause() != FFMT_OK) {
    fprintf(stderr, "mgr_state_pause: ffmt_state_pause failed.\n");
    mgr_error_quit();
    return retval;
  }
  mgr_conf->state = STATE_PAUSE;
  debug("mgr_state_pause: done.\n");
  return MGR_OK;
}

int
mgr_state_toggle_pause(void)
{
  MGR_CONFIG *mgr_conf = &mgr_config;
  int retval = MGR_FAIL;
  if (mgr_conf->state != STATE_START &&
      mgr_conf->state != STATE_PAUSE)
    return MGR_OK;
  if (vgrab_state_toggle_pause() != VGRAB_OK) {
   fprintf(stderr,"mgr_state_toggle_pause: vgrab_state_toggle_pause failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_toggle_pause: vgrab_state_toggle_pause done.\n");
  if (venc_state_toggle_pause() != VENC_OK) {
    fprintf(stderr,"mgr_state_toggle_pause: venc_state_toggle_pause failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_toggle_pause: venc_state_toggle_pause done.\n");
  if (aenc_state_toggle_pause() != AENC_OK) {
    fprintf(stderr,"mgr_state_toggle_pause: aenc_state_toggle_pause failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_toggle_pause: aenc_state_toggle_pause done.\n");
  if (ffmt_state_toggle_pause() != FFMT_OK) {
    fprintf(stderr,"mgr_state_toggle_pause: ffmt_state_toggle_pause failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_toggle_pause: ffmt_state_toggle_pause done.\n");
  if (mgr_conf->state == STATE_START)
    mgr_conf->state = STATE_PAUSE;
  else if (mgr_conf->state == STATE_PAUSE)
    mgr_conf->state = STATE_PAUSE;
  else
    return MGR_FAIL;
  debug("mgr_state_toggle_pause: done.\n");
  return MGR_OK;
}

int
mgr_state_stop(void)
{
  MGR_CONFIG *mgr_conf = &mgr_config;
  int retval = MGR_FAIL;
  if (mgr_conf->state == STATE_STOP)
    return MGR_OK;
  if (vgrab_state_stop() != VGRAB_OK) {
    fprintf(stderr, "mgr_state_stop: vgrab_state_stop failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_stop: vgrab_state_stop done\n");
  if (aread_state_stop() != AREAD_OK) {
    fprintf(stderr, "mgr_state_stop: aread_state_stop failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_stop: aread_state_stop done\n");
  if (venc_state_stop() != VENC_OK) {
    fprintf(stderr, "mgr_state_stop: venc_state_stop failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_stop: venc_state_stop done\n");
  if (aenc_state_stop() != AENC_OK) {
    fprintf(stderr, "mgr_state_stop: aenc_state_stop failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_stop: aenc_state_stop done\n");
  if (ffmt_state_stop() != FFMT_OK) {
    fprintf(stderr, "mgr_state_stop: ffmt_state_stop failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_stop: ffmt_state_stop done\n");
  mgr_conf->state = STATE_STOP;
  debug("mgr_state_stop: done\n");
  return MGR_OK;
}

int
mgr_state_quit(void)
{
  MGR_CONFIG *mgr_conf = &mgr_config;
  int retval = MGR_FAIL;

  if (mgr_conf->state == STATE_QUIT)
    return MGR_OK;
  if (vgrab_state_quit() != VGRAB_OK) {
    fprintf(stderr, "mgr_state_quit: vgrab_state_quit failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_quit: vgrab_state_quit done\n");
  if (aread_state_quit() != AREAD_OK) {
    fprintf(stderr, "mgr_state_quit: aread_state_quit failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_quit: aread_state_quit done\n");
  if (venc_state_quit() != VENC_OK) {
    fprintf(stderr, "mgr_state_quit: venc_state_quit failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_quit: venc_state_quit done\n");
  if (aenc_state_quit() != AENC_OK) {
    fprintf(stderr, "mgr_state_quit: aenc_state_quit failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_quit: aenc_state_quit done\n");
  if (ffmt_state_quit() != FFMT_OK) {
    fprintf(stderr, "mgr_state_quit: ffmt_state_quit failed.\n");
    mgr_error_quit();
    return retval;
  }
//  debug("mgr_state_quit: ffmt_state_quit done\n");
  mgr_conf->state = STATE_QUIT;
  debug("mgr_state_quit: done\n");
  return MGR_OK;
}

int
mgr_state_error(void)
{
  MGR_CONFIG *mgr_conf = &mgr_config;
//  int retval = MGR_FAIL;

  if (mgr_conf->state == STATE_QUIT)
    return MGR_OK;
  if (vgrab_state_error() != VGRAB_OK) {
    fprintf(stderr, "mgr_state_error: vgrab_state_error failed.\n");
//    return retval;
  }
//  debug("mgr_state_error: vgrab_state_error done\n");
  if (aread_state_error() != AREAD_OK) {
    fprintf(stderr, "mgr_state_error: aread_state_error failed.\n");
//    return retval;
  }
//  debug("mgr_state_error: aread_state_error done\n");
  if (venc_state_error() != VENC_OK) {
    fprintf(stderr, "mgr_state_error: venc_state_error failed.\n");
//    return retval;
  }
//  debug("mgr_state_error: venc_state_error done\n");
  if (aenc_state_error() != AENC_OK) {
    fprintf(stderr, "mgr_state_error: aenc_state_error failed.\n");
//    return retval;
  }
//  debug("mgr_state_error: aenc_state_error done\n");
  if (ffmt_state_error() != FFMT_OK) {
    fprintf(stderr, "mgr_state_error: ffmt_state_error failed.\n");
//    return retval;
  }
//  debug("mgr_state_error: ffmt_state_error done\n");
  mgr_conf->state = STATE_ERROR;
  debug("mgr_state_error: done\n");
  return MGR_OK;
}

void
mgr_set_restart(void)
{
  vgrab_set_restart();
}

int
mgr_error_quit(void)
{
  MGR_CONFIG *mgr_conf = &mgr_config;
  int ret = MGR_OK;

  if (mgr_conf->state != STATE_ERROR)
    ret = mgr_state_error();
  vgrab_error_quit();
//  debug("mgr_error_quit: vgrab_error_quit done\n");
  venc_error_quit();
//  debug("mgr_error_quit: venc_error_quit done\n");
  aread_error_quit();
//  debug("mgr_error_quit: aread_error_quit done\n");
  aenc_error_quit();
//  debug("mgr_error_quit: aenc_error_quit done\n");
  ffmt_error_quit();
//  debug("mgr_error_quit: ffmt_error_quit done\n");

  if (vframe_mgr_error_quit() == VFRAME_MGR_FAIL) {
    fprintf(stderr, "mgr_error_quit: vframe_mgr_error_quit() failed\n");
    //exit(1);
  }
  vcodec_encode_quit();
  v4l_quit();

  if (aframe_mgr_error_quit() == AFRAME_MGR_FAIL) {
    fprintf(stderr, "mgr_error_quit: aframe_mgr_error_quit() failed\n");
    //exit(1);
  }
  acodec_encode_quit();
  oss_quit();

  debug("mgr_error_quit: done.\n");

  return ret;
}

int
mgr_quit(void)
{
  MGR_CONFIG *mgr_conf = &mgr_config;
  int ret = MGR_OK;

  if (mgr_conf->state != STATE_QUIT)
    ret = mgr_state_quit();
  vgrab_quit();
//  debug("mgr_quit: vgrab_quit done\n");
  venc_quit();
//  debug("mgr_quit: venc_quit done\n");
  aread_quit();
//  debug("mgr_quit: aread_quit done\n");
  aenc_quit();
//  debug("mgr_quit: aenc_quit done\n");
  ffmt_quit();
//  debug("mgr_quit: ffmt_quit done\n");

  if (vframe_mgr_quit() == VFRAME_MGR_BUSY) {
    fprintf(stderr, "mgr_quit: vframe_mgr_quit() failed\n");
    exit(1);
  }
  vcodec_encode_quit();
  v4l_quit();

  if (aframe_mgr_quit() == AFRAME_MGR_BUSY) {
    fprintf(stderr, "mgr_quit: aframe_mgr_quit() failed\n");
    exit(1);
  }
  acodec_encode_quit();
  oss_quit();

  debug("mgr_quit: done.\n");
  return ret;
}

static int
mgr_video_init(MGR_CONFIG *mgr_conf)
{
  int ret;
  int width, height;
  int grab_csp, v4l_csp;
  double fps;
  int in_depth, out_depth, max_pixel_depth;

//  vc_xvid_init();

  if (vcodec_encode_set_codec(mgr_conf->vcodec) == VCODEC_FAIL) {
    fprintf(stderr, "mgr_video_init: vcodec_encode_set_codec failed\n");
    return MGR_FAIL;
  }
  grab_csp = vcodec_encode_csp_cap(mgr_conf->grab_csp);
  if (grab_csp == CSP_UNKNOWN) {
    fprintf(stderr, "mgr_video_init: grab_csp invalied.\n");
    return MGR_FAIL;
  }
  mgr_conf->grab_csp = grab_csp;

  ret = v4l_init(mgr_conf->vdev_name, mgr_conf->vdev_source_channel, mgr_conf->vdev_tuner, mgr_conf->vdev_norm, mgr_conf->vdev_audio);
  if (ret != V4L_OK) {
    fprintf(stderr, "mgr_video_init: v4l_init failed.\n");
    return MGR_FAIL;
  }

  width = mgr_conf->width;
  height = mgr_conf->height;
  if (v4l_set_window_size(width, height) == V4L_FAIL) {
    fprintf(stderr, "mgr_video_init: v4l_set_window_size() failed.\n");
    v4l_quit();
    return MGR_FAIL;
  }
  if (v4l_get_window_size(&width, &height) == V4L_FAIL) {
    fprintf(stderr, "mgr_video_init: v4l_get_window_size() failed.\n");
    v4l_quit();
    return MGR_FAIL;
  }
  mgr_conf->width = width;
  mgr_conf->height = height;

  v4l_set_brightness(mgr_conf->vdev_brightness);
  v4l_set_hue(mgr_conf->vdev_hue);
  v4l_set_colour(mgr_conf->vdev_colour);
  v4l_set_contrast(mgr_conf->vdev_contrast);
  v4l_set_whiteness(mgr_conf->vdev_whiteness);

  if (mgr_conf->vdev_freq > 0) {
//    printf("set frequency: %d\n", mgr_conf->vdev_freq);
    v4l_set_freq(mgr_conf->vdev_freq * 16 / 1000);
  }

  v4l_csp = vgrab_csp_to_vdev_csp(grab_csp);
//  printf("v4l_csp %d\n", v4l_csp);
  if (v4l_set_csp(v4l_csp) == V4L_FAIL) {
    fprintf(stderr, "mgr_video_init: v4l_set_csp() failed.\n");
    v4l_quit();
    return MGR_FAIL;
  }

  fps = mgr_conf->fps;
  ret = vgrab_init(width, height, fps, grab_csp);
  if (ret != VGRAB_OK) {
    fprintf(stderr, "mgr_video_init: vgrab_init failed.\n");
    v4l_quit();
    return MGR_FAIL;
  }

  if (fps == 0) {
    switch (v4l_get_norm()) {
      case NORM_NTSC:
	fps = 29.97;
	break;
      default:
	fps = 25;
	break;
    }
  }
  if (vcodec_encode_init(mgr_conf->vcodec, width, height, fps, grab_csp) != VCODEC_OK){
    fprintf(stderr, "mgr_video_init: vcodec_encode_init failer.\n");
    vgrab_quit();
    v4l_quit();
    return MGR_FAIL;
  }

  mgr_conf->out_csp = vcodec_encode_get_out_csp();
  in_depth = csp_to_pixel_depth(grab_csp);
  out_depth = csp_to_pixel_depth(mgr_conf->out_csp);
  max_pixel_depth = (in_depth > out_depth) ? in_depth : out_depth;

  ret = vframe_mgr_init(width, height, max_pixel_depth, mgr_conf->max_buffers);
  vframe_set_enc_wait_cb(venc_enc_wait_cb);
  vframe_set_dst_wait_cb(ffmt_dst_wait_cb);

  ret = venc_init(width, height, fps, mgr_conf->vcodec, grab_csp);
  if (ret != VENC_OK) {
    fprintf(stderr, "mgr_video_init: venc_init failer.\n");
    vgrab_quit();
    vcodec_encode_quit();
    vframe_mgr_quit();
    v4l_quit();
    return MGR_FAIL;
  }

  return MGR_OK;
}

int
mgr_audio_init(MGR_CONFIG *mgr_conf)
{
  int rate, channels, bits;
  int aframe_spl_size;
  int ret;

  rate = mgr_conf->audio_rate;
  channels = mgr_conf->audio_channels;
  bits = mgr_conf->audio_bits;

  if (acodec_encode_set_codec(mgr_conf->audio_codec) == ACODEC_FAIL) {
    fprintf(stderr, "mgr_audio_init: acodec_encode_set_codec failed\n");
    return MGR_FAIL;
  }

  if ((rate = acodec_encode_rate_cap(mgr_conf->audio_rate)) != mgr_conf->audio_rate) {
    fprintf(stderr, "mgr_audio_init: audio rate failed\n");
    return MGR_FAIL;
  }
  if ((channels = acodec_encode_channels_cap(mgr_conf->audio_channels)) != mgr_conf->audio_channels) {
    fprintf(stderr, "mgr_audio_init: audio channels failed\n");
    return MGR_FAIL;
  }
  mgr_conf->audio_bits = 16;
  if ((bits = acodec_encode_bits_cap(mgr_conf->audio_bits)) != mgr_conf->audio_bits) {
    fprintf(stderr, "mgr_audio_init: audio bits failed\n");
    return MGR_FAIL;
  }

  if (oss_mixer_open(mgr_conf->adev_mixer_name) == OSS_FAIL) {
    fprintf(stderr, "mgr_audio_init: audio mixer device %s cannot open.\n", mgr_conf->adev_mixer_name);
    return MGR_FAIL;
  }
  if (mgr_conf->adev_source != NULL) {
    int id;
    if ((id = oss_mixer_dev_id(mgr_conf->adev_source)) == OSS_FAIL) {
      fprintf(stderr, "mgr_audio_init: audio mixer %s cannot find.\n", mgr_conf->adev_source);
      oss_mixer_close();
      return MGR_FAIL;
    }
    if (oss_mixer_record_enable(id) == OSS_FAIL) {
      fprintf(stderr, "mgr_audio_init: audio mixer record source cannot set, %s.\n", mgr_conf->adev_source);
      oss_mixer_close();
      return MGR_FAIL;
    }
  }
  oss_mixer_close();

  if (oss_init(mgr_conf->adev_name, rate, channels, bits) < 0) {
    fprintf(stderr,"mgr_audio_init: oss_init fialed.\n");
    return MGR_FAIL;
  }
  rate = oss_get_rate();
  channels = oss_get_channels();
  bits = oss_get_bitspspl();
  if (rate != mgr_conf->audio_rate || channels != mgr_conf->audio_channels ||
      bits != mgr_conf->audio_bits) {
    fprintf(stderr,"mgr_audio_init: oss_init parameter fialed.\n");
    oss_quit();
    return MGR_FAIL;
  }

  if (acodec_encode_init(mgr_conf->audio_codec,rate,channels,bits) != ACODEC_OK){
    fprintf(stderr, "mgr_audio_init: acodec_encode_init failed\n");
    oss_quit();
    return MGR_FAIL;
  }

  aframe_spl_size = acodec_encode_get_frame_size();
  if (aframe_mgr_init(rate,channels,bits,aframe_spl_size) != AFRAME_MGR_OK){
    fprintf(stderr, "mgr_audio_init: aframe_mgr_audio_init failed\n");
    acodec_encode_quit();
    oss_quit();
    return MGR_FAIL;
  }
  aframe_set_enc_wait_cb(aenc_enc_wait_cb);
  aframe_set_dst_wait_cb(ffmt_dst_wait_cb);

  ret = aread_init(mgr_conf->audio_rate, mgr_conf->audio_channels, mgr_conf->audio_bits);
  if (ret != AREAD_OK) {
    fprintf(stderr, "mgr_audio_init: aread_init failed\n");
    aframe_mgr_quit();
    acodec_encode_quit();
    oss_quit();
    return MGR_FAIL;
  }

  if (aenc_init(mgr_conf->audio_codec, rate, channels, bits) != AENC_OK) {
    fprintf(stderr, "mgr_audio_init: aenc_init failed\n");
    aframe_mgr_quit();
    aread_quit();
    acodec_encode_quit();
    oss_quit();
    return MGR_FAIL;
  }

  mgr_conf->audio_out_rate = acodec_encode_get_out_rate();

  return MGR_OK;
}

int
mgr_init(void)
{
  int ret;
  MGR_CONFIG *mgr_conf = &mgr_config;

  pthread_mutex_init(&mgr_conf->mutex, NULL);

  mgr_conf->state = STATE_STOP;

  if (mgr_conf->file_format == FFMT_FORMAT_MOV &&
      mgr_conf->audio_codec == ACODEC_PCM) {
    mgr_conf->audio_codec = ACODEC_PCM_BE;
  }

/* audio */
  if (mgr_audio_init(mgr_conf) == MGR_FAIL) {
    fprintf(stderr, "mgr_init: mgr_audio_init failed\n");
    mgr_error_quit();
    return MGR_FAIL;
  }

/* video */
  if (mgr_video_init(mgr_conf) == MGR_FAIL) {
    fprintf(stderr, "mgr_init: mgr_video_init failed\n");
    mgr_error_quit();
    return MGR_FAIL;
  }

/* file format */
  ret = ffmt_init(mgr_conf->output_fname, mgr_conf->file_format,
      mgr_conf->width, mgr_conf->height, mgr_conf->fps, mgr_conf->vcodec,
      mgr_conf->audio_codec, mgr_conf->audio_out_rate, mgr_conf->audio_channels,
      mgr_conf->audio_bits);
  if (ret != FFMT_OK) {
    fprintf(stderr, "mgr_init: ffmt_init failer.\n");
    mgr_error_quit();
    return MGR_FAIL;
  }

  ret = mgr_state_stop();
  if (ret != MGR_OK) {
    fprintf(stderr, "mgr_init: mgr_state_stop failer.\n");
    mgr_error_quit();
    return MGR_FAIL;
  }

  return MGR_OK;
}

static void
mgr_print_param_video(void)
{
  fprintf(stdout, "\n");
  fprintf(stdout, "VIDEO SIZE:              %dx%d\n", mgr_config.width, mgr_config.height);
  if (mgr_config.fps == 0)
    fprintf(stdout, "VIDEO FPS:               0 [auto]\n");
  else
    fprintf(stdout, "VIDEO FPS:               %f\n", mgr_config.fps);
  if (mgr_config.grab_csp != CSP_UNKNOWN)
    fprintf(stdout, "VIDEO GRAB CSP:          %s\n", csp_to_str(mgr_config.grab_csp));
  fprintf(stdout, "VIDEO MAX BUFFERS:       %d\n", mgr_config.max_buffers);
  v4l_print_param();
  vcodec_encode_print_param();
}

static void
mgr_print_param_audio(void)
{
  fprintf(stdout, "\n");
  fprintf(stdout, "AUDIO RATE:              %d\n", mgr_config.audio_rate);
  fprintf(stdout, "AUDIO CHANNELS:          %d\n", mgr_config.audio_channels);
  fprintf(stdout, "AUDIO BITS:              %d\n", mgr_config.audio_bits);
  oss_print_param();
  acodec_encode_print_param();
}

void
mgr_print_param(void)
{
  fprintf(stdout, "ntvrec version 0.0.15\n");
  mgr_print_param_video();
  mgr_print_param_audio();
  ffmt_print_param();
}

