/* stts.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_sample_info.h"
#include "atom.h"
#include "stts.h"

void
stts_init(stts_t *stts)
{
  atom_init(&stts->atom);
  stts->atom.type = QTIME_TYPE_STTS;
  stts->version = 0;
  qtime_flags_set(stts->flags, 0);
  stts->number_of_entries = 0;
  stts->table_max = 0;
  stts->table = NULL;

  stts->max_count = 0;
  stts->max_duration = 0;

  stts->idx = 0;
  stts->st_count = 0;
  stts->st_duration = 0;
  stts->ed_count = 0;
  stts->ed_duration = 0;
}

void
stts_clean(stts_t *stts)
{
  qtime_error_type_check(QTIME_TYPE_STTS, stts->atom.type)
  if (stts->table)
    qtime_free(stts->table);
  stts_init(stts);
}

stts_t*
stts_new(void)
{
  stts_t *stts;
  stts = (stts_t*)qtime_malloc(sizeof(stts_t));
  if (!stts) return NULL;
  stts_init(stts);
  return stts;
}

void
stts_delete(stts_t *stts)
{
  qtime_error_type_check(QTIME_TYPE_STTS, stts->atom.type)
  if (stts->table)
    qtime_free(stts->table);
  qtime_free(stts);
}

int
stts_valid(stts_t *stts)
{
  if (!stts || stts->number_of_entries <= 0)
    return 0;
  return 1;
}

stts_t*
stts_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, stts_t *stts_ptr)
{
  int table_byte_size;
  int i, table_num;
  time_to_sample_t *table;
  stts_t *stts;

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_STTS, atom_head->type, NULL)
  if (stts_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_STTS, stts_ptr->atom.type, NULL)
    stts = stts_ptr;
    stts_clean(stts);
  } else {
    if ((stts = stts_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  stts->atom.size = atom_head->size;
  stts->atom.type = atom_head->type;
  stts->atom.parent = atom_head->parent;

  qtime_io_read(qtio, &stts->version, 1);
  qtime_io_read(qtio, &stts->flags, 3);
  qtime_io_read32(qtio, &stts->number_of_entries);

  if (stts->number_of_entries <= 0) {
    fprintf(stderr, "QTIME_ERROR: stts_read_atom: number_of_entries invalid, %d\n", stts->number_of_entries);
    qtime_error_debug_info(QTIME_ERROR);
    atom_head->error_code = QTIME_ERROR;
    goto fail;
  }

  table_num = stts->number_of_entries;
  table_byte_size = table_num * sizeof(time_to_sample_t);
  table = (time_to_sample_t*)qtime_malloc(table_byte_size);
  if (!table) {
    qtime_error_debug_info(QTIME_ERROR);
    atom_head->error_code = QTIME_ERROR_MEMORY;
    goto fail;
  }
  stts->table = table;
  stts->table_max = table_num;

  for (i = 0; i < table_num; i++, table++) {
    qtime_io_read32(qtio, &table->sample_count);
    qtime_io_read32(qtio, &table->sample_duration);
  }

  stts->max_count = stts_get_max_count(stts);
  stts->max_duration = stts_get_max_duration(stts);
  stts_move_init(stts);

  return stts;

fail:
  if (stts_ptr)
    stts_clean(stts);
  else
    stts_delete(stts);
  qtime_error_atom_read(QTIME_TYPE_STTS);
  return NULL;
}

stts_t*
stts_create(stts_t *stts_ptr)
{
  stts_t *stts;

  if (stts_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_STTS, stts_ptr->atom.type, NULL)
    stts = stts_ptr;
    stts_clean(stts);
  } else {
    if ((stts = stts_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }

  return stts;
}

int64_t
stts_calc_size(stts_t *stts)
{
  int64_t size;

  qtime_error_type_check_v(QTIME_TYPE_STTS, stts->atom.type, 0)
  if (stts->number_of_entries <= 0)
    return 0;
  size = (int64_t)(8 + STTS_PROP_SIZE) +
         (int64_t)stts->number_of_entries * (int64_t)(TIME_TO_SAMPLE_T_SIZE);
  if (size & SIZE64MASK)
    size += 8;
  stts->atom.size = size;

  return size;
}

int
stts_write_atom(qtime_io_t *qtio, stts_t *stts)
{
  int i;
  int table_num;
  time_to_sample_t *table;
  atom_head_t atom_head;

  qtime_error_type_check_v(QTIME_TYPE_STTS, stts->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  if (stts->number_of_entries <= 0)
    return QTIME_OK;

  atom_head.size = stts->atom.size;
  atom_head.type = stts->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_STTS);
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, &stts->version, 1);
  qtime_io_write(qtio, &stts->flags, 3);
  qtime_io_write32(qtio, &stts->number_of_entries);

  table_num = stts->number_of_entries;
  table = stts->table;

  for (i = 0; i < table_num; i++, table++) {
    qtime_io_write32(qtio, &table->sample_count);
    qtime_io_write32(qtio, &table->sample_duration);
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_STTS);
    return QTIME_ERROR_ATOM_WRITE;
  }

//  stts_set_constant_duration(stts, 0, 0);

  return QTIME_OK;
}

void
stts_dump(const char *parent_types, stts_t *stts)
{
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  uint32_t flags;

  if (!stts_valid(stts))
    return;

  qtime_type_to_str(stts->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  qtime_flags_get(stts->flags, &flags);

  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)stts->atom.size);
  fprintf(stdout, "%s: version      %d\n", types, stts->version);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
  fprintf(stdout, "%s: number of entries %d\n", types, stts->number_of_entries);

  {
    int i;
    for (i = 0; i < stts->number_of_entries; i++) {
      fprintf(stdout,"%s: table[%d] count = %u, duration = %u\n", types, i, stts->table[i].sample_count, stts->table[i].sample_duration);
    }
  }
}

uint32_t
stts_get_max_count(stts_t *stts)
{
  uint32_t count;
  int i;

  if (stts->max_count > 0)
    return stts->max_count;

  count = 0;
  for (i = 0; i < stts->number_of_entries; i++)
    count += stts->table[i].sample_count;
  stts->max_count = count;
  return count;
}

uint32_t
stts_get_max_duration(stts_t *stts)
{
  uint32_t duration;
  int i;

  if (stts->max_duration > 0)
    return stts->max_duration;

  duration = 0;
  for (i = 0; i < stts->number_of_entries; i++)
    duration += stts->table[i].sample_count * stts->table[i].sample_duration;
  stts->max_duration = duration;
  return duration;
}

int32_t
stts_move_init(stts_t *stts)
{
  stts->idx = 0;
  stts->st_count = 0;
  stts->st_duration = 0;
  stts->ed_count = stts->table[0].sample_count;
  stts->ed_duration =
               stts->table[0].sample_count * stts->table[0].sample_duration;
  return stts->idx;
}

uint32_t
stts_move_duration(stts_t *stts, uint32_t duration)
{
  int idx;
  uint32_t sd, ed, sc, ec;
  uint32_t count;
 
  if (stts->ed_count == 0)
    stts_move_init(stts);

  idx = stts->idx;
  sc = stts->st_count;
  sd = stts->st_duration;
  ec = stts->ed_count;
  ed = stts->ed_duration;

  if (duration >= ed) {
    if (idx < stts->number_of_entries-1) {
      for (idx++; idx < stts->number_of_entries; idx++) {
        sc = ec;
        sd = ed;
        ec = ec + stts->table[idx].sample_count;
        ed = ed + (stts->table[idx].sample_count *
	           stts->table[idx].sample_duration);
        if (sd <= duration && duration < ed)
	  break;
      }
      stts->idx = idx;
      stts->st_count = sc;
      stts->ed_count = ec;
      stts->st_duration = sd;
      stts->ed_duration = ed;
    }
  } else if (duration < sd) {
    if (idx > 0) {
      for (idx--; idx >= 0 ; idx--) {
        ec = sc;
        ed = sd;
        sc = sc - stts->table[idx].sample_count;
        sd = sd - (stts->table[idx].sample_count *
	           stts->table[idx].sample_duration);
        if (sd <= duration && duration < ed)
	  break;
      }
      if (idx < 0) {
	idx = 0;
        // stts_move_init(stts);
      }
      stts->idx = idx;
      stts->st_count = sc;
      stts->ed_count = ec;
      stts->st_duration = sd;
      stts->ed_duration = ed;
    }
  }

#ifndef NDEBUG
  if (duration < sd) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "stts_move_duration: duration = %u < sd %u.\n", duration, sd);
  }
#endif

  count = sc + ((duration - sd) / stts->table[idx].sample_duration);

  return count;
}

uint32_t
stts_move_count(stts_t *stts, uint32_t count)
{
  int32_t idx;
  uint32_t sd, ed, sc, ec;
  uint32_t duration;
 
  if (stts->ed_count == 0)
    stts_move_init(stts);

  idx = stts->idx;
  sc = stts->st_count;
  sd = stts->st_duration;
  ec = stts->ed_count;
  ed = stts->ed_duration;

  if (count >= ec) {
    if (idx < stts->number_of_entries-1) {
      for (idx++; idx < stts->number_of_entries; idx++) {
        sc = ec;
        sd = ed;
        ec = ec + stts->table[idx].sample_count;
        ed = ed + (stts->table[idx].sample_count *
	           stts->table[idx].sample_duration);
        if (sc <= count && count < ec)
	  break;
      }
      stts->idx = idx;
      stts->st_count = sc;
      stts->ed_count = ec;
      stts->st_duration = sd;
      stts->ed_duration = ed;
    }
  } else if (count < sc) {
    if (idx > 0) {
      for (idx--; idx >= 0 ; idx--) {
        ec = sc;
        ed = sd;
        sc = sc - stts->table[idx].sample_count;
        sd = sd - (stts->table[idx].sample_count *
	           stts->table[idx].sample_duration);
        if (sc <= count && count < ec)
	  break;
      }
      if (idx < 0) {
	idx = 0;
	// stts_move_init(stts);
      }
      stts->idx = idx;
      stts->st_count = sc;
      stts->ed_count = ec;
      stts->st_duration = sd;
      stts->ed_duration = ed;
    }
  }

#ifndef NDEBUG
  if (count < sc) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: invalid value, count %u, sc %u.\n",count,sc);
  }
#endif

  duration = sd + ((count - sc) * stts->table[idx].sample_duration);

  return duration;
}

uint32_t
stts_get_count(stts_t *stts, uint32_t duration)
{
  uint32_t count;
  if (duration >= stts->max_duration) {
#ifndef NDEBUG
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: requested duration is over the maximum duration in 'stts' atom, duration = %u, max = %u.\n", duration, stts->max_duration);
#endif
    return stts->max_count - 1;
  }
  count = stts_move_duration(stts, duration);
  return count;
}

uint32_t
stts_get_duration(stts_t *stts, uint32_t count)
{
  uint32_t duration;
  if (count >= stts->max_count) {
#ifndef NDEBUG
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: requested count is over the maximum count in 'stts' atom, count = %u, max = %u.\n", count, stts->max_count);
#endif
    return stts->max_duration - 1;
  }
  duration = stts_move_count(stts, count);
  return duration;
}

uint32_t
stts_get_sample_duration(stts_t *stts, uint32_t count)
{
  if (count >= stts->max_count) {
#ifndef NDEBUG
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: requested count is over the maximum count in 'stts' atom, count = %u, max = %u.\n", count, stts->max_count);
#endif
    return 0;
  }
  stts_move_count(stts, count);
  return stts->table[stts->idx].sample_duration;
}

int32_t
stts_set_constant_duration(stts_t *stts, uint32_t sample_count, uint32_t sample_duration)
{
  if (stts->table == NULL) {
    stts_add_sample_duration(stts, sample_count, sample_duration);
    stts->table[0].sample_count = 0;
  } 

  if (stts->number_of_entries > 1) {
    int table_max = 1;
    time_to_sample_t *table;
    uint32_t count;

//    fprintf(stderr, "stts_set_constant_duration: number_of_entries > 1, existing durations are discarded.\n");
    fprintf(stderr, "QTIME_ERROR: stts_set_constant_duration: number_of_entries %d, existing durations are discarded.\n", stts->number_of_entries);
    count = stts_get_max_count(stts);
    table = (time_to_sample_t*)qtime_realloc(stts->table,
                                     sizeof(time_to_sample_t) * table_max);
    if (!table)
      return QTIME_ERROR_MEMORY;
    table->sample_count = count;
    table->sample_duration = sample_duration;
    stts->table_max = table_max;
    stts->table = table;
    stts->number_of_entries = 1;
  }

  stts->table[0].sample_duration = sample_duration;
  if (sample_count) {
    stts->table[0].sample_count = sample_count;
  }
  stts->max_duration = stts->table[0].sample_count * sample_duration;
  stts->max_count = stts->table[0].sample_count;

  return stts->number_of_entries;
}

uint32_t
stts_add_sample_duration(stts_t *stts, uint32_t sample_count, uint32_t sample_duration)
{
#ifndef NDEBUG
  if (stts->number_of_entries > (MAX_INT32-(4096*2))) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: number_of_enries is reached max in 'stts' atom, entries = %d.\n", stts->number_of_entries);
  }
#endif

  if (stts->table == NULL) {
    int table_max = 1;
    time_to_sample_t *table;
    table = (time_to_sample_t*)qtime_realloc(stts->table,
                                     sizeof(time_to_sample_t) * table_max);
    if (!table)
      return QTIME_ERROR_MEMORY;
    table->sample_count = sample_count;
    table->sample_duration = sample_duration;
    stts->table_max = table_max;
    stts->table = table;
    stts->number_of_entries = 1;
    stts->max_duration = sample_count * sample_duration;
    stts->max_count = sample_count;
    return stts->table[stts->number_of_entries-1].sample_count;
  } 

  if (sample_duration==stts->table[stts->number_of_entries-1].sample_duration) {
    stts->table[stts->number_of_entries-1].sample_count += sample_count;
  } else {
    if ((stts->number_of_entries+1) > stts->table_max) {
      int table_max = stts->table_max + 4096;
      time_to_sample_t *table;
      table = (time_to_sample_t*)qtime_realloc(stts->table,
          sizeof(time_to_sample_t) * table_max);
      if (!table)
        return QTIME_ERROR_MEMORY;
      stts->table_max = table_max;
      stts->table = table;
    }
    stts->table[stts->number_of_entries].sample_count = sample_count;
    stts->table[stts->number_of_entries].sample_duration = sample_duration;
    stts->number_of_entries++;
  }

  stts->max_duration += sample_count * sample_duration;
  stts->max_count += sample_count;

  stts->ed_count = 0;

  return stts->number_of_entries;
}

int
stts_get_sample_info(stts_t *stts, qtime_sample_info_t *spinfo)
{
  uint32_t duration;
  uint32_t end_count;

  if (stts->max_count == 0) {
    stts_get_max_count(stts);
  }

  if (spinfo->count >= stts->max_count) {
#ifndef NDEBUG
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: count over the maximum count in 'stts' atom, count %u, max %u.\n", spinfo->count, stts->max_count);
#endif
    spinfo->duration = stts->max_duration;
    spinfo->flag |= QTIME_SAMPLE_INFO_FLAG_OVER_MAX;
    return QTIME_SAMPLE_INFO_FLAG_OVER_MAX;
  }

  duration = stts_move_count(stts, spinfo->count);
  spinfo->duration = duration;
  end_count = spinfo->count + spinfo->samples;

  if (end_count >= stts->max_count)
    spinfo->sample_duration = stts->max_duration - duration;
  else
    spinfo->sample_duration = stts_move_count(stts, end_count) - duration;

  return QTIME_OK;
}

int
stts_table_resize(stts_t *stts, int32_t entry_num)
{
  int table_max;
  time_to_sample_t *table;

  if (entry_num <= stts->table_max)
    return QTIME_OK;
  if (!stts->table)
    table_max = 2;
  else
    table_max = stts->table_max + 4096;
  table = (time_to_sample_t*)qtime_realloc(stts->table,
                                     sizeof(time_to_sample_t) * table_max);
  if (!table)
    return QTIME_ERROR_MEMORY;
  stts->table_max = table_max;
  stts->table = table;
  return QTIME_OK;
}

int
stts_add_sample_info(stts_t *stts, qtime_sample_info_t *spinfo)
{
  int idx;
#ifndef NDEBUG
  if (stts->number_of_entries > (MAX_INT32-(4096*2))) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: number_of_enries is reached max in 'stts' atom, entries = %d.\n", stts->number_of_entries);
  }
#endif

  if (stts->table == NULL) {
    if (stts_table_resize(stts, 1) < 0)
      return QTIME_ERROR;
    idx = 0;
    stts->table[idx].sample_count = spinfo->samples;
    stts->table[idx].sample_duration = spinfo->sample_duration;
    stts->number_of_entries = 1;
    stts->max_duration += spinfo->samples * spinfo->sample_duration;
    stts->max_count += spinfo->samples;
    return stts->number_of_entries;
  } 

  idx = stts->number_of_entries-1;
  if (spinfo->sample_duration == stts->table[idx].sample_duration) {
    stts->table[idx].sample_count += spinfo->samples;
  } else {
    if (stts_table_resize(stts, stts->number_of_entries+1) < 0)
      return QTIME_ERROR;
    stts->table[idx].sample_count = spinfo->samples;
    stts->table[idx].sample_duration = spinfo->sample_duration;
    stts->number_of_entries++;
  }
  spinfo->duration = stts->max_duration;
  stts->max_duration += spinfo->samples * spinfo->sample_duration;
  stts->max_count += spinfo->samples;

  stts->ed_count = 0;

  return stts->number_of_entries;
}

