/* stsz.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_sample_info.h"
#include "atom.h"
#include "stsz.h"

void
stsz_init(stsz_t *stsz)
{
  atom_init(&stsz->atom);
  stsz->atom.type = QTIME_TYPE_STSZ;
  stsz->version = 0;
  qtime_flags_set(stsz->flags, 0);
  stsz->sample_size = 0;
  stsz->number_of_entries = 0;
  stsz->table_max = 0;
  stsz->table = NULL;

  stsz->sample_size_count = 0;
  stsz->total_size = 0;
}

void
stsz_clean(stsz_t *stsz)
{
  qtime_error_type_check(QTIME_TYPE_STSZ, stsz->atom.type)
  if (stsz->table)
    qtime_free(stsz->table);
  stsz_init(stsz);
}

stsz_t*
stsz_new(void)
{
  stsz_t *stsz;
  stsz = (stsz_t*)qtime_malloc(sizeof(stsz_t));
  if (!stsz) return NULL;
  stsz_init(stsz);
  return stsz;
}

void
stsz_delete(stsz_t *stsz)
{
  qtime_error_type_check(QTIME_TYPE_STSZ, stsz->atom.type)
  if (stsz->table)
    qtime_free(stsz->table);
  qtime_free(stsz);
}

int
stsz_valid(stsz_t *stsz)
{
  if (!stsz || (stsz->number_of_entries <= 0 && stsz->sample_size <= 0))
    return 0;
  return 1;
}

stsz_t*
stsz_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, stsz_t *stsz_ptr)
{
  int table_byte_size;
  int i, table_num;
  int32_t *table;
  stsz_t *stsz;

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_STSZ, atom_head->type, NULL)
  if (stsz_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_STSZ, stsz_ptr->atom.type, NULL)
    stsz = stsz_ptr;
    stsz_clean(stsz);
  } else {
    if ((stsz = stsz_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  stsz->atom.size = atom_head->size;
  stsz->atom.type = atom_head->type;
  stsz->atom.parent = atom_head->parent;

  qtime_io_read(qtio, &stsz->version, 1);
  qtime_io_read(qtio, &stsz->flags, 3);
  qtime_io_read32(qtio, &stsz->sample_size);
  qtime_io_read32(qtio, &stsz->number_of_entries);

  if (stsz->sample_size != 0) {
#ifndef NDEBUG
    if (stsz->sample_size < 0) {
      qtime_error_debug_info(QTIME_ERROR);
      fprintf(stderr, "QTIME_ERROR: stsz_read_atom: error, sample_size value is negative, %d.\n", stsz->sample_size);
    }
    if (stsz->number_of_entries) {
      qtime_error_debug_info(QTIME_ERROR);
      fprintf(stderr, "QTIME_ERROR: stsz_read_atom: error, sample_size %d, number_of_entries %d.\n", stsz->sample_size, stsz->number_of_entries);
    }
#endif
    stsz->number_of_entries = 0;
    stsz->table = NULL;
    stsz->table_max = 0;
    return stsz;
  }

#ifndef NDEBUG
  if (stsz->number_of_entries < 0) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: stsz_read_atom: error, number_of_entries value is negative, %d.\n", stsz->number_of_entries);
  }
#endif

  table_num = stsz->number_of_entries;
  table_byte_size = table_num * sizeof(int32_t);
  table = (int32_t*)qtime_malloc(table_byte_size);
  if (!table) {
    qtime_error_debug_info(QTIME_ERROR_MEMORY);
    atom_head->error_code = QTIME_ERROR_MEMORY;
    goto fail;
  }
  stsz->table = table;
  stsz->table_max = table_num;

  for (i = 0; i < table_num; i++, table++) {
    qtime_io_read32(qtio, table);
  }

  return stsz;

fail:
  if (stsz_ptr)
    stsz_clean(stsz);
  else
    stsz_delete(stsz);
  qtime_error_atom_read(QTIME_TYPE_STSZ);
  return NULL;
}

stsz_t*
stsz_create(stsz_t *stsz_ptr)
{
  stsz_t *stsz;

  if (stsz_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_STSZ, stsz_ptr->atom.type, NULL)
    stsz = stsz_ptr;
    stsz_clean(stsz);
  } else {
    if ((stsz = stsz_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return stsz;
}

int64_t
stsz_calc_size(stsz_t *stsz)
{
  int64_t size;

  qtime_error_type_check_v(QTIME_TYPE_STSZ, stsz->atom.type, 0)
  if (stsz->number_of_entries <= 0 && stsz->sample_size <= 0)
    return 0;
  size = (int64_t)(8 + STSZ_PROP_SIZE) +
          (int64_t)stsz->number_of_entries * (int64_t)sizeof(uint32_t);
  if (size & SIZE64MASK)
    size += 8;
  stsz->atom.size = size;
  return size;
}

int
stsz_write_atom(qtime_io_t *qtio, stsz_t *stsz)
{
  int i;
  int table_num;
  uint32_t *table;
  atom_head_t atom_head;

  qtime_error_type_check_v(QTIME_TYPE_STSZ, stsz->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  if (stsz->number_of_entries <= 0 && stsz->sample_size <= 0)
    return QTIME_OK;

  atom_head.size = stsz->atom.size;
  atom_head.type = stsz->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_STSZ);
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, &stsz->version, 1);
  qtime_io_write(qtio, &stsz->flags, 3);
  qtime_io_write32(qtio, &stsz->sample_size);
  qtime_io_write32(qtio, &stsz->number_of_entries);

  table_num = stsz->number_of_entries;
  table = stsz->table;

  for (i = 0; i < table_num; i++, table++) {
    qtime_io_write32(qtio, table);
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_STSZ);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
stsz_dump(const char *parent_types, stsz_t *stsz)
{
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  uint32_t flags;

  if (!stsz_valid(stsz))
    return;

  qtime_type_to_str(stsz->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  qtime_flags_get(stsz->flags, &flags);

  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)stsz->atom.size);
  fprintf(stdout, "%s: version      %d\n", types, stsz->version);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
  fprintf(stdout, "%s: sample size  %d\n", types, stsz->sample_size);
  fprintf(stdout, "%s: number of entries %d\n", types, stsz->number_of_entries);

#if 0
  {
    int i;
    for (i = 0; i < stsz->number_of_entries; i++)
      fprintf(stdout, "%s: table[%d] offset = %d\n", types,i,stsz->table[i]);
  }
#endif
}

uint32_t
stsz_get_sample_size(stsz_t *stsz)
{
  return stsz->sample_size;
}

int64_t
stsz_get_total_size(stsz_t *stsz)
{
  int64_t size = 0;
  int i;

  if (stsz->total_size != 0)
    return stsz->total_size;
  if (stsz->number_of_entries <= 0)
    return size;
  for (i = 0; i < stsz->number_of_entries; i++)
    size += stsz->table[i];
  stsz->total_size = size;
  return size;
}

void
stsz_set_total_size(stsz_t *stsz, int64_t total_size)
{
  stsz->total_size = total_size;
}

int32_t
stsz_add_size(stsz_t *stsz, int32_t size)
{
  int32_t *table;

#ifndef NDEBUG
  if (stsz->number_of_entries > (MAX_INT32-(4096*2)) ||
      stsz->sample_size_count > (MAX_INT32-(4096*2))) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: number_of_enries is reached max in 'stsz' atom, entries = %d, sample_size_count = %u.\n", stsz->number_of_entries, stsz->sample_size_count);
  }
#endif

  stsz->total_size += size;

  if (stsz->number_of_entries == 0) {
    if (stsz->sample_size == 0) {
      stsz->sample_size = size;
      stsz->sample_size_count++;
      return stsz->sample_size_count;
    } else if (size == stsz->sample_size) {
      stsz->sample_size_count++;
      return stsz->sample_size_count;
    }
  }

  if (stsz->sample_size_count) {
    uint32_t i, table_max = stsz->sample_size_count + 1;
    table = (int32_t*)qtime_realloc(stsz->table,
        sizeof(int32_t) * table_max);
    if (!table)
      return QTIME_ERROR_MEMORY;
    for (i = 0; i < stsz->sample_size_count; i++)
      table[i] = stsz->sample_size;
    stsz->table_max = table_max;
    stsz->table = table;
    stsz->number_of_entries = stsz->sample_size_count;
    stsz->sample_size = 0;
    stsz->sample_size_count = 0;
  }

  if ((stsz->number_of_entries+1) > stsz->table_max) {
    int table_max = stsz->table_max + 4096;
    table = (uint32_t*)qtime_realloc(stsz->table,
        sizeof(uint32_t) * table_max);
    if (!table)
      return QTIME_ERROR_MEMORY;
    stsz->table_max = table_max;
    stsz->table = table;
  }
  stsz->table[stsz->number_of_entries] = size;
  stsz->number_of_entries++;

  return stsz->number_of_entries;
}

int32_t
stsz_add_sizes(stsz_t *stsz, int32_t size, uint32_t count)
{
  uint32_t i;

#ifndef NDEBUG
  if ((uint32_t)stsz->number_of_entries > (MAX_INT32-(4096*2)-count) ||
      stsz->sample_size_count > (MAX_INT32-(4096*2)-count)) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: number_of_enries is reached max in 'stsz' atom, entries = %d, sample_size_count = %u.\n", stsz->number_of_entries, stsz->sample_size_count);
  }
#endif

  stsz->total_size += size * count;

  if (stsz->number_of_entries == 0) {
    if (stsz->sample_size == 0) {
      stsz->sample_size = size;
      stsz->sample_size_count = count;
      return stsz->sample_size_count;
    } else if (size == stsz->sample_size) {
      stsz->sample_size_count += count;
      return stsz->sample_size_count;
    }
  }

  for (i = 0; i < count; i++) {
    stsz_add_size(stsz, size);
  }

  return stsz->number_of_entries;
}

#if 0
int32_t
stsz_set_size(stsz_t *stsz, int32_t number, uint32_t size)
{
  if (number < 0 || number >= (int32_t)stsz->number_of_entries) {
    fprintf(stderr, "stsz_set: invalid entry number, %d, %d\n", number, stsz->number_of_entries);
    return QTIME_ERROR;
  }
  stsz->table[number] = size;
  return number;
}
#endif

int32_t
stsz_get_size(stsz_t *stsz, uint32_t count)
{
  if (stsz->number_of_entries == 0)
    return stsz->sample_size;

  if (count >= (uint32_t)(stsz->number_of_entries)) {
#ifndef NDEBUG
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: requested count is over the maximum entry in 'stsz' atom, count = %u, entries = %d.\n", count, stsz->number_of_entries);
#endif
    return QTIME_ERROR;
  }
  return stsz->table[count];
}

int64_t
stsz_get_sizes(stsz_t *stsz, uint32_t start_count, uint32_t samples)
{
  int64_t size;
  uint32_t end_count;

  if (stsz->sample_size) {
    size = stsz->sample_size;
    size *= samples;
    return size;
  }

  size = 0;
  end_count = start_count + samples;
  for (; start_count < end_count; start_count++)
    size += stsz->table[start_count];

  return size;
}

int
stsz_get_sample_info(stsz_t *stsz, qtime_sample_info_t *spinfo)
{
  uint32_t st_count, ed_count;
  int64_t size, offset;
  uint32_t sample_size;

  if (spinfo->bytes_per_sample)
    sample_size = spinfo->bytes_per_sample;
  else if (stsz->sample_size)
    sample_size = stsz->sample_size;
  else
    sample_size = 0;
    
  if (sample_size) {
    offset = spinfo->last_data_offset;
    if (spinfo->count > spinfo->last_count)
      offset += (spinfo->count - spinfo->last_count) * sample_size;
    else if (spinfo->count < spinfo->last_count)
      offset -= (spinfo->last_count - spinfo->count) * sample_size;
    sample_size *= spinfo->samples;
    spinfo->sample_size = sample_size;
    spinfo->data_start_offset = offset;
    offset += sample_size;
    spinfo->data_end_offset = offset;
    spinfo->last_count = spinfo->count + spinfo->samples;
    spinfo->last_data_offset = offset;
    return QTIME_OK;
  }

  size = 0;
  offset = spinfo->last_data_offset;
  if (spinfo->count > spinfo->last_count) {
    st_count = spinfo->last_count;
    ed_count = spinfo->count;
    for (; st_count < ed_count; st_count++)
      offset += stsz->table[st_count];
  } else if (spinfo->count < spinfo->last_count) {
    st_count = spinfo->count;
    ed_count = spinfo->last_count;
    for (; st_count < ed_count; st_count++)
      offset -= stsz->table[st_count];
  }
  spinfo->data_start_offset = offset;
  st_count = spinfo->count;
  ed_count = st_count + spinfo->samples;
  for (; st_count < ed_count; st_count++)
    size += stsz->table[st_count];
  spinfo->sample_size = size;
  spinfo->last_count = spinfo->count + spinfo->samples;
  offset += size;
  spinfo->data_end_offset = offset;
  spinfo->last_data_offset = offset;

  return QTIME_OK;
}

int
stsz_add_sample_info(stsz_t *stsz, qtime_sample_info_t *spinfo)
{
  if (spinfo->bytes_per_sample)
    stsz_add_sizes(stsz, 1, spinfo->samples);
  else
    stsz_add_size(stsz, spinfo->sample_size);
  return QTIME_OK;
}

