/* mdia.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_media.h"
#include "qtime_media_type.h"
#include "atom.h"
#include "mdia.h"
#include "mdhd.h"
#include "hdlr.h"
#include "minf.h"
#include "udta.h"
#include "stts.h"



#ifndef NDEBUG

int
mdia_check_media_type(mdia_t *mdia)
{
  hdlr_t *hdlr;
  minf_t *minf;
  uint32_t hdlr_type;
  uint32_t minf_type;
  int ret = QTIME_OK;

  hdlr = (hdlr_t*)atom_find_child((atom_t*)mdia, QTIME_TYPE_HDLR);
  hdlr_type = hdlr_get_media_type(hdlr);
  if (hdlr_type == QTIME_MEDIA_TYPE_UNKNOWN) {
    qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
    fprintf(stderr, "QTIME_ERROR_INVALID_MEDIA: 'hdlr' atom's media_type is unknown in 'mdia' atom.\n");
    ret = QTIME_ERROR_INVALID_MEDIA;
  }

  minf = (minf_t*)atom_find_child((atom_t*)mdia, QTIME_TYPE_MINF);
  minf_type = minf_get_media_type(minf);
  if (minf_type == QTIME_MEDIA_TYPE_UNKNOWN) {
    qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
    fprintf(stderr, "QTIME_ERROR_INVALID_MEDIA: 'minf' atom's media_type is unknown in 'mdia' atom.\n");
    ret = QTIME_ERROR_INVALID_MEDIA;
  }

  if (hdlr_type != minf_type) {
    qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
    fprintf(stderr, "QTIME_ERROR_INVALID_MEDIA: 'hdlr' atom's media_type and 'minf' atom's media_type doesn't suit in 'mdia' atom.\n");
    ret = QTIME_ERROR_INVALID_MEDIA;
  }

  return ret;
}

#endif

void
mdia_init(mdia_t *mdia)
{
  atom_init(&mdia->atom);
  mdia->atom.type = QTIME_TYPE_MDIA;
  mdhd_init(&mdia->mdhd);
  hdlr_init(&mdia->hdlr);
  minf_init(&mdia->minf);
  mdia->udta = NULL;
  mdia->media_type = QTIME_MEDIA_TYPE_UNKNOWN;
}

void
mdia_clean(mdia_t *mdia)
{
  int i;
  atom_t *atom = &mdia->atom;

  qtime_error_type_check(QTIME_TYPE_MDIA, mdia->atom.type)

  for (i = 0; i < atom->number_of_childs; i++) {
    if (atom->childs[i] == (atom_t*)&mdia->mdhd) {
      mdhd_clean((mdhd_t*)atom->childs[i]);
      continue;
    } else if (atom->childs[i] == (atom_t*)&mdia->hdlr) {
      hdlr_clean((hdlr_t*)atom->childs[i]);
      continue;
    } else if (atom->childs[i] == (atom_t*)&mdia->minf) {
      minf_clean((minf_t*)atom->childs[i]);
      continue;
    }

    switch (atom->childs[i]->type) {
      case QTIME_TYPE_MDHD:
	mdhd_delete((mdhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_HDLR:
	hdlr_delete((hdlr_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_MINF:
	minf_delete((minf_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_UDTA:
	udta_delete((udta_t*)atom->childs[i]);
	break;
      default:
        qtime_error_debug_info(QTIME_ERROR_UNKNOWN_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_MDIA, atom->childs[i]->type);
	break;
    }
  }
  mdia->udta = NULL;
  atom_clean(&mdia->atom);
  mdia->atom.type = QTIME_TYPE_MDIA;
}

mdia_t*
mdia_new(void)
{
  mdia_t *mdia;
  mdia = (mdia_t*)qtime_malloc(sizeof(mdia_t));
  if (!mdia) return NULL;
  mdia_init(mdia);
  return mdia;
}

void
mdia_delete(mdia_t *mdia)
{
  qtime_error_type_check(QTIME_TYPE_MDIA, mdia->atom.type)
  mdia_clean(mdia);
  qtime_free(mdia);
}

mdia_t*
mdia_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, mdia_t *mdia_ptr)
{
  atom_head_t subatom;
  atom_t *atom;
  int mdhd_index, hdlr_index, minf_index, udta_index;
  mdia_t *mdia;
  mdhd_t *mdhd;
  hdlr_t *hdlr;
  minf_t *minf;
  udta_t *udta;
  uint32_t media_type;

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_MDIA, atom_head->type, NULL)

  if (mdia_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_MDIA, mdia_ptr->atom.type, NULL)
    mdia = mdia_ptr;
    mdia_clean(mdia);
  } else {
    if ((mdia = mdia_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  mdia->atom.size = atom_head->size;
  mdia->atom.type = QTIME_TYPE_MDIA;
  mdia->atom.parent = atom_head->parent;

  atom_head_init(&subatom);
  subatom.parent = (atom_t*)mdia;
  subatom.media_type = QTIME_MEDIA_TYPE_UNKNOWN;

  mdhd = &mdia->mdhd;
  hdlr = &mdia->hdlr;
  minf = &mdia->minf;
  udta = NULL;
  mdhd_index = hdlr_index = minf_index = udta_index = 0;
  media_type = QTIME_MEDIA_TYPE_UNKNOWN;

  while (atom_head->end_offset > subatom.end_offset) {
    if (atom_read_header(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }

    atom = NULL;
    switch (subatom.type) {
      case QTIME_TYPE_MDHD:
	if ((mdhd = mdhd_read_atom(qtio, &subatom, mdhd)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)mdhd;
	atom->index = ++mdhd_index;
	mdhd = NULL;
	break;
      case QTIME_TYPE_HDLR:
	if ((hdlr = hdlr_read_atom(qtio, &subatom, hdlr)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)hdlr;
	atom->index = ++hdlr_index;
	hdlr = NULL;
	break;
      case QTIME_TYPE_MINF:
	if ((minf = minf_read_atom(qtio, &subatom, minf)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)minf;
	atom->index = ++minf_index;
	minf = NULL;
	break;
      case QTIME_TYPE_UDTA:
	if ((udta = udta_read_atom(qtio, &subatom, udta)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)udta;
	atom->index = ++udta_index;
	udta = NULL;
	break;
      default:
        qtime_error_unknown_atom(QTIME_TYPE_MDIA, subatom.type, (int32_t)subatom.size);
	break;
    }

    if (atom) {
      atom->parent = (atom_t*)mdia;
      if ((atom_head->error_code = atom_add_child((atom_t*)mdia, atom)) < 0) {
	qtime_error_debug_info(atom_head->error_code);
	goto fail;
      }
    }

    if (media_type == QTIME_MEDIA_TYPE_UNKNOWN) {
      media_type = subatom.media_type;
    } else if (media_type != subatom.media_type) {
      qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
      qtime_error_invalid_media(QTIME_TYPE_MDIA,media_type,subatom.media_type);
    }

    if (atom_read_footer(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }
  }

  mdia->media_type = media_type;
  atom_head->media_type = media_type;

  mdia->udta = (udta_t*)atom_find((atom_t*)mdia, QTIME_TYPE_UDTA, ATOM_FIND_CHILDS_ONLY);

#ifndef NDEBUG
  if (mdia_check_media_type(mdia) != QTIME_OK) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: 'mdia' atom's media_type is invalid.\n");
  }
#endif

  return mdia;

fail:
  if (mdia_ptr)
    mdia_clean(mdia);
  else
    mdia_delete(mdia);
  qtime_error_atom_read(QTIME_TYPE_MDIA);
  return NULL;
}

mdia_t*
mdia_create(mdia_t *mdia_ptr)
{
  mdia_t *mdia;
  int error_code;

  if (mdia_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_MDIA, mdia_ptr->atom.type, NULL)
    mdia = mdia_ptr;
    mdia_clean(mdia);
  } else {
    if ((mdia = mdia_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }

  mdhd_create(&mdia->mdhd);
  hdlr_create(&mdia->hdlr);
  minf_create(&mdia->minf);

  if ((error_code = atom_add_child((atom_t*)mdia, (atom_t*)&mdia->mdhd)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }
  if ((error_code = atom_add_child((atom_t*)mdia, (atom_t*)&mdia->hdlr)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }
  if ((error_code = atom_add_child((atom_t*)mdia, (atom_t*)&mdia->minf)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }

  return mdia;

fail:
  if (mdia_ptr)
    mdia_clean(mdia);
  else
    mdia_delete(mdia);
  qtime_error_debug_info(error_code);
  return NULL;
}

int64_t
mdia_calc_size(mdia_t *mdia)
{
  int64_t size = 0;
  int i;
  atom_t *atom = &mdia->atom;

  qtime_error_type_check_v(QTIME_TYPE_MDIA, mdia->atom.type, 0)

  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_MDHD:
	size += mdhd_calc_size((mdhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_HDLR:
	size += hdlr_calc_size((hdlr_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_MINF:
	size += minf_calc_size((minf_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_UDTA:
	size += udta_calc_size((udta_t*)atom->childs[i]);
	break;
      default:
	qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_MDIA, atom->childs[i]->type);
	break;
    }
  }

  if (size <= 8) {
    mdia->atom.size = 0;
    return 0;
  }

  if (size & SIZE64MASK)
    size += 8 + 8;
  else
    size += 8;
  mdia->atom.size = size;

  return size;
}

int
mdia_write_atom_pre(mdia_t *mdia)
{
  uint32_t media_duration;
  media_duration = mdia_get_media_duration(mdia);
  return mdia_set_media_duration(mdia, media_duration);
}

int
mdia_write_atom(qtime_io_t *qtio, mdia_t *mdia)
{
  atom_head_t atom_head;
  int i;
  atom_t *atom = &mdia->atom;

  qtime_error_type_check_v(QTIME_TYPE_MDIA, mdia->atom.type, QTIME_ERROR_ILLEGAL_ATOM)
#ifndef NDEBUG
  if (mdia_check_media_type(mdia) != QTIME_OK) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: 'mdia' atom's media_type is invalid.\n");
  }
#endif

  mdia_write_atom_pre(mdia);

  atom_head_init(&atom_head);
  atom_head.size = mdia->atom.size;
  atom_head.type = mdia->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_MDIA);
    return QTIME_ERROR_ATOM_WRITE;
  }

  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_MDHD:
	mdhd_write_atom(qtio, (mdhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_HDLR:
	hdlr_write_atom(qtio, (hdlr_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_MINF:
	minf_write_atom(qtio, (minf_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_UDTA:
	udta_write_atom(qtio, (udta_t*)atom->childs[i]);
	break;
      default:
	qtime_error_debug_info(QTIME_ERROR_UNKNOWN_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_MDIA, atom->childs[i]->type);
	break;
    }
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_MDIA);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
mdia_dump(const char *parent_types, mdia_t *mdia)
{
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  int i;
  atom_t *atom = &mdia->atom;

  qtime_type_to_str(mdia->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);

  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)mdia->atom.size);
  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_MDHD:
        mdhd_dump(types, (mdhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_HDLR:
        hdlr_dump(types, (hdlr_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_MINF:
        minf_dump(types, (minf_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_UDTA:
        udta_dump(types, (udta_t*)atom->childs[i]);
	break;
      default:
	qtime_error_debug_info(QTIME_ERROR_UNKNOWN_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_MDIA, atom->childs[i]->type);
	break;
    }
  }
}

uint32_t
mdia_get_media_time_scale(mdia_t *mdia)
{
  mdhd_t *mdhd;

  qtime_error_type_check_v(QTIME_TYPE_MDIA, mdia->atom.type, QTIME_ERROR_ILLEGAL_ATOM)
  mdhd = (mdhd_t*)atom_find_child((atom_t*)mdia, QTIME_TYPE_MDHD);
  return mdhd_get_time_scale(mdhd);
}

uint32_t
mdia_get_media_duration(mdia_t *mdia)
{
  stts_t *stts;

  qtime_error_type_check_v(QTIME_TYPE_MDIA, mdia->atom.type, QTIME_ERROR_ILLEGAL_ATOM)
  stts = (stts_t*)atom_find_recursive((atom_t*)mdia, QTIME_TYPE_STTS);
  return stts_get_max_duration(stts);
}

int
mdia_set_media_duration(mdia_t *mdia, uint32_t duration)
{
  mdhd_t *mdhd;

  qtime_error_type_check_v(QTIME_TYPE_MDIA, mdia->atom.type, QTIME_ERROR_ILLEGAL_ATOM)
  mdhd = (mdhd_t*)atom_find_child((atom_t*)mdia, QTIME_TYPE_MDHD);
  mdhd_set_duration(mdhd, duration);
  return QTIME_OK;
}

uint32_t
mdia_get_media_type(mdia_t *mdia)
{
  hdlr_t *hdlr;
  minf_t *minf;
  uint32_t media_type;

  hdlr = (hdlr_t*)atom_find_child((atom_t*)mdia, QTIME_TYPE_HDLR);
  media_type = hdlr_get_media_type(hdlr);
  if (media_type != QTIME_MEDIA_TYPE_UNKNOWN)
    return media_type;
  minf = (minf_t*)atom_find_child((atom_t*)mdia, QTIME_TYPE_MINF);
  media_type = minf_get_media_type(minf);
  return media_type;
}

int
mdia_set_media_type(mdia_t *mdia, uint32_t media_type)
{
  if (media_type == QTIME_MEDIA_TYPE_UNKNOWN) {
    fprintf(stderr, "mdia_set_media_type: invalid media_type.\n");
    return QTIME_ERROR;
  }

  if (mdia->media_type != QTIME_MEDIA_TYPE_UNKNOWN &&
      mdia->media_type != media_type) {
    fprintf(stderr, "mdia_set_media_type: media_type doesn't suit.\n");
    return QTIME_ERROR;
  }
  mdia->media_type = media_type;

  if (hdlr_set_media_handler(&mdia->hdlr, media_type) != QTIME_OK)
    return QTIME_ERROR;

  if (minf_set_media_type(&mdia->minf, media_type) != QTIME_OK)
    return QTIME_ERROR;

  return QTIME_OK;
}

