/* aenc.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <inttypes.h>
#include <pthread.h>
#include <errno.h>

#include "acodec.h"
#include "aenc.h"
#include "state.h"
#include "aframe_mgr.h"

static AENC_CONFIG aenc_conf;

static int aenc_initialized = 0;
static int aenc_thread_running = 0;

#ifdef DEBUG
#if 0
static void
print_flag(uint32_t flag)
{
  if (flag & START_AFRAME)
    printf("aenc_thread: START_AFRAME\n");
  if (flag & END_AFRAME)
    printf("aenc_thread: END_AFRAME\n");
  if (flag & DROP_AFRAME)
    printf("aenc_thread: DROP_AFRAME\n");
  if (flag & QUIT_AFRAME)
    printf("aenc_thread: QUIT_AFRAME\n");
  if (flag & CHANGE_AFRAME)
    printf("aenc_thread: CHANGE_AFRAME\n");
}
#endif
#endif /* DEBUG */

static inline int
get_state(AENC_CONFIG *conf)
{
  int state;
  pthread_mutex_lock(&conf->mutex);
  state = conf->state;
  pthread_mutex_unlock(&conf->mutex);
  return state;
}

static inline void
state_change_done(AENC_CONFIG *conf)
{
  pthread_mutex_lock(&conf->mutex);
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
}

static inline void
set_state(AENC_CONFIG *conf, int state)
{
  pthread_mutex_lock(&conf->mutex);
  conf->state &= ~(STATE_START|STATE_PAUSE|STATE_STOP|STATE_QUIT);
  conf->state |= state;
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
}

static int
put_flush_frame(AENC_CONFIG *conf)
{
  AFRAME *dafrm;
  int stream_length;
  int ret = AENC_OK;

  do {
    while((dafrm = aframe_enc_dst_frame_get()) == NULL) {
      pthread_mutex_lock(&conf->mutex);
      if (!conf->wait_cb_called && !(conf->state & STATE_ERROR)) {
        conf->state |= STATE_THREAD_WAIT;
        pthread_cond_wait(&conf->thread_wait_cond, &conf->mutex);
      }
      conf->wait_cb_called = 0;
      if (conf->state & STATE_ERROR) {
        pthread_mutex_unlock(&conf->mutex);
        return AENC_FAIL;
      }
      pthread_mutex_unlock(&conf->mutex);
    }

    stream_length = acodec_encode_flush(&dafrm->sample_length, dafrm->stream, dafrm->stream_buf_size);
    if (stream_length < 0) {
      fprintf(stderr, "put_flush_frame: acodec_encode_flush() error %d.\n", stream_length);
#ifdef DEBUG
      exit(1);
#else
      ret = AENC_FAIL;
      stream_length = 0;
#endif /* DEBUG */
    }
#ifdef DEBUG
    if (stream_length == 0) {
      fprintf(stderr, "put_flush_frame: acodec_encode_flush() stream_length %d.\n", stream_length);
    }
#endif /* DEBUG */
    dafrm->stream_length = stream_length ;
//    dafrm->sample_length = 0;
#if 0
    if (acodec_encode_reset() != ACODEC_OK) {
      fprintf(stderr, "put_flush_frame: acodec_encode_reset() error.\n");
      pthread_mutex_lock(&conf->mutex);
      conf->state = STATE_ERROR;
      pthread_mutex_unlock(&conf->mutex);
      ret = AENC_FAIL;
    } else {
      ret = AENC_OK;
    }
#endif
    aframe_enc_dst_frame_update(dafrm);
  } while (stream_length > 0);
  return ret;
}

static int
put_enc_frame(AENC_CONFIG *conf, AFRAME *safrm)
{
  AFRAME *dafrm;
  int stream_length;
  int ret;

  if (safrm->sample_length <= 0) {
    fprintf(stderr, "put_enc_frame: safrm sample_length invalid %d.\n",safrm->sample_length);
#ifdef DEBUG
    exit(1);
#else
    return AENC_FAIL;
#endif /* DEBUG */
  }

  while((dafrm = aframe_enc_dst_frame_get()) == NULL) {
    pthread_mutex_lock(&conf->mutex);
    if (!conf->wait_cb_called && !(conf->state & STATE_ERROR)) {
      conf->state |= STATE_THREAD_WAIT;
      pthread_cond_wait(&conf->thread_wait_cond, &conf->mutex);
    }
    conf->wait_cb_called = 0;
    if (conf->state & STATE_ERROR) {
      pthread_mutex_unlock(&conf->mutex);
      return AENC_FAIL;
    }
    pthread_mutex_unlock(&conf->mutex);
  }

  stream_length = acodec_encode((void**)safrm->sample, &safrm->sample_length,
	     dafrm->stream, dafrm->stream_buf_size);
  if (stream_length < 0) {
    fprintf(stderr, "put_enc_frame: acodec_encode() error %d.\n",stream_length);
#ifdef DEBUG
    exit(1);
#else
    stream_length = 0;
    ret = AENC_FAIL;
#endif /* DEBUG */
  } else {
    ret = AENC_OK;
  }

  dafrm->stream_length = stream_length;
  dafrm->sample_length = safrm->sample_length;
  dafrm->flag = safrm->flag;
  aframe_enc_dst_frame_update(dafrm);

  return ret;
}

/* thread function */

static void*
aenc_thread (void *data)
{
  AENC_CONFIG *conf = &aenc_conf;
  AFRAME *safrm;
  int state;
  int rec_start_flag = 0;
  int quit_flag = 0;

  // avoid compile warnning, unnecessary
  data = NULL;
  // avoid compile warnning, unnecessary

  pthread_mutex_lock(&conf->mutex);
  aenc_thread_running = 1;
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);


  safrm = NULL;

  while (1) {
    state = get_state(conf);

    if (quit_flag || state & STATE_ERROR) {
      break;
    }
//    else
//      state_change_done(conf);
   
    while(!(state & STATE_ERROR) &&
       	(safrm = aframe_enc_src_frame_get()) == NULL) {
      pthread_mutex_lock(&conf->mutex);
      if (!conf->wait_cb_called && !(conf->state & STATE_ERROR)) {
        conf->state |= STATE_THREAD_WAIT;
        pthread_cond_wait(&conf->thread_wait_cond, &conf->mutex);
      }
      conf->wait_cb_called = 0;
      state = conf->state;
      pthread_mutex_unlock(&conf->mutex);
    }

    if (!safrm)
      continue;
    else if (state & STATE_ERROR) {
#ifdef DEBUG
      if (state & STATE_ERROR) {
        fprintf(stderr, "aenc_thread: state ERROR, safrm %p.\n", safrm);
      }
#endif /* DEBUG */
      aframe_enc_src_frame_update(safrm);
      continue;
    }

#ifdef DEBUG
    if (safrm->flag & NORMAL_AFRAME &&
	safrm->flag & (START_AFRAME|END_AFRAME|QUIT_AFRAME)) {
      fprintf(stderr, "aenc_thread: invalid ext frame.\n");
      exit(1);
    }
#endif /* DEBUG */

    if (safrm->flag & START_AFRAME) {
#ifdef DEBUG
      if (rec_start_flag) {
	fprintf(stderr, "aenc_thread: invalid start.\n");
	exit(1);
      }
#endif /* DEBUG */
      rec_start_flag = 1;
      set_state(conf, STATE_START);
    } else if (safrm->flag & END_AFRAME) {
      if (rec_start_flag) {
	if (put_flush_frame(conf) == AENC_FAIL) {
	  set_state(conf, STATE_ERROR);
#ifdef DEBUG
	  fprintf(stderr, "aenc_thread: put_flush_frame() failed.\n");
	  exit(1);
#endif /* DEBUG */
	  //break;
	}
      }
      rec_start_flag = 0;
      set_state(conf, STATE_STOP);
    } else if (safrm->flag & QUIT_AFRAME) {
      if (rec_start_flag) {
	if (put_flush_frame(conf) == AENC_FAIL) {
	  set_state(conf, STATE_ERROR);
#ifdef DEBUG
	  fprintf(stderr, "aenc_thread: put_flush_frame() failed.\n");
	  exit(1);
#endif /* DEBUG */
	  break;
	}
      }
      rec_start_flag = 0;
      set_state(conf, STATE_QUIT);
      quit_flag = 1;
    } else {
#ifdef DEBUG
      if (safrm->flag & EXT_AFRAME) {
        fprintf(stderr, "aenc_thread: invalid src frame.\n");
        exit(1);
      }
      if (!rec_start_flag) {
        fprintf(stderr, "aenc_thread: invalid state frame.\n");
        exit(1);
      }
#endif /* DEBUG */
      if (put_enc_frame(conf, safrm) == AENC_FAIL) {
        set_state(conf, STATE_ERROR);
#ifdef DEBUG
	fprintf(stderr, "aenc_thread: put_enc_frame() failed.\n");
	exit(1);
#endif /* DEBUG */
	break;
      }
    }

    aframe_enc_src_frame_update(safrm);
  }

  pthread_mutex_lock(&conf->mutex);
  aenc_thread_running = 0;
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
//  state_change_done(conf);

#ifdef DEBUG
  fprintf(stderr, "aenc_thread: done\n");
#endif /* DEBUG */
  return NULL;
}

/* interface functions */

void
aenc_enc_wait_cb(void)
{
  AENC_CONFIG *conf = &aenc_conf;

  if (!aenc_initialized)
    return;

  pthread_mutex_lock(&conf->mutex);
  conf->wait_cb_called = 1;
  if (conf->state & STATE_THREAD_WAIT) {
    conf->state &= ~STATE_THREAD_WAIT;
    pthread_cond_signal(&conf->thread_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
}

int
aenc_state_error(void)
{
  AENC_CONFIG *conf = &aenc_conf;
  int ret = AENC_FAIL;

  if (!aenc_initialized)
    return AENC_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!aenc_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return AENC_OK;
  }
  if (conf->state & STATE_THREAD_WAIT) {
    conf->state &= ~STATE_THREAD_WAIT;
    pthread_cond_signal(&conf->thread_wait_cond);
  }
  if (conf->state & STATE_ERROR) {
    ret = AENC_OK;
  } else {
    conf->state &= ~(STATE_START|STATE_PAUSE|STATE_STOP);
    conf->state |= STATE_ERROR;
#if 0
    conf->state |= STATE_INTERFACE_WAIT;
    while (conf->state & STATE_INTERFACE_WAIT) {
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    }
#endif
    ret = AENC_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
aenc_state_start(void)
{
  AENC_CONFIG *conf = &aenc_conf;
  int ret = AENC_FAIL;

  if (!aenc_initialized)
    return AENC_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!aenc_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return AENC_OK;
  }
  if (conf->state & (STATE_START|STATE_QUIT|STATE_ERROR))
    ret = AENC_OK;
  else {
    while (!(conf->state & STATE_START)) {
      conf->state |= STATE_INTERFACE_WAIT;
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    }
    ret = AENC_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
aenc_state_stop(void)
{
  AENC_CONFIG *conf = &aenc_conf;
  int ret = AENC_FAIL;

  if (!aenc_initialized)
    return AENC_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!aenc_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return AENC_OK;
  }
  if (conf->state & (STATE_STOP|STATE_QUIT|STATE_ERROR))
    ret = AENC_OK;
  else {
    while (!(conf->state & STATE_STOP)) {
      conf->state |= STATE_INTERFACE_WAIT;
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    }
    ret = AENC_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
aenc_state_quit(void)
{
  AENC_CONFIG *conf = &aenc_conf;
  int ret = AENC_FAIL;

  if (!aenc_initialized)
    return AENC_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!aenc_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return AENC_OK;
  }
  if (conf->state & (STATE_QUIT|STATE_ERROR))
    ret = AENC_OK;
  else {
    while (!(conf->state & STATE_QUIT)) {
      conf->state |= STATE_INTERFACE_WAIT;
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    }
    ret = AENC_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
aenc_state_pause(void)
{
  if (!aenc_initialized)
    return AENC_OK;
  return AENC_OK;
}

int
aenc_state_toggle_pause(void)
{
  if (!aenc_initialized)
    return AENC_OK;
  return AENC_OK;
}

int
aenc_error_quit (void)
{
  AENC_CONFIG *conf = &aenc_conf;
  int retcode;
  void* retval;

  if (!aenc_initialized)
    return AENC_OK;

  aenc_state_error();

  retcode = pthread_join (conf->aenc_th, &retval);
  if (retcode != 0)
    fprintf (stderr, "aenc_quit() thread join error %d\n", retcode);

//  aframe_mgr_quit();
//  acodec_encode_quit();

  aenc_initialized = 0;
//  fprintf (stderr, "aenc_quit: done\n");
  return AENC_OK;
}

int
aenc_quit (void)
{
  AENC_CONFIG *conf = &aenc_conf;
  int retcode;
  void* retval;

  if (!aenc_initialized)
    return AENC_OK;

  aenc_state_quit();

  retcode = pthread_join (conf->aenc_th, &retval);
  if (retcode != 0)
    fprintf (stderr, "aenc_quit() thread join error %d\n", retcode);

//  aframe_mgr_quit();
//  acodec_encode_quit();

  aenc_initialized = 0;
//  fprintf (stderr, "aenc_quit: done\n");
  return AENC_OK;
}

int
aenc_init (uint32_t codec, int rate, int channels, int bitspspl)
{
  AENC_CONFIG *conf = &aenc_conf;
  int retcode;
  int frame_spl_size;
  int need_config = 0;
static int first = 1;

  if (first) {
    aenc_initialized = 0;
    aenc_thread_running = 0;
    conf->rate = 0;
    conf->channels = 0;
    conf->bitspspl = 0;
    conf->wait_cb_called = 0;
    conf->codec = 0;
    conf->state = 0;
    first = 0;
    need_config = 1;
  }

  if (rate == 0)
    rate = 22050;
  if (channels == 0)
    channels = 2;
  if (bitspspl == 0)
    bitspspl = 16;

  if (!aenc_initialized)
    need_config = 1;
  else if (conf->rate != rate)
    need_config = 1;
  else if (conf->channels != channels)
    need_config = 1;
  else if (conf->bitspspl != bitspspl)
    need_config = 1;
  else if (conf->codec != codec)
    need_config = 1;

  if (!need_config) {
    return AENC_OK;
  }

  if (aenc_initialized) {
    fprintf(stderr, "aenc_init: aenc_initialized TRUE !!!\n");
    aenc_quit();
  }

  conf->rate = rate;
  conf->channels = channels;
  conf->bitspspl = bitspspl;

#if 0
  retcode = acodec_encode_init(codec, rate, channels, bitspspl);
  if (retcode != ACODEC_OK) {
    fprintf(stderr, "aenc_config: acodec_init failer.\n");
    return AENC_FAIL;
  }
#endif

  frame_spl_size = acodec_encode_get_frame_size();
  retcode = aframe_mgr_init(rate, channels, bitspspl, frame_spl_size);
  if (retcode != AFRAME_MGR_OK) {
    fprintf(stderr, "aenc_config: aframe_mgr_init failer.\n");
    return AENC_FAIL;
  }
  aframe_set_enc_wait_cb(aenc_enc_wait_cb);

  conf->wait_cb_called = 0;
  conf->state = STATE_STOP;
  pthread_mutex_init (&conf->mutex, NULL);
  pthread_cond_init (&conf->interface_wait_cond, NULL);
  pthread_cond_init (&conf->thread_wait_cond, NULL);

  retcode = pthread_create (&conf->aenc_th, NULL, aenc_thread, NULL);
  if (retcode != 0) {
    fprintf (stderr, "aenc_init() thread create failed %d\n", retcode);
    return AENC_FAIL;
  }

  pthread_mutex_lock(&conf->mutex);
  if (!aenc_thread_running) {
    conf->state |= STATE_INTERFACE_WAIT;
    pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
  }
  pthread_mutex_unlock(&conf->mutex);

  aenc_initialized = 1;

  return AENC_OK;
}

#if 0
void
aenc_print_param(void)
{
  AENC_CONFIG *conf = &aenc_conf;
//  SEL_COMPO *sel;
//  int i;

  fprintf (stderr, "AUDIO PARAM\n");
#if 0
  sel = codecs;
  for (i=0; i < sizeof(codecs)/sizeof(SEL_COMPO); i++, sel++) {
    if (sel->id == conf->fourcc) {
      fprintf (stderr, "video codec:            %s\n", sel->name);
      break;
    }
  }
#endif
  fprintf (stderr, "fps:                    %f\n", conf->fps);
  fprintf (stderr, "dimmention:             %dx%d\n", conf->width, conf->height);
  fprintf (stderr, "buffers:                %d\n", conf->max_buffers);
}
#endif

