/* ac_pcm.c */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif


#include <stdio.h>
#include <string.h>
#include <inttypes.h>
//#include <endian.h>

#include "acodec.h"
#include "ac_pcm.h"

static PCM_PARAM pcm_enc_param;
static PCM_PARAM pcm_dec_param;


const char*
pcm_get_codec_name(void)
{
  return "PCM";
}

uint32_t
pcm_get_cap(uint32_t codec)
{
  if (codec != ACODEC_PCM && codec != ACODEC_PCM_BE && codec != ACODEC_PCM_LE) {
    fprintf(stderr, "pcm_get_cap: invalid codec.\n");
  }
  return ACODEC_CAP_NONE;
}

uint32_t
pcm_fourcc_to_codec(const char *fourcc)
{
  if (strncmp(fourcc, "twos", 4) == 0) {
    return ACODEC_PCM_BE;
  } else if (strncmp(fourcc, "sowt", 4) == 0) {
    return ACODEC_PCM_LE;
  }
  return ACODEC_UNKNOWN;
}

uint32_t
pcm_code_to_codec(int code)
{
  if (code == 0x1)
    return ACODEC_PCM_LE;
  return ACODEC_UNKNOWN;
}

const char*
pcm_codec_to_fourcc(uint32_t codec)
{
  if (codec == ACODEC_PCM) {
    return "sowt";
  } else if (codec == ACODEC_PCM_LE) {
    return "sowt";
  } else if (codec == ACODEC_PCM_BE) {
    return "twos";
  }
  return NULL;
}

int
pcm_codec_to_code(uint32_t codec)
{
  if (codec == ACODEC_PCM)
    return 0x1;
  else if (codec == ACODEC_PCM_LE)
    return 0x1;
  return 0;
}


/* encoder functions */

int
pcm_encode (void *pcm_buf[], int *num_samples, uint8_t *stream_buf,int stream_buf_size)
{
  int c, i, byte_size;
  int num_spl = *num_samples;
  int chan = pcm_enc_param.channels;

  if (pcm_enc_param.bits == 16) {
    unsigned short *sp;
    unsigned short *dp;

    byte_size = num_spl * pcm_enc_param.bytespspl;
    if (byte_size > stream_buf_size) {
      fprintf(stderr, "pcm_encode: stread_buf_size too small.\n");
      return 0;
    }
    if (pcm_enc_param.swap) {
      for (c = 0; c < chan; c++) {
        sp = (unsigned short*)pcm_buf[c];
        dp = (unsigned short*)stream_buf;
	dp += c;
        for (i = 0; i < num_spl; i++) {
          *dp = (((sp[i])<<8)|(sp[i]>>8));
	  dp += chan;
	}
      }
    } else {
      for (c = 0; c < chan; c++) {
        sp = (unsigned short*)pcm_buf[c];
        dp = (unsigned short*)stream_buf;
	dp += c;
        for (i = 0; i < num_spl; i++) {
          *dp = sp[i];
	  dp += chan;
	}
      }
    }
  } else {
    short *sp;
    unsigned char *dp;

    byte_size = num_spl * pcm_enc_param.bytespspl;
    if (byte_size > stream_buf_size) {
      fprintf(stderr, "pcm_encode: stread_buf_size too small.\n");
      return 0;
    }
    for (c = 0; c < chan; c++) {
      sp = (short*)pcm_buf[c];
      dp = (unsigned char*)stream_buf;
      dp += c;
      for (i = 0; i < num_spl; i++) {
        *dp = (sp[i]>>8) + 128;
        dp += chan;
      }
    }
  }

  return byte_size;
}

int
pcm_encode_flush (int *num_samples, uint8_t *stream_buf, int stream_buf_size)
{
  stream_buf = NULL;
  stream_buf_size = 0;
  *num_samples = 0;
  return 0;
}

int
pcm_encode_rate_cap(int rate)
{
  if (rate < 100)
    rate *= 1000;

  if (rate < 8000)
    rate = 8000;
  else if (rate < 11025)
    rate = 8000;
  else if (rate < 12000)
    rate = 11025;
  else if (rate < 16000)
    rate = 12000;
  else if (rate < 22050)
    rate = 16000;
  else if (rate < 24000)
    rate = 22050;
  else if (rate < 32000)
    rate = 24000;
  else if (rate < 44100)
    rate = 32000;
  else if (rate < 48000)
    rate = 44100;
  else if (rate >= 48000)
    rate = 48000;
  return rate;
}

int
pcm_encode_channels_cap(int channels)
{
  int ret;
  switch(channels) {
    default:
    case 2:
      ret = 2;
      break;
    case 1:
      ret = 1;
      break;
  }
  return ret;
}

int
pcm_encode_bits_cap(int bits)
{
  int ret;
  switch(bits) {
    default:
    case 16:
      ret = 16;
      break;
  }
  return ret;
}

int
pcm_encode_get_frame_size(void)
{
  return pcm_enc_param.frame_size;
}

int
pcm_encode_get_bitrate(void)
{
  return pcm_enc_param.bitrate;
}

int
pcm_encode_get_out_rate(void)
{
  return pcm_enc_param.rate;
}

int
pcm_encode_quit (void)
{
  pcm_enc_param.rate = 0;
  pcm_enc_param.channels = 0;
  pcm_enc_param.bits = 0;
  pcm_enc_param.bytespspl = 0;
  pcm_enc_param.bitrate = 0;
  pcm_enc_param.frame_size = 0;
  pcm_enc_param.swap = 0;
  return ACODEC_OK;
}

int
pcm_encode_init (ACODEC *acodec)
{
  double b;
  int rate, channels, bits;

  rate = acodec->rate;
  channels = acodec->channels;
  bits = acodec->bits;
  pcm_enc_param.rate = rate;
  pcm_enc_param.channels = channels;
  pcm_enc_param.bits = bits;
  pcm_enc_param.bytespspl = channels * ((bits+7)/8);
  b = (double)rate * (double)channels * (double)((bits+7)/8) * 8.0;
  pcm_enc_param.bitrate = ((int)b) / 1000;
  if (rate <= 22050)
    pcm_enc_param.frame_size = 576;
  else
    pcm_enc_param.frame_size = 1152;

  if (acodec->codec == ACODEC_PCM_BE) {
//#if __BYTE_ORDER == __BIG_ENDIAN
#ifdef WORDS_BIGENDIAN 
    pcm_enc_param.swap = 0;
#else
    pcm_enc_param.swap = 1;
#endif
  } else if (acodec->codec == ACODEC_PCM_LE ||
             acodec->codec == ACODEC_PCM) {
//#if __BYTE_ORDER == __BIG_ENDIAN
#ifdef WORDS_BIGENDIAN 
    pcm_enc_param.swap = 1;
#else
    pcm_enc_param.swap = 0;
#endif
    acodec->codec = ACODEC_PCM_LE;
  } else {
    fprintf(stderr, "pcm_encode_init: unsupported codec found.\n");
    return ACODEC_FAIL;
  }
  pcm_enc_param.codec = acodec->codec;

  return ACODEC_OK;
}

void
pcm_encode_print_param(void)
{
  fprintf (stderr, "ENCPARAM PCM\n");
  fprintf (stderr, "rate:             %d\n", pcm_enc_param.rate);
  fprintf (stderr, "channels:         %d\n", pcm_enc_param.channels);
  fprintf (stderr, "bits:             %d\n", pcm_enc_param.bits);
  if (pcm_enc_param.codec == ACODEC_PCM_LE ||
      pcm_enc_param.codec == ACODEC_PCM)
    fprintf (stderr, "byte order:       little endian\n");
  else if (pcm_enc_param.codec == ACODEC_PCM_BE)
    fprintf (stderr, "byte order:       big endian\n");
  fflush (stderr);
  return;
}


/* decoder functions */

int
pcm_decode (void *pcm_buf[], uint8_t *stream,int stream_size)
{
  int c, i;
  int num_spl;
  int chan = pcm_dec_param.channels;

  if (pcm_dec_param.bits == 16) {
    unsigned short *sp;
    unsigned short *dp;

    num_spl = stream_size / pcm_dec_param.bytespspl;
    if (pcm_dec_param.swap) {
      for (c = 0; c < chan; c++) {
        sp = (unsigned short*)stream;
        dp = (unsigned short*)pcm_buf[c];
	sp += c;
        for (i = 0; i < num_spl; i++) {
	  dp[i] = (*sp<<8)|(*sp>>8);
	  sp += chan;
	}
      }
    } else {
      for (c = 0; c < chan; c++) {
        sp = (unsigned short*)stream;
        dp = (unsigned short*)pcm_buf[c];
	sp += c;
        for (i = 0; i < num_spl; i++) {
	  dp[i] = *sp;
	  sp += chan;
	}
      }
    }
  } else {
    short *dp;
    unsigned char *sp;

    num_spl = stream_size / pcm_dec_param.bytespspl;
    for (c = 0; c < chan; c++) {
      sp = (unsigned char*)stream;
      dp = (short*)pcm_buf[c];
      sp += c;
      for (i = 0; i < num_spl; i++) {
        dp[i] = (short)((*sp-128)<<8);
        sp += chan;
      }
    }
  }

  return num_spl;
}

int
pcm_decode_get_rate(void)
{
  return pcm_dec_param.rate;
}

int
pcm_decode_get_channels(void)
{
  return pcm_dec_param.channels;
}

int
pcm_decode_get_bits(void)
{
  return pcm_dec_param.bits;
}

int
pcm_decode_get_bitrate(void)
{
  return pcm_dec_param.bitrate;
}

int
pcm_decode_analyze_stream(ACODEC *acodec, uint8_t *stream, int stream_size)
{
  // avoid warnning, unnecessary
  acodec = acodec; stream = stream; stream_size = stream_size;
  // avoid warnning, unnecessary

  return ACODEC_FAIL;
}


int
pcm_decode_quit (void)
{
  pcm_dec_param.rate = 0;
  pcm_dec_param.channels = 0;
  pcm_dec_param.bits = 0;
  pcm_dec_param.bytespspl = 0;
  pcm_dec_param.bitrate = 0;
  pcm_dec_param.frame_size = 0;
  pcm_dec_param.swap = 0;
  return ACODEC_OK;
}

int
pcm_decode_init (ACODEC *acodec)
{
  double b;
  int rate, channels, bits;

  rate = acodec->rate;
  channels = acodec->channels;
  bits = acodec->bits;
  pcm_dec_param.rate = rate;
  pcm_dec_param.channels = channels;
  pcm_dec_param.bits = bits;
  pcm_dec_param.bytespspl = channels * ((bits+7)/8);
  b = (double)rate * (double)channels * (double)((bits+7)/8) * 8.0;
  pcm_dec_param.bitrate = ((int)b) / 1000;
  if (rate <= 22050)
    pcm_dec_param.frame_size = 576;
  else
    pcm_dec_param.frame_size = 1152;

  if (acodec->codec == ACODEC_PCM_BE) {
//#if __BYTE_ORDER == __BIG_ENDIAN
#ifdef WORDS_BIGENDIAN 
    pcm_dec_param.swap = 0;
#else
    pcm_dec_param.swap = 1;
#endif
  } else if (acodec->codec == ACODEC_PCM_LE ||
             acodec->codec == ACODEC_PCM) {
//#if __BYTE_ORDER == __BIG_ENDIAN
#ifdef WORDS_BIGENDIAN 
    pcm_dec_param.swap = 1;
#else
    pcm_dec_param.swap = 0;
#endif
    acodec->codec = ACODEC_PCM_LE;
  } else {
    fprintf(stderr, "pcm_decode_init: unsupported codec found.\n");
    return ACODEC_FAIL;
  }
  pcm_dec_param.codec = acodec->codec;

  return ACODEC_OK;
}

void
pcm_decode_print_param(void)
{
  fprintf (stderr, "ENCPARAM PCM\n");
  fprintf (stderr, "rate:             %d\n", pcm_dec_param.rate);
  fprintf (stderr, "channels:         %d\n", pcm_dec_param.channels);
  fprintf (stderr, "bits:             %d\n", pcm_dec_param.bits);
  if (pcm_dec_param.codec == ACODEC_PCM_LE ||
      pcm_dec_param.codec == ACODEC_PCM)
    fprintf (stderr, "byte order:       little endian\n");
  else if (pcm_dec_param.codec == ACODEC_PCM_BE)
    fprintf (stderr, "byte order:       big endian\n");
  fflush (stderr);
  return;
}


/* function structure */

ACODEC_FUNCS pcm_funcs = {
  pcm_get_codec_name,
  pcm_get_cap,
  pcm_fourcc_to_codec,
  pcm_code_to_codec,
  pcm_codec_to_fourcc,
  pcm_codec_to_code,

  pcm_encode_init,
  pcm_encode_quit,
  pcm_encode,
  pcm_encode_flush,
  pcm_encode_rate_cap,
  pcm_encode_channels_cap,
  pcm_encode_bits_cap,
  pcm_encode_get_frame_size,
  pcm_encode_get_bitrate,
  pcm_encode_get_out_rate,
  pcm_encode_print_param,

  pcm_decode_init,
  pcm_decode_quit,
  pcm_decode,
  pcm_decode_get_rate,
  pcm_decode_get_channels,
  pcm_decode_get_bits,
  pcm_decode_get_bitrate,
  pcm_decode_analyze_stream,
  pcm_decode_print_param,
};

