/* vc_raw.c */

#include <stdio.h>
#include <string.h>

//#include "xvid.h"
//#include "image/image.h"

#include "util.h"
#include "csp.h"
#include "vcodec.h"
#include "vc_raw.h"
//#include "vc_xvid.h"

static int vc_encode_initialized = 0;
static int vc_decode_initialized = 0;

static VC_RAW_PARAM vc_raw_enc;
static VC_RAW_PARAM vc_raw_dec;

static struct CODEC_FOURCC {
  uint32_t codec;
  const char *fourcc_str;
  uint32_t codec_cap;
} codec_fourcc[] = {
  { VCODEC_RGB24, "RGB", VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
  { VCODEC_YV12,  "YV12", VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
};

static int codec_fourcc_num = sizeof(codec_fourcc)/sizeof(struct CODEC_FOURCC);

uint32_t
vc_raw_codec_cap(uint32_t codec, uint32_t cap_flag)
{
  int i;

  for (i = 0; i < codec_fourcc_num; i++) {
    if (codec == codec_fourcc[i].codec)
      return (cap_flag & codec_fourcc[i].codec_cap);
  }
  return VCODEC_CAP_NONE;
}

const char*
vc_raw_codec_to_fourcc_str(uint32_t codec)
{
  int i;

  for (i = 0; i < codec_fourcc_num; i++) {
    if (codec == codec_fourcc[i].codec)
      return codec_fourcc[i].fourcc_str;
  }
  return NULL;
}

uint32_t
vc_raw_fourcc_str_to_codec(const char *fourcc_str)
{
  int i;

  if (fourcc_str == NULL)
    return VCODEC_RGB24;

  for (i = 0; i < codec_fourcc_num; i++) {
    if (!strcmp(fourcc_str, codec_fourcc[i].fourcc_str))
      return codec_fourcc[i].codec;
  }
  return VCODEC_UNKNOWN;
}

/* encode functions */

int
vc_encode_yv12_csp_cap(int csp)
{
  int ret;
  switch (csp) {
    default:
      ret = CSP_UNKNOWN;
      break;
    case CSP_UNKNOWN:
      ret = CSP_YV12;
      break;
    case CSP_I420:
    case CSP_YUV420P:
    case CSP_IYUV:
    case CSP_YV12:
//    case CSP_RGB24:
      ret = csp;
      break;
  }
  return ret;
}

int
vc_encode_rgb24_csp_cap(int csp)
{
  int ret;
  switch (csp) {
    default:
      ret = CSP_UNKNOWN;
      break;
    case CSP_UNKNOWN:
      ret = CSP_RGB24;
      break;
    case CSP_RGB24:
//    case CSP_I420:
//    case CSP_YUV420P:
//    case CSP_IYUV:
//    case CSP_YV12:
      ret = csp;
      break;
  }
  return ret;
}

int
vc_raw_encode (unsigned char *pic_data, unsigned char *stream_buf,
    int stream_buf_size, unsigned int *flag)
{
  int data_size = vc_raw_enc.data_size;

  if (!vc_encode_initialized) {
    fprintf(stderr, "vc_raw_encode: uninitialized.\n");
    return VCODEC_FAIL;
  }

  if (stream_buf_size < vc_raw_enc.data_size) {
    fprintf(stderr, "vc_raw_encode: stream_buf_size is too small.\n");
  }

  if (vc_raw_enc.codec == VCODEC_YV12) {
    data_size = vc_raw_enc.width * vc_raw_enc.height * 12 / 8;
    if (vc_raw_enc.in_csp == CSP_YV12) {
      memcpy (stream_buf, pic_data, vc_raw_enc.data_size);
    } else if (vc_raw_enc.in_csp == CSP_I420 || vc_raw_enc.in_csp == CSP_YUV420P
	|| vc_raw_enc.in_csp == CSP_IYUV) {
      memcpy (stream_buf, pic_data, vc_raw_enc.y_size);
      memcpy (stream_buf + vc_raw_enc.y_size,
	      pic_data + vc_raw_enc.y_size + vc_raw_enc.uv_size,
	      vc_raw_enc.uv_size);
      memcpy (stream_buf + vc_raw_enc.y_size + vc_raw_enc.uv_size,
	      pic_data + vc_raw_enc.y_size,
	      vc_raw_enc.uv_size);
#if 0
    } else if (vc_raw_enc.in_csp == CSP_RGB24) {
      IMAGE image;
      int xcsp;
      image.y = stream_buf;
      image.v = stream_buf + vc_raw_enc.y_size;
      image.u = stream_buf + vc_raw_enc.y_size + vc_raw_enc.uv_size;
      xcsp = XVID_CSP_RGB24;
      image_input(&image, vc_raw_enc.width, vc_raw_enc.height, vc_raw_enc.width,
	          pic_data, xcsp);
#endif
    } else {
      fprintf(stderr, "vc_raw_encode: in_csp invalid %s.\n", csp_to_str(vc_raw_enc.in_csp));
      return VCODEC_FAIL;
    }
  } else if (vc_raw_enc.codec == VCODEC_RGB24) {
//    IMAGE image;
//    int xcsp;
    data_size = vc_raw_enc.width * vc_raw_enc.height * 24 / 8;
    if (vc_raw_enc.in_csp == CSP_RGB24) {
      if (vc_raw_enc.vflip) {
        int i;
        int height = vc_raw_enc.height;
        int bytes_per_line = vc_raw_enc.bytes_per_line;
        char *dest = stream_buf;
        char *src = pic_data + vc_raw_enc.data_size - bytes_per_line;
        for (i = 0; i < height; i++) {
          memcpy (dest, src, bytes_per_line);
          src -= bytes_per_line;
          dest += bytes_per_line;
        }
      } else {
        memcpy (stream_buf, pic_data, vc_raw_enc.data_size);
      }
#if 0
    } else if (vc_raw_enc.in_csp == CSP_I420 || vc_raw_enc.in_csp == CSP_YUV420P
	|| vc_raw_enc.in_csp == CSP_IYUV) {
      image.y = pic_data;
      image.u = pic_data + vc_raw_enc.y_size;
      image.v = pic_data + vc_raw_enc.y_size + vc_raw_enc.uv_size;
      xcsp = XVID_CSP_RGB24;
      image_output(&image,vc_raw_enc.width, vc_raw_enc.height, vc_raw_enc.width,
	          stream_buf, vc_raw_enc.width, xcsp);
    } else if (vc_raw_enc.in_csp == CSP_YV12) {
      image.y = pic_data;
      image.v = pic_data + vc_raw_enc.y_size;
      image.u = pic_data + vc_raw_enc.y_size + vc_raw_enc.uv_size;
      xcsp = XVID_CSP_RGB24;
      image_input(&image,vc_raw_enc.width, vc_raw_enc.height, vc_raw_enc.width,
	          stream_buf, xcsp);
#endif
    } else {
      fprintf(stderr, "vc_raw_encode: in_csp invalid %s.\n", csp_to_str(vc_raw_enc.in_csp));
      return VCODEC_FAIL;
    }
  }

  *flag = VCODEC_IS_INTRA;
  return data_size;
}

int
vc_raw_encode_quit (void)
{
  vc_raw_enc.codec = 0;
  vc_raw_enc.width = 0;
  vc_raw_enc.height = 0;
  vc_raw_enc.in_csp = CSP_UNKNOWN;
  vc_raw_enc.out_csp = CSP_UNKNOWN;
  vc_raw_enc.bytes_per_line = 0;
  vc_raw_enc.data_size = 0;
  vc_raw_enc.y_size = 0;
  vc_raw_enc.uv_size = 0;

  vc_encode_initialized = 0;
  return VCODEC_OK;
}

void
vc_raw_encode_print_param(void)
{
  if (!vc_encode_initialized) {
    fprintf (stdout, "ENCPARAM VIDEO CODEC uninitialized.\n");
    return;
  }
  fprintf (stdout, "%s ENCODE PARAMETER\n",
      (vc_raw_enc.codec == VCODEC_YV12) ? "YV12" : "RGB");
  fprintf (stdout,"dimmention: %dx%d\n",vc_raw_enc.width,vc_raw_enc.height);
  fflush (stdout);
}

int
vc_raw_encode_init (VCODEC *vcodec)
{
  static int first = 1;
  int csp;
  int depth;

  if (vcodec->codec != VCODEC_RGB24 && vcodec->codec != VCODEC_YV12) {
    fprintf(stderr, "vc_raw_init: invalid codec.\n");
    return VCODEC_FAIL;
  }

  if (first) {
    vc_raw_enc.codec = 0;
    vc_raw_enc.width = 0;
    vc_raw_enc.height = 0;
    vc_raw_enc.in_csp = CSP_UNKNOWN;
    vc_raw_enc.out_csp = CSP_UNKNOWN;
    first = 0;
//    vc_xvid_init();
  }

  vc_raw_enc.codec = vcodec->codec;

  csp = CSP_UNKNOWN;
  if (vcodec->codec == VCODEC_RGB24) {
    csp = vc_encode_rgb24_csp_cap(vcodec->in_csp);
    if (csp != vcodec->in_csp) {
      fprintf(stderr,"vc_raw_init: invalid in csp %s.\n",csp_to_str(vcodec->in_csp));
      return VCODEC_FAIL;
    }
  } else if (vcodec->codec == VCODEC_YV12) {
    csp = vc_encode_yv12_csp_cap(vcodec->in_csp);
    if (csp != vcodec->in_csp) {
      fprintf(stderr,"vc_raw_init: invalid in csp %s.\n",csp_to_str(vcodec->in_csp));
      return VCODEC_FAIL;
    }
  }
  vc_raw_enc.in_csp = csp;

  switch (vcodec->codec) {
    case VCODEC_RGB24:
      vcodec->out_csp = CSP_RGB24;
      break;
    case VCODEC_YV12:
      vcodec->out_csp = CSP_YV12;
      break;
    default:
      return VCODEC_FAIL;
  }
  vc_raw_enc.out_csp = vcodec->out_csp;

  vc_raw_enc.width = vcodec->width;
  vc_raw_enc.height = vcodec->height;

  depth = csp_to_pixel_depth(vc_raw_enc.in_csp);

  switch (vc_raw_enc.codec) {
    case VCODEC_RGB24:
      vc_raw_enc.bytes_per_line = vc_raw_enc.width * depth / 8;
      vc_raw_enc.data_size = vc_raw_enc.width * vc_raw_enc.height * depth / 8;
      break;
    case VCODEC_YV12:
      vc_raw_enc.y_size = vc_raw_enc.width * vc_raw_enc.height;
      vc_raw_enc.uv_size = vc_raw_enc.y_size / 4;
      vc_raw_enc.data_size = vc_raw_enc.width * vc_raw_enc.height * depth / 8;
      break;
  }

  vc_raw_dec.vflip = 1;

  vc_encode_initialized = 1;

  return VCODEC_OK;
}

/* decode functions */

int
vc_decode_yv12_csp_cap(int csp)
{
  int ret;
  switch (csp) {
    default:
      ret = CSP_UNKNOWN;
      break;
    case CSP_UNKNOWN:
      ret = CSP_YV12;
      break;
//    case CSP_RGB24:
    case CSP_I420:
    case CSP_YUV420P:
    case CSP_IYUV:
    case CSP_YV12:
      ret = csp;
      break;
  }
  return ret;
}

int
vc_decode_rgb24_csp_cap(int csp)
{
  int ret;
  switch (csp) {
    default:
      ret = CSP_UNKNOWN;
      break;
    case CSP_UNKNOWN:
      ret = CSP_RGB24;
      break;
    case CSP_RGB24:
//    case CSP_I420:
//    case CSP_YUV420P:
//    case CSP_IYUV:
//    case CSP_YV12:
      ret = csp;
      break;
  }
  return ret;
}

int
vc_raw_decode (unsigned char *pic_data, unsigned char *stream,
    int stream_length, unsigned int *flag)
{
  int data_size = vc_raw_dec.data_size;

  if (!vc_decode_initialized) {
    fprintf(stderr, "vc_raw_decode: uninitialized.\n");
    return VCODEC_FAIL;
  }

  if (stream_length < vc_raw_enc.data_size) {
    fprintf(stderr, "vc_raw_decode: stream_length is too small.\n");
  }

  if (flag && *flag & VCODEC_NULL_DECODE)
    return 0;

  if (vc_raw_dec.codec == VCODEC_YV12) {
    if (vc_raw_dec.out_csp == CSP_YV12) {
      memcpy (pic_data, stream, vc_raw_dec.data_size);
      data_size = vc_raw_enc.width * vc_raw_enc.height * 12 / 8;
    } else if (vc_raw_dec.out_csp == CSP_I420|| vc_raw_dec.in_csp == CSP_YUV420P
	|| vc_raw_dec.in_csp == CSP_IYUV) {
      memcpy (pic_data, stream, vc_raw_dec.y_size);
      memcpy (pic_data + vc_raw_dec.y_size,
	      stream + vc_raw_dec.y_size + vc_raw_dec.uv_size,
	      vc_raw_dec.uv_size);
      memcpy (pic_data + vc_raw_dec.y_size + vc_raw_dec.uv_size,
	      stream + vc_raw_dec.y_size,
	      vc_raw_dec.uv_size);
      data_size = vc_raw_enc.width * vc_raw_enc.height * 12 / 8;
#if 0
    } else if (vc_raw_dec.out_csp == CSP_RGB24) {
      IMAGE image;
      int xcsp;
      data_size = vc_raw_enc.width * vc_raw_enc.height * 24 / 8;
      image.y = stream;
      image.v = stream + vc_raw_dec.y_size;
      image.u = stream + vc_raw_dec.y_size + vc_raw_dec.uv_size;
      xcsp = XVID_CSP_RGB24;
      image_output(&image,vc_raw_dec.width, vc_raw_dec.height, vc_raw_dec.width,
	          pic_data, vc_raw_dec.width, xcsp);
#endif
    } else {
      fprintf(stderr, "vc_raw_decode: out_csp invalid %s.\n", csp_to_str(vc_raw_dec.out_csp));
      return VCODEC_FAIL;
    }
  } else if (vc_raw_dec.codec == VCODEC_RGB24) {
//    IMAGE image;
//    int xcsp;
    if (vc_raw_dec.out_csp == CSP_RGB24) {
      data_size = vc_raw_enc.width * vc_raw_enc.height * 24 / 8;
      if (vc_raw_dec.vflip) {
        int i;
        int height = vc_raw_dec.height;
        int bytes_per_line = vc_raw_dec.bytes_per_line;
        char *dest = pic_data;
        char *src = stream + vc_raw_dec.data_size - bytes_per_line;
        for (i = 0; i < height; i++) {
          memcpy (dest, src, bytes_per_line);
          src -= bytes_per_line;
          dest += bytes_per_line;
        }
      } else {
        memcpy (pic_data, stream, vc_raw_dec.data_size);
      }
#if 0
    } else if (vc_raw_dec.out_csp==CSP_I420 || vc_raw_dec.out_csp == CSP_YUV420P
	|| vc_raw_dec.out_csp == CSP_IYUV) {
      data_size = vc_raw_enc.width * vc_raw_enc.height * 12 / 8;
      image.y = pic_data;
      image.u = pic_data + vc_raw_dec.y_size;
      image.v = pic_data + vc_raw_dec.y_size + vc_raw_dec.uv_size;
      xcsp = XVID_CSP_RGB24;
      image_input(&image,vc_raw_dec.width, vc_raw_dec.height, vc_raw_dec.width,
	          stream, xcsp);
    } else if (vc_raw_dec.out_csp == CSP_YV12) {
      data_size = vc_raw_enc.width * vc_raw_enc.height * 12 / 8;
      image.y = pic_data;
      image.v = pic_data + vc_raw_dec.y_size;
      image.u = pic_data + vc_raw_dec.y_size + vc_raw_dec.uv_size;
      xcsp = XVID_CSP_RGB24;
      image_input(&image,vc_raw_dec.width, vc_raw_dec.height, vc_raw_dec.width,
	          stream, xcsp);
#endif
    } else {
      fprintf(stderr, "vc_raw_decode: out_csp invalid %s.\n", csp_to_str(vc_raw_dec.out_csp));
      return VCODEC_FAIL;
    }
  }

  *flag = VCODEC_IS_INTRA;
  return data_size;
}

int
vc_raw_decode_quit (void)
{
  vc_raw_dec.codec = 0;
  vc_raw_dec.width = 0;
  vc_raw_dec.height = 0;
  vc_raw_dec.in_csp = CSP_UNKNOWN;
  vc_raw_dec.out_csp = CSP_UNKNOWN;
  vc_raw_dec.bytes_per_line = 0;
  vc_raw_dec.data_size = 0;
  vc_raw_dec.y_size = 0;
  vc_raw_dec.uv_size = 0;

  vc_decode_initialized = 0;
  return VCODEC_OK;
}

void
vc_raw_decode_print_param(void)
{
  if (!vc_decode_initialized) {
    fprintf (stdout, "RAW DECODE uninitialized.\n");
    return;
  }
  fprintf (stdout, "%s DECODE PARAMETER\n",
      (vc_raw_dec.codec == VCODEC_YV12) ? "YV12" : "RGB");
  fprintf (stdout,"dimmention: %dx%d\n",vc_raw_dec.width,vc_raw_dec.height);
  fprintf (stdout,"out csp:    %s\n",csp_to_str(vc_raw_dec.out_csp));
  fflush (stdout);
}

int
vc_raw_decode_init (VCODEC *vcodec)
{
  static int first = 1;
  int csp;
  int depth;

  if (vcodec->codec != VCODEC_RGB24 && vcodec->codec != VCODEC_YV12) {
    fprintf(stderr, "vc_raw_init: invalid codec.\n");
    return FAIL;
  }

  if (first) {
    vc_raw_dec.codec = 0;
    vc_raw_dec.width = 0;
    vc_raw_dec.height = 0;
    vc_raw_dec.in_csp = CSP_UNKNOWN;
    vc_raw_dec.out_csp = CSP_UNKNOWN;
    first = 0;
//    vc_xvid_init();
  }

  vc_raw_dec.codec = vcodec->codec;

  csp = CSP_UNKNOWN;
  if (vcodec->codec == VCODEC_RGB24) {
    csp = vc_decode_rgb24_csp_cap(vcodec->out_csp);
    if (csp != vcodec->out_csp) {
      fprintf(stderr,"vc_raw_decode_init: invalid in csp %s.\n",csp_to_str(vcodec->out_csp));
      return VCODEC_FAIL;
    }
  } else if (vcodec->codec == VCODEC_YV12) {
    csp = vc_decode_yv12_csp_cap(vcodec->out_csp);
    if (csp != vcodec->out_csp) {
      fprintf(stderr,"vc_raw_decode_init: invalid in csp %s.\n",csp_to_str(vcodec->out_csp));
      return VCODEC_FAIL;
    }
  }
  vc_raw_dec.out_csp = csp;

  switch (vcodec->codec) {
    case VCODEC_RGB24:
      vcodec->in_csp = CSP_RGB24;
      break;
    case VCODEC_YV12:
      vcodec->in_csp = CSP_YV12;
      break;
    default:
      return VCODEC_FAIL;
  }
  vc_raw_dec.in_csp = vcodec->in_csp;

  vc_raw_dec.width = vcodec->width;
  vc_raw_dec.height = vcodec->height;

  depth = csp_to_pixel_depth(vc_raw_dec.out_csp);

  switch (vc_raw_dec.codec) {
    case VCODEC_RGB24:
      vc_raw_dec.bytes_per_line = vc_raw_dec.width * depth / 8;
      vc_raw_dec.data_size = vc_raw_dec.width * vc_raw_dec.height * depth / 8;
      break;
    case VCODEC_YV12:
      vc_raw_dec.y_size = vc_raw_dec.width * vc_raw_dec.height;
      vc_raw_dec.uv_size = vc_raw_dec.y_size / 4;
      vc_raw_dec.data_size = vc_raw_dec.width * vc_raw_dec.height * depth / 8;
      break;
  }

  vc_raw_dec.vflip = 0;

  vc_decode_initialized = 1;

  return VCODEC_OK;
}

/* vcodec structs */

VCODEC_FUNCS vc_yv12_funcs = {
  vc_raw_encode_init,
  vc_raw_encode_quit,
  vc_encode_yv12_csp_cap,
  vc_raw_encode,
  vc_raw_encode_print_param,

  vc_raw_decode_init,
  vc_raw_decode_quit,
  vc_decode_yv12_csp_cap,
  vc_raw_decode,
  vc_raw_decode_print_param,

  vc_raw_codec_to_fourcc_str,
  vc_raw_fourcc_str_to_codec,
  vc_raw_codec_cap,
};

VCODEC_FUNCS vc_rgb24_funcs = {
  vc_raw_encode_init,
  vc_raw_encode_quit,
  vc_encode_rgb24_csp_cap,
  vc_raw_encode,
  vc_raw_encode_print_param,

  vc_raw_decode_init,
  vc_raw_decode_quit,
  vc_decode_rgb24_csp_cap,
  vc_raw_decode,
  vc_raw_decode_print_param,

  vc_raw_codec_to_fourcc_str,
  vc_raw_fourcc_str_to_codec,
  vc_raw_codec_cap,
};
