/* resample.c - sampling rate conversion subroutines */

/*
 * Reference:
 * Digital Audio Resampling Home Page
 * <http://www-ccrma.stanford.edu/~jos/resample/>
 * resample-1.7.tar.gz
 * Author:  Julius O. Smith III.
 *
 * COPYING:
 * see LGPL file.
 *
 */

#include "resample.h"

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>

#ifndef TRUE
#define TRUE  1
#endif

#ifndef FALSE
#define FALSE 0
#endif

#ifndef MAX
#define MAX(x,y) ((x)>(y) ?(x):(y))
#endif
#ifndef MIN
#define MIN(x,y) ((x)<(y) ?(x):(y))
#endif

typedef short          HWORD;
typedef unsigned short UHWORD;
typedef int            WORD;
typedef unsigned int   UWORD;


#define MAX_HWORD (32767)
#define MIN_HWORD (-32768)

#ifndef NDEBUG
#define INLINE
#else
#define INLINE inline
#endif

/* Conversion constants */
#define Nhc       8
#define Na        7
#define Np       (Nhc+Na)
#define Npc      (1<<Nhc)
#define Amask    ((1<<Na)-1)
#define Pmask    ((1<<Np)-1)
#define Nh       16
#define Nb       16
#define Nhxn     14
#define Nhg      (Nh-Nhxn)
#define NLpScl   13

/* Description of constants:
 *
 * Npc - is the number of look-up values available for the lowpass filter
 *    between the beginning of its impulse response and the "cutoff time"
 *    of the filter.  The cutoff time is defined as the reciprocal of the
 *    lowpass-filter cut off frequence in Hz.  For example, if the
 *    lowpass filter were a sinc function, Npc would be the index of the
 *    impulse-response lookup-table corresponding to the first zero-
 *    crossing of the sinc function.  (The inverse first zero-crossing
 *    time of a sinc function equals its nominal cutoff frequency in Hz.)
 *    Npc must be a power of 2 due to the details of the current
 *    implementation. The default value of 512 is sufficiently high that
 *    using linear interpolation to fill in between the table entries
 *    gives approximately 16-bit accuracy in filter coefficients.
 *
 * Nhc - is log base 2 of Npc.
 *
 * Na - is the number of bits devoted to linear interpolation of the
 *    filter coefficients.
 *
 * Np - is Na + Nhc, the number of bits to the right of the binary point
 *    in the integer "time" variable. To the left of the point, it indexes
 *    the input array (X), and to the right, it is interpreted as a number
 *    between 0 and 1 sample of the input X.  Np must be less than 16 in
 *    this implementation.
 *
 * Nh - is the number of bits in the filter coefficients. The sum of Nh and
 *    the number of bits in the input data (typically 16) cannot exceed 32.
 *    Thus Nh should be 16.  The largest filter coefficient should nearly
 *    fill 16 bits (32767).
 *
 * Nb - is the number of bits in the input data. The sum of Nb and Nh cannot
 *    exceed 32.
 *
 * Nhxn - is the number of bits to right shift after multiplying each input
 *    sample times a filter coefficient. It can be as great as Nh and as
 *    small as 0. Nhxn = Nh-2 gives 2 guard bits in the multiply-add
 *    accumulation.  If Nhxn=0, the accumulation will soon overflow 32 bits.
 *
 * Nhg - is the number of guard bits in mpy-add accumulation (equal to Nh-Nhxn)
 *
 * NLpScl - is the number of bits allocated to the unity-gain normalization
 *    factor.  The output of the lowpass filter is multiplied by LpScl and
 *    then right-shifted NLpScl bits. To avoid overflow, we must have 
 *    Nb+Nhg+NLpScl < 32.
 */

#include "smallfilter.h"
#include "largefilter.h"
    
#define IBUFFSIZE 4096                         /* Input buffer size */

typedef struct {
  double factor;              /* factor = Sndout/Sndin */
  int    irate;               /* input and output sample rate */
  int    orate;
  int    nChans;              /* number of sound channels (1 or 2) */
  int    inCount;             /* number of input samples to convert */
  int    outCount;            /* number of output samples to compute */
  int    interpFilt;          /* TRUE means interpolate filter coeffs */
  int    largeFilter;         /* TRUE means use 65-tap FIR filter */

  UHWORD LpScl;               /* Unity-gain scale factor */
  UHWORD Nwing;               /* Filter table size */
  UHWORD Nmult;               /* Filter length for up-conversions */
  HWORD  *Imp;                /* Filter coefficients */
  HWORD  *ImpD;               /* ImpD[n] = Imp[n+1]-Imp[n] */

  UWORD  Time, Time2;         /* Current time/pos in input sample */
  UWORD  Xoff;                /* reach reach of LP filter wing + creep room */
  UHWORD Xp;                  /* Current "now"-sample pointer for input */
  UHWORD Xread;               /* Position in input array to read into */
  UHWORD Nx;                  /* # of samples to process each iteration */
  HWORD  *X1, *X2;            /* I buffers */
  int    Xbuf_size;           /* I buffer size */
  HWORD  *Y1, *Y2;            /* O buffers */
  int    Ybuf_size;           /* O buffer size */

} resample_t;

static resample_t resample_param;

#ifndef NDEBUG
static int pof = 0;             /* positive overflow count */
static int nof = 0;             /* negative overflow count */
#endif

static INLINE HWORD WordToHword(WORD v, int scl)
{
    HWORD out;
    WORD llsb = (1<<(scl-1));
    v += llsb;          /* round */
    v >>= scl;
    if (v>MAX_HWORD) {
#ifndef NDEBUG
        if (pof == 0)
          fprintf(stderr, "*** resample: sound sample overflow\n");
        else if ((pof % 10000) == 0)
          fprintf(stderr, "*** resample: another ten thousand overflows\n");
        pof++;
#endif
        v = MAX_HWORD;
    } else if (v < MIN_HWORD) {
#ifndef NDEBUG
        if (nof == 0)
          fprintf(stderr, "*** resample: sound sample (-) overflow\n");
        else if ((nof % 1000) == 0)
          fprintf(stderr, "*** resample: another thousand (-) overflows\n");
        nof++;
#endif
        v = MIN_HWORD;
    }   
    out = (HWORD) v;
    return out;
}


static int err_ret(char *s)
{
    fprintf(stderr,"resample: %s \n\n",s); /* Display error message  */
    return -1;
}

static WORD FilterUp(HWORD Imp[], HWORD ImpD[], 
		     UHWORD Nwing, int  Interp,
		     HWORD *Xp, HWORD Ph, HWORD Inc)
{
    HWORD *Hp, *Hdp = NULL, *End;
    HWORD a = 0;
    WORD v, t;
    
    v=0;
    Hp = &Imp[Ph>>Na];
    End = &Imp[Nwing];
    if (Interp) {
	Hdp = &ImpD[Ph>>Na];
	a = Ph & Amask;
    }
    if (Inc == 1)		/* If doing right wing...              */
    {				/* ...drop extra coeff, so when Ph is  */
	End--;			/*    0.5, we don't do too many mult's */
	if (Ph == 0)		/* If the phase is zero...           */
	{			/* ...then we've already skipped the */
	    Hp += Npc;		/*    first sample, so we must also  */
	    Hdp += Npc;		/*    skip ahead in Imp[] and ImpD[] */
	}
    }
    if (Interp)
      while (Hp < End) {
	  t = *Hp;		/* Get filter coeff */
	  t += (((WORD)*Hdp)*a)>>Na; /* t is now interp'd filter coeff */
	  Hdp += Npc;		/* Filter coeff differences step */
	  t *= *Xp;		/* Mult coeff by input sample */
	  if (t & (1<<(Nhxn-1)))  /* Round, if needed */
	    t += (1<<(Nhxn-1));
	  t >>= Nhxn;		/* Leave some guard bits, but come back some */
	  v += t;			/* The filter output */
	  Hp += Npc;		/* Filter coeff step */
	  Xp += Inc;		/* Input signal step. NO CHECK ON BOUNDS */
      } 
    else 
      while (Hp < End) {
	  t = *Hp;		/* Get filter coeff */
	  t *= *Xp;		/* Mult coeff by input sample */
	  if (t & (1<<(Nhxn-1)))  /* Round, if needed */
	    t += (1<<(Nhxn-1));
	  t >>= Nhxn;		/* Leave some guard bits, but come back some */
	  v += t;			/* The filter output */
	  Hp += Npc;		/* Filter coeff step */
	  Xp += Inc;		/* Input signal step. NO CHECK ON BOUNDS */
      }
    return(v);
}

static WORD FilterUD( HWORD Imp[], HWORD ImpD[],
		     UHWORD Nwing, int  Interp,
		     HWORD *Xp, HWORD Ph, HWORD Inc, UHWORD dhb)
{
    HWORD a;
    HWORD *Hp, *Hdp, *End;
    WORD v, t;
    UWORD Ho;
    
    v=0;
    Ho = (Ph*(UWORD)dhb)>>Np;
    End = &Imp[Nwing];
    if (Inc == 1)		/* If doing right wing...              */
    {				/* ...drop extra coeff, so when Ph is  */
	End--;			/*    0.5, we don't do too many mult's */
	if (Ph == 0)		/* If the phase is zero...           */
	  Ho += dhb;		/* ...then we've already skipped the */
    }				/*    first sample, so we must also  */
				/*    skip ahead in Imp[] and ImpD[] */
    if (Interp)
      while ((Hp = &Imp[Ho>>Na]) < End) {
	  t = *Hp;		/* Get IR sample */
	  Hdp = &ImpD[Ho>>Na];  /* get interp (lower Na) bits from diff table*/
	  a = Ho & Amask;	/* a is logically between 0 and 1 */
	  t += (((WORD)*Hdp)*a)>>Na; /* t is now interp'd filter coeff */
	  t *= *Xp;		/* Mult coeff by input sample */
	  if (t & 1<<(Nhxn-1))	/* Round, if needed */
	    t += 1<<(Nhxn-1);
	  t >>= Nhxn;		/* Leave some guard bits, but come back some */
	  v += t;			/* The filter output */
	  Ho += dhb;		/* IR step */
	  Xp += Inc;		/* Input signal step. NO CHECK ON BOUNDS */
      }
    else 
      while ((Hp = &Imp[Ho>>Na]) < End) {
	  t = *Hp;		/* Get IR sample */
	  t *= *Xp;		/* Mult coeff by input sample */
	  if (t & 1<<(Nhxn-1))	/* Round, if needed */
	    t += 1<<(Nhxn-1);
	  t >>= Nhxn;		/* Leave some guard bits, but come back some */
	  v += t;			/* The filter output */
	  Ho += dhb;		/* IR step */
	  Xp += Inc;		/* Input signal step. NO CHECK ON BOUNDS */
      }
    return(v);
}


/* Sampling rate up-conversion only subroutine;
 * Slightly faster than down-conversion;
 */
static int SrcUp(HWORD X[], HWORD Y[], double factor, UWORD *Time,
                 UHWORD Nx, UHWORD Nwing, UHWORD LpScl,
                 HWORD Imp[], HWORD ImpD[], int  Interp)
{
    HWORD *Xp, *Ystart;
    WORD v;
    
    double dt;                  /* Step through input signal */ 
    UWORD dtb;                  /* Fixed-point version of Dt */
    UWORD endTime;              /* When Time reaches EndTime, return to user */
    
    dt = 1.0/factor;            /* Output sampling period */
    dtb = dt*(1<<Np) + 0.5;     /* Fixed-point representation */
    
    Ystart = Y;
    endTime = *Time + (1<<Np)*(WORD)Nx;
    while (*Time < endTime)
    {
        Xp = &X[*Time>>Np];      /* Ptr to current input sample */
        /* Perform left-wing inner product */
        v = FilterUp(Imp, ImpD, Nwing, Interp, Xp, (HWORD)(*Time&Pmask),-1);
        /* Perform right-wing inner product */
        v += FilterUp(Imp, ImpD, Nwing, Interp, Xp+1, 
		      /* previous (triggers warning): (HWORD)((-*Time)&Pmask),1); */
                      (HWORD)((((*Time)^Pmask)+1)&Pmask),1);
        v >>= Nhg;              /* Make guard bits */
        v *= LpScl;             /* Normalize for unity filter gain */
        *Y++ = WordToHword(v,NLpScl);   /* strip guard bits, deposit output */
        *Time += dtb;           /* Move to next sample by time increment */
    }
    return (Y - Ystart);        /* Return the number of output samples */
}


/* Sampling rate conversion subroutine */

static int SrcUD(HWORD X[], HWORD Y[], double factor, UWORD *Time,
                 UHWORD Nx, UHWORD Nwing, UHWORD LpScl,
                 HWORD Imp[], HWORD ImpD[], int  Interp)
{
    HWORD *Xp, *Ystart;
    WORD v;
    
    double dh;                  /* Step through filter impulse response */
    double dt;                  /* Step through input signal */
    UWORD endTime;              /* When Time reaches EndTime, return to user */
    UWORD dhb, dtb;             /* Fixed-point versions of Dh,Dt */
    
    dt = 1.0/factor;            /* Output sampling period */
    dtb = dt*(1<<Np) + 0.5;     /* Fixed-point representation */
    
    dh = MIN(Npc, factor*Npc);  /* Filter sampling period */
    dhb = dh*(1<<Na) + 0.5;     /* Fixed-point representation */
    
    Ystart = Y;
    endTime = *Time + (1<<Np)*(WORD)Nx;
    while (*Time < endTime)
    {
        Xp = &X[*Time>>Np];     /* Ptr to current input sample */
        v = FilterUD(Imp, ImpD, Nwing, Interp, Xp, (HWORD)(*Time&Pmask),
                     -1, dhb);  /* Perform left-wing inner product */
        v += FilterUD(Imp, ImpD, Nwing, Interp, Xp+1, 
		      /* previous (triggers warning): (HWORD)((-*Time)&Pmask), */
                      (HWORD)((((*Time)^Pmask)+1)&Pmask),
                      1, dhb);  /* Perform right-wing inner product */
        v >>= Nhg;              /* Make guard bits */
        v *= LpScl;             /* Normalize for unity filter gain */
        *Y++ = WordToHword(v,NLpScl);   /* strip guard bits, deposit output */
        *Time += dtb;           /* Move to next sample by time increment */
    }
    return (Y - Ystart);        /* Return the number of output samples */
}

int resample(                   /* number of output samples returned */
    short *isample[],           /* input samples buffer */
    int   isample_num,          /* number of input samples*/
    short *osample[],           /* output samples buffer */
    int   osample_buf_size)     /* size of samples which can permit a buffer */
{
  resample_t *prm = &resample_param;
  UHWORD Nout, Ncreep;
  int i, Ycount;
  int ipos = 0;
  int opos = 0;
  int num_spl;
    
  Ycount = 0;

  if (osample_buf_size < prm->Ybuf_size) {
    return err_ret("output sample buffer too small");
  }

  do {
    num_spl = prm->Xbuf_size - prm->Xread;
    num_spl = (num_spl < (isample_num - ipos)) ? num_spl : isample_num - ipos;
    for (i = 0; i < num_spl; i++)
      prm->X1[prm->Xread+i] = isample[0][ipos+i];
    if (prm->nChans > 1)
      for (i = 0; i < num_spl; i++)
        prm->X2[prm->Xread+i] = isample[1][ipos+i];
    ipos += num_spl;
    prm->Xread += num_spl;

    if (prm->Xread < prm->Xbuf_size) {
      break;
    }

    /* Resample stuff in input buffer */
    prm->Time2 = prm->Time;
    if (prm->factor >= 1) {      /* SrcUp() is faster if we can use it */
      Nout=SrcUp(prm->X1, prm->Y1, prm->factor, &prm->Time, prm->Nx,
	         prm->Nwing, prm->LpScl, prm->Imp, prm->ImpD, prm->interpFilt);
      if (prm->nChans==2)
        Nout=SrcUp(prm->X2,prm->Y2,prm->factor,&prm->Time2,prm->Nx,
	           prm->Nwing,prm->LpScl,prm->Imp,prm->ImpD,prm->interpFilt);
    }
    else {
      Nout=SrcUD(prm->X1, prm->Y1, prm->factor, &prm->Time, prm->Nx,
	         prm->Nwing, prm->LpScl, prm->Imp, prm->ImpD, prm->interpFilt);
      if (prm->nChans==2)
        Nout=SrcUD(prm->X2,prm->Y2,prm->factor,&prm->Time2,prm->Nx,
	           prm->Nwing,prm->LpScl,prm->Imp,prm->ImpD,prm->interpFilt);
    }

    prm->Time -= (prm->Nx<<Np); /* Move converter Nx samples back in time */
    prm->Xp += prm->Nx;         /* Advance by number of samples processed */
    Ncreep = (prm->Time>>Np) - prm->Xoff; /* Calc time accumulation in Time */
    if (Ncreep) {
      prm->Time -= (Ncreep<<Np);    /* Remove time accumulation */
      prm->Xp += Ncreep;            /* and add it to read pointer */
    }
    /* Copy part of input signal */
    for (i=0; (UWORD)i<prm->Xbuf_size-prm->Xp+prm->Xoff; i++) {
      prm->X1[i] = prm->X1[i+prm->Xp-prm->Xoff]; /* that must be re-used */
      if (prm->nChans==2)
        prm->X2[i] = prm->X2[i+prm->Xp-prm->Xoff]; /* that must be re-used */
    }
    prm->Xread = i;              /* Pos in input buff to read new data into */
    prm->Xp = prm->Xoff;

    /* Check to see if output buff overflowed */
    if (opos + Nout > osample_buf_size) {
      return err_ret("Output array overflow");
    }
        
    if (prm->nChans==1) {
      for (i = 0; i < Nout; i++)
        osample[0][i+opos] = prm->Y1[i];
    } else {
      for (i = 0; i < Nout; i++) {
        osample[0][i+opos] = prm->Y1[i];
        osample[1][i+opos] = prm->Y2[i];
      }
    }
    opos += Nout;
    Ycount += Nout;
  } while (ipos < isample_num); /* Continue until done */

  prm->outCount += Ycount;

  return(Ycount);             /* Return # of output samples */
}

int resample_drain(             /* number of output samples returned */
    short *osample[],           /* output samples buffer */
    int   osample_buf_size)     /* size of samples which can permit a buffer */
{
  resample_t *prm = &resample_param;
  UHWORD Nout;
  int i, Ycount;
  int opos = 0;
  int num_spl;
    
  Ycount = 0;

  if (osample_buf_size < prm->Ybuf_size) {
    return err_ret("output sample buffer too small");
  }

  prm->Nx = prm->Xread - prm->Xoff;
  if (prm->Nx <= 0)
    return(Ycount);             /* Return 0 */

  num_spl = prm->Xbuf_size - prm->Xread;
  for (i = 0; i < num_spl; i++)
    prm->X1[prm->Xread+i] = 0;
  if (prm->nChans > 1)
    for (i = 0; i < num_spl; i++)
      prm->X2[prm->Xread+i] = 0;

  /* Resample stuff in input buffer */
  prm->Time2 = prm->Time;
  if (prm->factor >= 1) {      /* SrcUp() is faster if we can use it */
    Nout=SrcUp(prm->X1, prm->Y1, prm->factor, &prm->Time, prm->Nx,
	         prm->Nwing, prm->LpScl, prm->Imp, prm->ImpD, prm->interpFilt);
    if (prm->nChans==2)
      Nout=SrcUp(prm->X2,prm->Y2,prm->factor,&prm->Time2,prm->Nx,
	           prm->Nwing,prm->LpScl,prm->Imp,prm->ImpD,prm->interpFilt);
  }
  else {
    Nout=SrcUD(prm->X1, prm->Y1, prm->factor, &prm->Time, prm->Nx,
	         prm->Nwing, prm->LpScl, prm->Imp, prm->ImpD, prm->interpFilt);
    if (prm->nChans==2)
      Nout=SrcUD(prm->X2,prm->Y2,prm->factor,&prm->Time2,prm->Nx,
	           prm->Nwing,prm->LpScl,prm->Imp,prm->ImpD,prm->interpFilt);
  }

  /* Check to see if output buff overflowed */
  if (opos + Nout > osample_buf_size) {
    return err_ret("Output array overflow, resample_drain");
  }
        
  if (prm->nChans==1) {
    for (i = 0; i < Nout; i++)
      osample[0][i+opos] = prm->Y1[i];
  } else {
    for (i = 0; i < Nout; i++) {
      osample[0][i+opos] = prm->Y1[i];
      osample[1][i+opos] = prm->Y2[i];
    }
  }

  Ycount = Nout;

  prm->outCount += Ycount;

  return(Ycount);             /* Return # of output samples */
}

int resample_init(              /* if ERROR occured, !0 returned. otherwise 0 */
    int    in_rate,             /* input and output sample rate */
    int    out_rate,
    int    nChans,              /* number of sound channels (1 or 2) */
    int    interpFilt,          /* TRUE means interpolate filter coeffs */
    int    largeFilter)         /* TRUE means use 65-tap FIR filter */
{
  static int first = 0;
  resample_t *prm = &resample_param;
  double factor;
  int i;

#ifndef NDEBUG
  /* Check for illegal constants */
  if (Np >= 16)
    return err_ret("Error: Np>=16");
  if (Nb+Nhg+NLpScl >= 32)
    return err_ret("Error: Nb+Nhg+NLpScl>=32");
  if (Nh+Nb > 32)
    return err_ret("Error: Nh+Nb>32");
#endif
    
  if (in_rate < 0 || out_rate < 0) {
    fprintf(stderr, "resample: invalid rate, in_rate %d, out_rate %d\n", in_rate, out_rate);
    return -1;
  }

  if (nChans < 1 || nChans > 2) {
    fprintf(stderr, "resample: invalid channels, nChans %d.\n", nChans);
    return -1;
  }

  factor = (double)out_rate / (double)in_rate;
  if (factor < 0) {
    return err_ret("Error: factor < 0");
  }

  if (!first) {
    memset(prm, 0, sizeof(resample_t));
    first = 1;
  }
  else {
    if (prm->X1) free(prm->X1);
    if (prm->X2) free(prm->X2);
    if (prm->Y1) free(prm->Y1);
    if (prm->Y2) free(prm->Y2);
    memset(prm, 0, sizeof(resample_t));
  }

  prm->nChans = nChans;
  prm->irate = in_rate;
  prm->orate = out_rate;
  prm->interpFilt = interpFilt;
  prm->factor = factor;

  if (largeFilter) {
    prm->Nmult = LARGE_FILTER_NMULT;
    prm->Imp   = LARGE_FILTER_IMP;         /* Impulse response */
    prm->ImpD  = LARGE_FILTER_IMPD;       /* Impulse response deltas */
    prm->LpScl = LARGE_FILTER_SCALE;     /* Unity-gain scale factor */
    prm->Nwing = LARGE_FILTER_NWING;     /* Filter table length */
  } else {
    prm->Nmult = SMALL_FILTER_NMULT;
    prm->Imp   = SMALL_FILTER_IMP;         /* Impulse response */
    prm->ImpD  = SMALL_FILTER_IMPD;       /* Impulse response deltas */
    prm->LpScl = SMALL_FILTER_SCALE;     /* Unity-gain scale factor */
    prm->Nwing = SMALL_FILTER_NWING;     /* Filter table length */
  }

#ifndef NDEBUG
  fprintf(stderr,"Attenuating resampler scale factor by 0.95 "
          "to reduce probability of clipping\n");
#endif
  prm->LpScl *= 0.95;

  /* Account for increased filter gain when using factors less than 1 */
  if (factor < 1)
    prm->LpScl = prm->LpScl*factor + 0.5;

  /* Calc reach of LP filter wing & give some creeping room */
  prm->Xoff = ((prm->Nmult+1)/2.0) * MAX(1.0,1.0/factor) + 10;

  prm->Xbuf_size = IBUFFSIZE;
  if ((UWORD)prm->Xbuf_size < 2*prm->Xoff)      /* Check input buffer size */
    return err_ret("input buffer size (or factor) is too small");
  prm->Ybuf_size = (int)(((double)prm->Xbuf_size)*factor+2.0);
  prm->X1 = (HWORD*)malloc(prm->Xbuf_size * sizeof(HWORD));
  prm->Y1 = (HWORD*)malloc(prm->Ybuf_size * sizeof(HWORD));
  if (!prm->X1 || !prm->Y1) {
    if (prm->X1) free(prm->X1);
    if (prm->Y1) free(prm->Y1);
    prm->X1 = prm->Y1 = NULL;
    return err_ret("cannot allocate memory");
  }
  if (prm->nChans > 1) {
    prm->X2 = (HWORD*)malloc(prm->Xbuf_size * sizeof(HWORD));
    prm->Y2 = (HWORD*)malloc(prm->Ybuf_size * sizeof(HWORD));
    if (!prm->X2 || !prm->Y2) {
      if (prm->X1) free(prm->X1);
      if (prm->X2) free(prm->X2);
      if (prm->Y1) free(prm->Y1);
      if (prm->Y2) free(prm->Y2);
      prm->X1 = prm->X2 = prm->Y1 = prm->Y2 = NULL;
      return err_ret("cannot allocate memory");
    }
  }

  /* # of samples to process each iteration */
  prm->Nx = prm->Xbuf_size - 2*prm->Xoff;
    
  prm->outCount = 0;             /* Current output sample */
  prm->Xp = prm->Xoff;           /* Current "now"-sample pointer for input */
  prm->Xread = prm->Xoff;        /* Position in input array to read into */
  prm->Time = (prm->Xoff<<Np);   /* Current-time pointer for converter */
    
  /* Need Xoff zeros at begining of sample */
  for (i=0; (UWORD)i<prm->Xoff; i++)
    prm->X1[i] = 0;
  if (prm->nChans > 1)
    for (i=0; (UWORD)i<prm->Xoff; i++)
      prm->X2[i] = 0;

  return 0;
}

int resample_get_input_sample_size(void) /* input buffer sample size returned */
{
  resample_t *prm = &resample_param;
  return prm->Xbuf_size - prm->Xoff*2;
}

int resample_get_output_sample_size(void) /* output buffer sample size returned */
{
  resample_t *prm = &resample_param;
  return prm->Ybuf_size;
}

void resample_quit(void)
{
  resample_t *prm = &resample_param;
  if (prm->X1) free(prm->X1);
  if (prm->X2) free(prm->X2);
  if (prm->Y1) free(prm->Y1);
  if (prm->Y2) free(prm->Y2);
  memset(prm, 0, sizeof(resample_t));
}

