/* mdhd.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "mdhd.h"

void
mdhd_init(mdhd_t *mdhd)
{
  atom_init(&mdhd->atom);
  mdhd->atom.type = QTIME_TYPE_MDHD;
  mdhd->version = 0;
  qtime_flags_set(mdhd->flags, 0);
  mdhd->creation_time = qtime_current_time();
  mdhd->modification_time = qtime_current_time();
  mdhd->time_scale = 0;
  mdhd->duration = 0;
  mdhd->language = 0;
  mdhd->quality = 100;
}

void
mdhd_clean(mdhd_t *mdhd)
{
  qtime_error_type_check(QTIME_TYPE_MDHD, mdhd->atom.type)
  mdhd_init(mdhd);
}

mdhd_t*
mdhd_new(void)
{
  mdhd_t *mdhd;
  mdhd = (mdhd_t*)qtime_malloc(sizeof(mdhd_t));
  if (!mdhd) return NULL;
  mdhd_init(mdhd);
  return mdhd;
}

void
mdhd_delete(mdhd_t *mdhd)
{
  qtime_error_type_check(QTIME_TYPE_MDHD, mdhd->atom.type)
  qtime_free(mdhd);
}

mdhd_t*
mdhd_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, mdhd_t *mdhd_ptr)
{
  mdhd_t *mdhd;

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_MDHD, atom_head->type, NULL)
  if (mdhd_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_MDHD, mdhd_ptr->atom.type, NULL)
    mdhd = mdhd_ptr;
    mdhd_clean(mdhd);
  } else {
    if ((mdhd = mdhd_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  mdhd->atom.size = atom_head->size;
  mdhd->atom.type = QTIME_TYPE_MDHD;
  mdhd->atom.parent = atom_head->parent;

  qtime_io_read(qtio, &mdhd->version, 1);
  qtime_io_read(qtio, mdhd->flags, 3);
  qtime_io_read32(qtio, &mdhd->creation_time);
  qtime_io_read32(qtio, &mdhd->modification_time);
  qtime_io_read32(qtio, &mdhd->time_scale);
  qtime_io_read32(qtio, &mdhd->duration);
  qtime_io_read16(qtio, &mdhd->language);
  qtime_io_read16(qtio, &mdhd->quality);

  return mdhd;
}

mdhd_t*
mdhd_create(mdhd_t *mdhd_ptr)
{
  mdhd_t *mdhd;

  if (mdhd_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_MDHD, mdhd_ptr->atom.type, NULL)
    mdhd = mdhd_ptr;
    mdhd_clean(mdhd);
  } else {
    if ((mdhd = mdhd_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }

  return mdhd;
}

int64_t
mdhd_calc_size(mdhd_t *mdhd)
{
  int64_t size = 0;
  qtime_error_type_check_v(QTIME_TYPE_MDHD, mdhd->atom.type, 0)
  size = 4 + 4 + MDHD_PROP_SIZE;
  mdhd->atom.size = size;
  return size;
}

int
mdhd_write_atom(qtime_io_t *qtio, mdhd_t *mdhd)
{
  atom_head_t atom_head;

  qtime_error_type_check_v(QTIME_TYPE_MDHD, mdhd->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  atom_head_init(&atom_head);
  atom_head.size = mdhd->atom.size;
  atom_head.type = mdhd->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_MDHD);
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, &mdhd->version, 1);
  qtime_io_write(qtio, mdhd->flags, 3);
  qtime_io_write32(qtio, &mdhd->creation_time);
  mdhd->modification_time = qtime_current_time();
  qtime_io_write32(qtio, &mdhd->modification_time);
  qtime_io_write32(qtio, &mdhd->time_scale);
  qtime_io_write32(qtio, &mdhd->duration);
  qtime_io_write16(qtio, &mdhd->language);
  qtime_io_write16(qtio, &mdhd->quality);

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_MDHD);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
mdhd_dump(const char *parent_types, mdhd_t *mdhd)
{
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  uint32_t flags;
  char buf[64];

  qtime_type_to_str(mdhd->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  qtime_flags_get(mdhd->flags, &flags);
  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)mdhd->atom.size);
  fprintf(stdout, "%s: version      %d\n", types, mdhd->version);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
  fprintf(stdout, "%s: creation time, %s", types, qtime_ctime_r(mdhd->creation_time,buf));
  fprintf(stdout, "%s: modification time, %s", types, qtime_ctime_r(mdhd->modification_time,buf));
  fprintf(stdout, "%s: time_scale   %d\n", types, mdhd->time_scale);
  fprintf(stdout, "%s: duration     %u\n", types, mdhd->duration);
  fprintf(stdout, "%s: language     %d\n", types, mdhd->language);
  fprintf(stdout, "%s: quality      %d\n", types, mdhd->quality);
}

uint32_t
mdhd_get_time_scale(mdhd_t *mdhd)
{
  return mdhd->time_scale;
}

void
mdhd_set_time_scale(mdhd_t *mdhd, uint32_t time_scale)
{
  mdhd->time_scale = time_scale;
}

uint32_t
mdhd_get_duration(mdhd_t *mdhd)
{
  return mdhd->duration;
}

void
mdhd_set_duration(mdhd_t *mdhd, uint32_t duration)
{
  mdhd->duration = duration;
}

