/* load.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "load.h"

void
load_init(load_t *load)
{
  atom_init(&load->atom);
  load->atom.type = QTIME_TYPE_LOAD;
  load->preload_start_time = 0;
  load->preload_duration = 0;
  load->preload_flags = 0;
  load->default_hints = 0;
}

void
load_clean(load_t *load)
{
#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_LOAD, load->atom.type)) {
    return;
  }
#endif
  load_init(load);
}

load_t*
load_new(void)
{
  load_t *load;
  load = (load_t*)qtime_malloc(sizeof(load_t));
  if (!load)
    return NULL;
  load_init(load);
  return load;
}

void
load_delete(load_t *load)
{
#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_LOAD, load->atom.type)) {
    return;
  }
#endif
  qtime_free(load);
}

load_t*
load_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, load_t *load_ptr)
{
  load_t *load;
#ifdef DEBUG
  if (qtime_type_check(QTIME_TYPE_LOAD, atom_head->type)) {
    atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
    return NULL;
  }
#endif

  if (load_ptr) {
#ifndef NDEBUG
    if (qtime_type_check(QTIME_TYPE_LOAD, load_ptr->atom.type)) {
      atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
      return NULL;
    }
#endif
    load = load_ptr;
    load_clean(load);
  } else {
    if ((load = load_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }

  load->atom.size = atom_head->size;
  load->atom.type = QTIME_TYPE_LOAD;
  load->atom.parent = atom_head->parent;

  qtime_io_read32(qtio, &load->preload_start_time);
  qtime_io_read32(qtio, &load->preload_duration);
  qtime_io_read32(qtio, &load->preload_flags);
  qtime_io_read32(qtio, &load->default_hints);

  return load;
}

load_t*
load_create(load_t *load_ptr)
{
  load_t *load;

  if (load_ptr) {
#ifndef NDEBUG
    if (qtime_type_check(QTIME_TYPE_LOAD, load_ptr->atom.type)) {
      return NULL;
    }
#endif
    load = load_ptr;
    load_clean(load_ptr);
  } else {
    if ((load = load_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return load;
}

int
load_valid(load_t* load)
{
  if (load->preload_start_time > 0)
    return 1;
  if (load->preload_duration > 0)
    return 1;
  if (load->preload_flags != 0)
    return 1;
  if (load->default_hints != 0)
    return 1;
  return 0;
}

int64_t
load_calc_size(load_t *load)
{
  int64_t size = 0;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_LOAD, load->atom.type)) {
    return 0;
  }
#endif
  if (!load_valid(load))
    return 0;
  size = 8 + LOAD_PROP_SIZE;
  load->atom.size = size;
  return size;
}

int
load_write_atom(qtime_io_t *qtio, load_t *load)
{
  atom_head_t atom_head;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_LOAD, load->atom.type)) {
    return QTIME_ERROR_ILLEGAL_ATOM;
  }
#endif

  if (!load_valid(load))
    return QTIME_OK;

  atom_head_init(&atom_head);
  atom_head.size = load->atom.size;
  atom_head.type = load->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    goto fail;
  }

  qtime_io_write32(qtio, &load->preload_start_time);
  qtime_io_write32(qtio, &load->preload_duration);
  qtime_io_write32(qtio, &load->preload_flags);
  qtime_io_write32(qtio, &load->default_hints);

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    goto fail;
  }

  return QTIME_OK;

fail:
  qtime_error_atom_write(QTIME_TYPE_LOAD);
  return QTIME_ERROR_ATOM_WRITE;
}

void
load_dump(const char *parent_types, load_t *load)
{
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];

  qtime_type_to_str(load->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);

  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)load->atom.size);
  fprintf(stdout, "%s: preload start time %u\n", types, load->preload_start_time);
  fprintf(stdout, "%s: preload duration %u\n", types, load->preload_duration);
  fprintf(stdout, "%s: preload flags 0x%x\n", types, load->preload_flags);
  fprintf(stdout, "%s: default hints 0x%x\n", types, load->default_hints);
}

