/* ffmt.c */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
//#define _GNU_SOURCE
#include <string.h>
#include <inttypes.h>
#include <limits.h>

#include "ffmt.h"
#include "util.h"

#include "state.h"
#include "venc.h"
#include "vcodec.h"
#include "acodec.h"
#include "vframe_mgr.h"
#include "aframe_mgr.h"
#include "mgr.h"

#include <avilib.h>

#include <qtime.h>

#define A_BUF_TIME 1.5

#ifdef DEBUG
static void
print_vflag(uint32_t flag)
{
  printf("ffmt: ");
  if (flag & NORMAL_VFRAME)
    printf("NORMAL_VFRAME ");
  if (flag & EXT_VFRAME)
    printf("EXT_VFRAME ");
  if (flag & START_VFRAME)
    printf("START_VFRAME ");
  if (flag & END_VFRAME)
    printf("END_VFRAME ");
  if (flag & QUIT_VFRAME)
    printf("QUIT_VFRAME ");
  if (flag & DROP_VFRAME)
    printf("DROP_VFRAME ");
  printf("!! \n");
}
static void
print_aflag(uint32_t flag)
{
  printf("ffmt: ");
  if (flag & NORMAL_AFRAME)
    printf("NORMAL_AFRAME ");
  if (flag & EXT_AFRAME)
    printf("EXT_AFRAME ");
  if (flag & START_AFRAME)
    printf("START_AFRAME ");
  if (flag & END_AFRAME)
    printf("END_AFRAME ");
  if (flag & QUIT_AFRAME)
    printf("QUIT_AFRAME ");
  if (flag & DROP_AFRAME)
    printf("DROP_AFRAME ");
  if (flag & CHANGE_AFRAME)
    printf("CHANGE_AFRAME ");
  printf("!! \n");
}
#endif /* DEBUG */

static const char *default_file_name = "movie";
static const char *avi_suffix = "avi";
static const char *raw_suffix = "raw";
static const char *mov_suffix = "mov";

static FFMT_CONFIG ffmt_conf;

static int ffmt_initialized = 0;
static int ffmt_thread_running = 0;


double
ffmt_get_bitrate(void)
{
  FFMT_CONFIG *conf = &ffmt_conf;
  double data, time, bitrate;

  pthread_mutex_lock(&conf->mutex);
  data = conf->total_data_size;
  time = conf->record_time;
  pthread_mutex_unlock(&conf->mutex);
  if (time == 0) {
    bitrate = 0;
  } else {
    data = data * 8;
    time = time / 1000000.0;
    bitrate = data / time;
  }
  return bitrate;
}

long long
ffmt_get_total_data_size(void)
{
  FFMT_CONFIG *conf = &ffmt_conf;
  long long size;

  pthread_mutex_lock(&conf->mutex);
  size = conf->total_data_size;
  pthread_mutex_unlock(&conf->mutex);
  return size;
}

int
ffmt_get_video_frame_count(void)
{
  FFMT_CONFIG *conf = &ffmt_conf;
  int count;

  pthread_mutex_lock(&conf->mutex);
  count = conf->video_frame_count;
  pthread_mutex_unlock(&conf->mutex);
  return count;
}

u_time
ffmt_get_record_time(void)
{
  FFMT_CONFIG *conf = &ffmt_conf;
  u_time time;

  pthread_mutex_lock(&conf->mutex);
  time = conf->record_time;
  pthread_mutex_unlock(&conf->mutex);
  return time;
}

static int file_close(FFMT_CONFIG *conf);
static int
file_open(FFMT_CONFIG *conf)
{
  char name[MAX_FILE_NAME_LENGTH+1];
  struct stat st;

  if (conf->file_opend)
    file_close(conf);
//    return FFMT_FAIL;

  if (conf->file_num == 0) {
    strncpy(name, conf->name, MAX_FILE_NAME_LENGTH);
  } else {
    sprintf(name, "%s.%d", conf->name, conf->file_num);
  }
  conf->file_num++;
  while (stat(name, &st) == 0) {
    sprintf(name, "%s.%d", conf->name, conf->file_num);
    conf->file_num++;
  }

  switch (conf->format) {
    case FFMT_FORMAT_AVI:
      if (conf->avifile == NULL) {
        if ((conf->avifile = AVI_open_output_file(name)) == NULL) {
          AVI_print_error("file_open: avi file open error");
          return FFMT_FAIL;
        }
        AVI_set_video(conf->avifile, conf->width, conf->height,
	    conf->fps, (char*)conf->vfourcc_str);
        AVI_set_audio (conf->avifile, conf->channels, conf->rate,
            conf->bitspspl, acodec_codec_to_code(conf->acodec), acodec_encode_get_bitrate());
      }
      break;
    case FFMT_FORMAT_RAW:
      if ((conf->raw_fd = open(name, O_WRONLY|O_CREAT, 0644)) < 0) {
	fprintf(stderr, "file_open: %s open error.\n", name);
	return FFMT_FAIL;
      }
      break;
    case FFMT_FORMAT_MOV:
      if (conf->qt == NULL) {
        if ((conf->qt = qtime_open_write(name)) == NULL) {
          fprintf(stderr, "file_open: mov file open error, %s", name);
          return FFMT_FAIL;
        }
        if ((conf->vqtrk = qtime_create_track(conf->qt)) == NULL) {
          fprintf(stderr, "file_open: video track create error, %s", name);
          return FFMT_FAIL;
        }
        qtime_track_set_video(conf->vqtrk, conf->width, conf->height,
	    conf->fps, (char*)conf->vfourcc_str);
        if ((conf->aqtrk = qtime_create_track(conf->qt)) == NULL) {
          fprintf(stderr, "file_open: audio track create error, %s", name);
          return FFMT_FAIL;
        }
        qtime_track_set_audio(conf->aqtrk, conf->channels, conf->rate,
            conf->bitspspl, (char*)acodec_codec_to_fourcc(conf->acodec));
	if (conf->acodec == ACODEC_MP3)
	  qtime_track_set_audio_ext(conf->aqtrk, 1, -2, acodec_encode_get_frame_size(), 0, 0, (conf->bitspspl/8)*conf->channels);
      }
      break;
    default:
      return FFMT_FAIL;
  }
  pthread_mutex_lock(&conf->mutex);
  conf->record_time = 0;
  conf->total_sample = 0;
  conf->video_frame_count = 0;
  conf->audio_frame_count = 0;
  conf->video_data_size = 0;
  conf->audio_data_size = 0;
  conf->total_data_size = 0;
  conf->video_idx_size = 0;
  conf->audio_idx_size = 0;
  conf->total_idx_size = 0;
  conf->file_opend = 0;
  conf->audio_buf_pos = 0;
  conf->drop_frames = 0;
  pthread_mutex_unlock(&conf->mutex);
  conf->file_opend = 1;

  fprintf(stdout, "\noutput file name: %s\n", name);

  return FFMT_OK;
}

static int
file_close(FFMT_CONFIG *conf)
{
  double fps, rec_time;
  long long file_size;
  const char *unit;
  if (!conf->file_opend)
    return FFMT_OK;

  if (conf->real_time_flag) {
    rec_time = conf->record_time / 1000000.0;
    fps = (double)(conf->video_frame_count) / rec_time;
#ifdef DEBUG
    fprintf(stderr, "file_close: video_frame_count %d, record_time %llu, fps %f\n", conf->video_frame_count, conf->record_time, fps);
#endif /* DEBUG */
  } else {
    fps = conf->fps;
  }

  switch (conf->format) {
    case FFMT_FORMAT_AVI:
      if (conf->audio_buf_pos > 0) {
        AVI_write_audio(conf->avifile, conf->audio_buf, conf->audio_buf_pos);
        pthread_mutex_lock(&conf->mutex);
        conf->audio_frame_count++;
        pthread_mutex_unlock(&conf->mutex);
      }
      AVI_set_video(conf->avifile, conf->width, conf->height,
	    fps, (char*)conf->vfourcc_str);
      AVI_close(conf->avifile);
      conf->avifile = NULL;
      break;
    case FFMT_FORMAT_RAW:
      close(conf->raw_fd);
      conf->raw_fd = -1;
      break;
    case FFMT_FORMAT_MOV:
#if 0
      if (conf->audio_buf_pos > 0) {
        quicktime_write_audio(conf->qtfile, conf->audio_buf, conf->audio_buf_pos, 0);
        pthread_mutex_lock(&conf->mutex);
        conf->audio_frame_count++;
        pthread_mutex_unlock(&conf->mutex);
      }
#endif
      qtime_track_set_video_fps(conf->vqtrk, fps);
      qtime_close(conf->qt);
      conf->qt = NULL;
      conf->vqtrk = NULL;
      conf->aqtrk = NULL;
      break;
    default:
      return FFMT_FAIL;
  }
  fprintf(stdout, "\nfile write:  %s", conf->name);
  if (conf->file_num > 1)
    fprintf(stdout, ".%d\n", conf->file_num-1);
  else
    fprintf(stdout, "\n");
  file_size = conf->total_data_size + conf->total_idx_size;
  if (file_size < 1024)
    unit = "";
  else {
    file_size /= 1024;
    if (file_size < 1024)
      unit = "k";
    else {
      file_size /= 1024;
      unit = "M";
    }
  }
  fprintf(stdout, "file size:  %lld%s bytes\n", file_size, unit);
  fprintf(stdout, "fps: %f\n", fps);

  pthread_mutex_lock(&conf->mutex);
  conf->record_time = 0;
  conf->total_sample = 0;
  conf->video_frame_count = 0;
  conf->audio_frame_count = 0;
  conf->video_data_size = 0;
  conf->audio_data_size = 0;
  conf->total_data_size = 0;
  conf->video_idx_size = 0;
  conf->audio_idx_size = 0;
  conf->total_idx_size = 0;
  conf->file_opend = 0;
  conf->audio_buf_pos = 0;
  conf->drop_frames = 0;
  pthread_mutex_unlock(&conf->mutex);
  return FFMT_OK;
}

static int
write_vframe(FFMT_CONFIG *conf, VFRAME *vfrm)
{
  int ret = 0;
  int is_key_frame = 0;

  if (vfrm->flag & DROP_VFRAME) {
    conf->drop_frames++;
//    fprintf(stderr, "write_vframe: frame %d drop, total %d\n", conf->video_frame_count, conf->drop_frames);
    fprintf(stderr, "frame %d drop, total %d\n", conf->video_frame_count, conf->drop_frames);
  }
#ifdef DEBUG
  if (vfrm->data_length <= 0) {
    fprintf(stderr, "write_vframe: frame write size %d, flag %u\n", vfrm->data_length, vfrm->flag);
    fprintf(stderr, "write_vframe: data %p\n", vfrm->data);
  }
#endif
  if (conf->format == FFMT_FORMAT_AVI) {
    if (vfrm->flag & INTRA_VFRAME)
      is_key_frame = 1;
    else
      is_key_frame = 0;
    ret = AVI_write_frame(conf->avifile,
       	vfrm->data, vfrm->data_length, is_key_frame);
  }
  else if (conf->format == FFMT_FORMAT_RAW) {
    write(conf->raw_fd, vfrm->data, vfrm->data_length);
  }
  else if (conf->format == FFMT_FORMAT_MOV) {
    if (vfrm->flag & INTRA_VFRAME)
      is_key_frame = 1;
    else
      is_key_frame = 0;
    qtime_track_write_video(conf->vqtrk, vfrm->data, vfrm->data_length, is_key_frame);
  }

  pthread_mutex_lock(&conf->mutex);
  conf->video_frame_count++;
  conf->video_data_size += vfrm->data_length;
  conf->total_data_size += vfrm->data_length;
  conf->video_idx_size += 16;
  conf->total_idx_size += 16;
  conf->record_time += vfrm->time;
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

static int
write_aframe(FFMT_CONFIG *conf, AFRAME *afrm)
{
  int ret = 0;

#ifndef NDEBUG
  if (afrm->stream_length <= 0)
    fprintf(stderr, "write_aframe: frame write size %d, flag %x\n", afrm->stream_length, afrm->flag);
#endif
  if (conf->format == FFMT_FORMAT_MOV && afrm->stream_length > 0) {
    qtime_track_write_audio(conf->aqtrk, afrm->stream, afrm->stream_length, afrm->sample_length);

    pthread_mutex_lock(&conf->mutex);
    conf->audio_frame_count++;
    conf->audio_data_size += afrm->stream_length;
    conf->total_data_size += afrm->stream_length;
    conf->total_sample += afrm->sample_length;
    pthread_mutex_unlock(&conf->mutex);
    return afrm->stream_length;
  } else
  if (afrm->stream_length > 0) {
    memcpy(conf->audio_buf + conf->audio_buf_pos,
       	afrm->stream, afrm->stream_length);
    conf->audio_buf_pos += afrm->stream_length;
  }
#ifndef NDEBUG
  else if (afrm->stream_length < 0) {
    fprintf(stderr, "write_aframe: aframe sample_length %d, flag %x\n", afrm->stream_length, afrm->flag);
    exit(1);
  }
#endif
  if (afrm->flag & CHANGE_AFRAME) {
    if (conf->format == FFMT_FORMAT_AVI) {
      ret = AVI_write_audio(conf->avifile,conf->audio_buf,conf->audio_buf_pos);
      conf->audio_buf_pos = 0;
    }
    pthread_mutex_lock(&conf->mutex);
    conf->audio_frame_count++;
    conf->audio_idx_size += 16;
    conf->total_idx_size += 16;
    pthread_mutex_unlock(&conf->mutex);
  } else
    ret = 0;
  pthread_mutex_lock(&conf->mutex);
  conf->audio_data_size += afrm->stream_length;
  conf->total_data_size += afrm->stream_length;
  conf->total_sample += afrm->sample_length;
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

static inline int
get_state(FFMT_CONFIG *conf)
{
  int state;
  pthread_mutex_lock(&conf->mutex);
  state = conf->state;
  pthread_mutex_unlock(&conf->mutex);
  return state;
}

static inline void
state_change_done(FFMT_CONFIG *conf)
{
  pthread_mutex_lock(&conf->mutex);
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
}

static inline void
set_state(FFMT_CONFIG *conf, int state)
{
  pthread_mutex_lock(&conf->mutex);
  conf->state &= ~(STATE_START|STATE_PAUSE|STATE_STOP|STATE_QUIT);
  conf->state |= state;
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
}

VFRAME*
sync_vframe(FFMT_CONFIG *conf, VFRAME *vfrm, uint32_t flag)
{
  while (!vfrm || !(vfrm->flag & flag)) {
    if (!vfrm) {
      while ((vfrm = vframe_dst_frame_get()) == NULL) {
        pthread_mutex_lock(&conf->mutex);
	if (!conf->wait_cb_called && !(conf->state & STATE_ERROR)) {
          conf->state |= STATE_THREAD_WAIT;
          pthread_cond_wait(&conf->thread_wait_cond, &conf->mutex);
	}
	conf->wait_cb_called = 0;
        if (conf->state & STATE_ERROR) {
          pthread_mutex_unlock(&conf->mutex);
	  return NULL;
        }
        pthread_mutex_unlock(&conf->mutex);
      }
    }
#ifdef DEBUG
//    fprintf(stderr, "sync_vframe: ");
//    print_vflag(vfrm->flag);
#endif /* DEBUG */

#ifdef DEBUG
    if (vfrm->flag & ((START_VFRAME|END_VFRAME|QUIT_VFRAME) & ~flag)) {
      fprintf(stderr, "sync_vframe: invalid flag frame, %x.\n", vfrm->flag);
      exit(1);
    }
#endif /* DEBUG */
    if (vfrm->flag & NORMAL_VFRAME) {
#ifdef DEBUG
      if (vfrm->flag & (START_VFRAME|END_VFRAME|QUIT_VFRAME)) {
        fprintf(stderr, "sync_vframe: invalid normal frame, %x.\n", vfrm->flag);
        exit(1);
      }
#endif /* DEBUG */
      write_vframe(conf, vfrm);
      vframe_dst_frame_update(vfrm);
      vfrm = NULL;
    } else {
#ifdef DEBUG
      if (!(vfrm->flag & EXT_VFRAME) ||
	  !(vfrm->flag & (START_VFRAME|END_VFRAME|QUIT_VFRAME|DROP_VFRAME))) {
        fprintf(stderr, "sync_vframe: invalid ext frame, %x.\n", vfrm->flag);
        exit(1);
      }
#endif /* DEBUG */
      if (vfrm->flag & DROP_VFRAME) {
        write_vframe(conf, vfrm);
        vframe_dst_frame_update(vfrm);
        vfrm = NULL;
      }
    }
  }

  return vfrm;
}

AFRAME*
sync_aframe(FFMT_CONFIG *conf, AFRAME *afrm, uint32_t flag)
{
  while (!afrm || !(afrm->flag & flag)) {
    if (!afrm) {
      while ((afrm = aframe_dst_frame_get()) == NULL) {
        pthread_mutex_lock(&conf->mutex);
	if (!conf->wait_cb_called && !(conf->state & STATE_ERROR)) {
          conf->state |= STATE_THREAD_WAIT;
          pthread_cond_wait(&conf->thread_wait_cond, &conf->mutex);
	}
	conf->wait_cb_called = 0;
        if (conf->state & STATE_ERROR) {
          pthread_mutex_unlock(&conf->mutex);
	  return NULL;
        }
        pthread_mutex_unlock(&conf->mutex);
      }
    }
#ifdef DEBUG
//    fprintf(stderr, "sync_aframe: ");
//    print_aflag(afrm->flag);
#endif /* DEBUG */
#ifdef DEBUG
    if (afrm->flag & ((START_AFRAME|END_AFRAME|QUIT_AFRAME) & ~flag)) {
      fprintf(stderr, "sync_aframe: invalid flag frame, %x.\n", afrm->flag);
      exit(1);
    }
#endif /* DEBUG */
    if (afrm->flag & NORMAL_AFRAME) {
#ifdef DEBUG
      if (afrm->flag & (START_AFRAME|END_AFRAME|QUIT_AFRAME)) {
        fprintf(stderr, "sync_aframe: invalid normal frame, %x.\n", afrm->flag);
        exit(1);
      }
#endif /* DEBUG */
      write_aframe(conf, afrm);
      aframe_dst_frame_update(afrm);
      afrm = NULL;
    }
#ifdef DEBUG
    else {
      if (!(afrm->flag & EXT_AFRAME) ||
	  !(afrm->flag & (START_AFRAME|END_AFRAME|QUIT_AFRAME))) {
        fprintf(stderr, "sync_aframe: invalid ext frame, %x.\n", afrm->flag);
        exit(1);
      }
    }
#endif /* DEBUG */
  }

  return afrm;
}

/* thread function */

static void*
ffmt_thread (void *data)
{
  FFMT_CONFIG *conf = &ffmt_conf;
  VFRAME *vfrm = NULL;
  AFRAME *afrm = NULL;
  int quit_flag = 0;
  int restart_flag = 0;

  // avoid compile warnning, unnecessary
  data = NULL;
  // avoid compile warnning, unnecessary

  pthread_mutex_lock(&conf->mutex);
  ffmt_thread_running = 1;
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);

  while (1) {
    int state;

    state = get_state(conf);

    if (quit_flag || state & STATE_ERROR) {
      break;
    }
//    else
//      state_change_done(conf);

    do {
      afrm = aframe_dst_frame_get();
      vfrm = vframe_dst_frame_get();
      if (afrm || vfrm || conf->state & STATE_ERROR)
	break;
      pthread_mutex_lock(&conf->mutex);
      if (!conf->wait_cb_called) {
        conf->state |= STATE_THREAD_WAIT;
        pthread_cond_wait(&conf->thread_wait_cond, &conf->mutex);
      }
      conf->wait_cb_called = 0;
      state = conf->state;
      pthread_mutex_unlock(&conf->mutex);
    } while (!(state & STATE_ERROR));

    if (conf->state & STATE_ERROR)
      continue;

    if (vfrm) {
#ifdef DEBUG
      if (vfrm->flag & NORMAL_VFRAME &&
          vfrm->flag & (START_VFRAME|END_VFRAME|QUIT_VFRAME|DROP_VFRAME)) {
        fprintf(stderr, "ffmt_thread: invalid video frame.\n");
        exit(1);
      } else if (vfrm->flag & EXT_VFRAME &&
          !(vfrm->flag & (START_VFRAME|END_VFRAME|QUIT_VFRAME|DROP_VFRAME))) {
        fprintf(stderr, "ffmt_thread: invalid video frame.\n");
        exit(1);
      }
#endif /* DEBUG */
      if (vfrm->flag & START_VFRAME) {
#ifdef DEBUG
        fprintf(stderr, "ffmt_thread: start video frame.\n");
#endif /* DEBUG */
	afrm = sync_aframe(conf, afrm, START_AFRAME);
	if (!afrm) {
	  set_state(conf, STATE_ERROR);
	  break;
	}
#ifdef DEBUG
	print_aflag(afrm->flag);
#endif /* DEBUG */
#ifdef DEBUG
        if (afrm->flag != (EXT_AFRAME|START_AFRAME)) {
          fprintf(stderr, "ffmt_thread: invalid start aframe.\n");
          exit(1);
        }
#endif /* DEBUG */
	aframe_dst_frame_update(afrm);
	afrm = NULL;
	file_close(conf);
	file_open(conf);
	set_state(conf, STATE_START);
        restart_flag = 0;
      } else if (vfrm->flag & END_VFRAME) {
#ifdef DEBUG
        fprintf(stderr, "ffmt_thread: end video frame.\n");
#endif /* DEBUG */
	afrm = sync_aframe(conf, afrm, END_AFRAME);
	if (!afrm) {
	  set_state(conf, STATE_ERROR);
	  break;
	}
#ifdef DEBUG
	print_aflag(afrm->flag);
#endif /* DEBUG */
#ifdef DEBUG
        if (afrm->flag != (EXT_AFRAME|END_AFRAME)) {
          fprintf(stderr, "ffmt_thread: invalid end aframe.\n");
          exit(1);
        }
#endif /* DEBUG */
	aframe_dst_frame_update(afrm);
	afrm = NULL;
	file_close(conf);
	set_state(conf, STATE_STOP);
      } else if (vfrm->flag & QUIT_VFRAME) {
#ifdef DEBUG
        fprintf(stderr, "ffmt_thread: quit video frame.\n");
#endif /* DEBUG */
	afrm = sync_aframe(conf, afrm, QUIT_AFRAME);
	if (!afrm) {
	  set_state(conf, STATE_ERROR);
	  break;
	}
#ifdef DEBUG
	print_aflag(afrm->flag);
#endif /* DEBUG */
#ifdef DEBUG
        if (afrm->flag != (EXT_AFRAME|QUIT_AFRAME)) {
          fprintf(stderr, "ffmt_thread: invalid quit aframe.\n");
          exit(1);
        }
#endif /* DEBUG */
	file_close(conf);
	aframe_dst_frame_update(afrm);
	afrm = NULL;
	vframe_dst_frame_update(vfrm);
	vfrm = NULL;
	quit_flag = 1;
	set_state(conf, STATE_QUIT);
	break;
      } else {
	write_vframe(conf, vfrm);
      }
      vframe_dst_frame_update(vfrm);
      vfrm = NULL;
    }

    if (afrm) {
#ifdef DEBUG
      if (afrm->flag & NORMAL_AFRAME &&
          afrm->flag & (START_AFRAME|END_AFRAME|QUIT_AFRAME)) {
        fprintf(stderr, "ffmt_thread: invalid audio frame, %x.\n",afrm->flag);
        exit(1);
      } else if (afrm->flag & EXT_AFRAME &&
          !(afrm->flag & (START_AFRAME|END_AFRAME|QUIT_AFRAME))) {
        fprintf(stderr, "ffmt_thread: invalid audio frame, %x.\n",afrm->flag);
        exit(1);
      }
#endif /* DEBUG */
      if (afrm->flag & START_AFRAME) {
#ifdef DEBUG
        fprintf(stderr, "ffmt_thread: start audio frame.\n");
#endif /* DEBUG */
	vfrm = sync_vframe(conf, vfrm, START_VFRAME);
	if (!vfrm) {
	  set_state(conf, STATE_ERROR);
	  break;
	}
#ifdef DEBUG
	print_vflag(vfrm->flag);
#endif /* DEBUG */
#ifdef DEBUG
        if (vfrm->flag != (EXT_VFRAME|START_VFRAME)) {
          fprintf(stderr, "ffmt_thread: invalid start vframe.\n");
          exit(1);
        }
#endif /* DEBUG */
	vframe_dst_frame_update(vfrm);
	vfrm = NULL;
	file_close(conf);
	file_open(conf);
	set_state(conf, STATE_START);
        restart_flag = 0;
      } else if (afrm->flag & END_AFRAME) {
#ifdef DEBUG
        fprintf(stderr, "ffmt_thread: end audio frame.\n");
#endif /* DEBUG */
	vfrm = sync_vframe(conf, vfrm, END_VFRAME);
	if (!vfrm) {
	  set_state(conf, STATE_ERROR);
	  break;
	}
#ifdef DEBUG
	print_vflag(vfrm->flag);
#endif /* DEBUG */
#ifdef DEBUG
        if (vfrm->flag != (EXT_VFRAME|END_VFRAME)) {
          fprintf(stderr, "ffmt_thread: invalid end vframe.\n");
          exit(1);
        }
#endif /* DEBUG */
	vframe_dst_frame_update(vfrm);
	vfrm = NULL;
	file_close(conf);
	set_state(conf, STATE_STOP);
      } else if (afrm->flag & QUIT_AFRAME) {
#ifdef DEBUG
        fprintf(stderr, "ffmt_thread: quit audio frame.\n");
#endif /* DEBUG */
	vfrm = sync_vframe(conf, vfrm, QUIT_VFRAME);
	if (!vfrm) {
	  set_state(conf, STATE_ERROR);
	  break;
	}
#ifdef DEBUG
	print_vflag(vfrm->flag);
#endif /* DEBUG */
#ifdef DEBUG
        if (vfrm->flag != (EXT_VFRAME|QUIT_VFRAME)) {
          fprintf(stderr, "ffmt_thread: invalid quit vframe.\n");
          exit(1);
        }
#endif /* DEBUG */
	file_close(conf);
	vframe_dst_frame_update(vfrm);
	vfrm = NULL;
	aframe_dst_frame_update(afrm);
	afrm = NULL;
	quit_flag = 1;
	set_state(conf, STATE_QUIT);
	break;
      } else {
	write_aframe(conf, afrm);
      }
      aframe_dst_frame_update(afrm);
      afrm = NULL;
    }

    if (conf->format == FFMT_FORMAT_AVI && !restart_flag &&
	conf->total_data_size + conf->total_idx_size > FILE_SIZE_MAX) {
      fprintf(stdout, "file size reach max %d, restart.\n", FILE_SIZE_MAX);
#ifdef DEBUG
      fprintf(stderr, "ffmt_thread: file_size %lld, data_size %lld, idx_size %lld\n", conf->total_data_size + conf->total_idx_size, conf->total_data_size, conf->total_idx_size);
#endif /* DEBUG */
      mgr_set_restart();
      restart_flag = 1;
    }
  }

  file_close(conf);

  pthread_mutex_lock(&conf->mutex);
  ffmt_thread_running = 0;
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
//  state_change_done(conf);

#ifdef DEBUG
  fprintf(stderr, "ffmt_thread: done\n");
#endif /* DEBUG */
  return NULL;
}

/* interface functions */

void
ffmt_dst_wait_cb(void)
{
  FFMT_CONFIG *conf = &ffmt_conf;

  if (!ffmt_initialized)
    return;

  if (pthread_mutex_trylock(&conf->wait_cb_mutex) == EBUSY) {
#ifdef DEBUG
//    fprintf(stderr, "ffmt_dst_wait_cb: return == EBUSY.\n");
#endif /* DEBUG */
    return;
  }

  pthread_mutex_lock(&conf->mutex);
  conf->wait_cb_called = 1;
  if (conf->state & STATE_THREAD_WAIT) {
    conf->state &= ~STATE_THREAD_WAIT;
    pthread_cond_signal(&conf->thread_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);

  pthread_mutex_unlock(&conf->wait_cb_mutex);
}

int
ffmt_state_error(void)
{
  FFMT_CONFIG *conf = &ffmt_conf;
  int ret = FFMT_FAIL;

  if (!ffmt_initialized)
    return FFMT_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!ffmt_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return FFMT_OK;
  }
  if (conf->state & STATE_THREAD_WAIT) {
    conf->state &= ~STATE_THREAD_WAIT;
    pthread_cond_signal(&conf->thread_wait_cond);
  }
  if (conf->state & STATE_ERROR) {
    ret = FFMT_OK;
  } else {
    conf->state &= ~(STATE_START|STATE_PAUSE|STATE_STOP);
    conf->state |= STATE_ERROR;
#if 0
    conf->state |= STATE_INTERFACE_WAIT;
    while (conf->state & STATE_INTERFACE_WAIT)
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
#endif
    ret = FFMT_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
ffmt_state_start(void)
{
  FFMT_CONFIG *conf = &ffmt_conf;
  int ret = FFMT_FAIL;

  if (!ffmt_initialized)
    return FFMT_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!ffmt_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return FFMT_OK;
  }
  if (conf->state & (STATE_START|STATE_QUIT|STATE_ERROR))
    ret = FFMT_OK;
  else {
    while (!(conf->state & STATE_START)) {
      conf->state |= STATE_INTERFACE_WAIT;
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    }
    ret = FFMT_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
ffmt_state_stop(void)
{
  FFMT_CONFIG *conf = &ffmt_conf;
  int ret = FFMT_FAIL;

  if (!ffmt_initialized)
    return FFMT_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!ffmt_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return FFMT_OK;
  }
  if (conf->state & (STATE_STOP|STATE_QUIT|STATE_ERROR))
    ret = FFMT_OK;
  else {
    while (!(conf->state & STATE_STOP)) {
      conf->state |= STATE_INTERFACE_WAIT;
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    }
    ret = FFMT_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
ffmt_state_quit(void)
{
  FFMT_CONFIG *conf = &ffmt_conf;
  int ret = FFMT_FAIL;

  if (!ffmt_initialized)
    return FFMT_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!ffmt_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return FFMT_OK;
  }
  if (conf->state & (STATE_QUIT|STATE_ERROR))
    ret = FFMT_OK;
  else {
    while (!(conf->state & STATE_QUIT)) {
      conf->state |= STATE_INTERFACE_WAIT;
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    }
    ret = FFMT_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
ffmt_state_pause(void)
{
  if (!ffmt_initialized)
    return FFMT_OK;
  return FFMT_OK;
}

int
ffmt_state_toggle_pause(void)
{
  if (!ffmt_initialized)
    return FFMT_OK;
  return FFMT_OK;
}

int
ffmt_error_quit (void)
{
  FFMT_CONFIG *conf = &ffmt_conf;
  int retcode;
  void* retval;

  if (!ffmt_initialized)
    return FFMT_OK;

  ffmt_state_error();

  file_close(conf);

  retcode = pthread_join (conf->ffmt_th, &retval);
  if (retcode != 0)
    fprintf (stderr, "ffmt_quit() thread join error %d\n", retcode);

  free(conf->audio_buf);
  conf->audio_buf = NULL;
  free(conf->name);
  conf->name = NULL;

  ffmt_initialized = 0;
//  fprintf(stderr, "ffmt_quit: done\n");
  return FFMT_OK;
}

int
ffmt_quit (void)
{
  FFMT_CONFIG *conf = &ffmt_conf;
  int retcode;
  void* retval;

  if (!ffmt_initialized)
    return FFMT_OK;

  ffmt_state_quit();

  file_close(conf);

  retcode = pthread_join (conf->ffmt_th, &retval);
  if (retcode != 0)
    fprintf (stderr, "ffmt_quit() thread join error %d\n", retcode);

  free(conf->audio_buf);
  conf->audio_buf = NULL;
  free(conf->name);
  conf->name = NULL;

  ffmt_initialized = 0;
//  fprintf(stderr, "ffmt_quit: done\n");
  return FFMT_OK;
}

int
ffmt_init (char *name, int format, int width, int height, double fps, unsigned long vcodec, int acodec, int rate, int channels, int bitspspl)
{
  FFMT_CONFIG *conf = &ffmt_conf;
  char buf[MAX_FILE_NAME_LENGTH];
  int retcode;
  int a_buf_size;
  int need_config = 0;
static int first = 1;

  if (first) {
    ffmt_initialized = 0;
    ffmt_thread_running = 0;
    conf->file_opend = 0;
    conf->name = NULL;
    conf->audio_buf = NULL;
    conf->avifile = NULL;
    conf->qt = NULL;
    conf->vqtrk = NULL;
    conf->aqtrk = NULL;
    need_config = 1;
    first = 0;
  }

  if (ffmt_initialized) {
    fprintf(stderr, "ffmt_init: ffmt_initialized TRUE !!!\n");
    ffmt_quit();
  }

#if 0
  if (!ffmt_initialized)
    need_config = 1;
  else if (conf->name != name)
    need_config = 1;
  else if (conf->format != format)
#endif

  file_close(conf);

  if (format != FFMT_FORMAT_UNKNOWN)
    conf->format = format;

  if (name == NULL) {
    const char *suff;
    switch (conf->format) {
      case FFMT_FORMAT_AVI:
	suff = avi_suffix;
	break;
      case FFMT_FORMAT_RAW:
	suff = raw_suffix;
	break;
      case FFMT_FORMAT_MOV:
	suff = mov_suffix;
	break;
      default:
	suff = "xxx";
	break;
    }
    sprintf(buf, "%s.%s", default_file_name, suff);
  }
  else {
    strncpy(buf, name, MAX_FILE_NAME_LENGTH-8);
  }
    
  if (conf->name) {
    if (strncmp(conf->name, buf, strlen(conf->name)) != 0) {
      free(conf->name);
      conf->name = NULL;
      conf->name = (char*)strndup(buf, MAX_FILE_NAME_LENGTH-8);
    }
  } else
    conf->name = (char*)strndup(buf, MAX_FILE_NAME_LENGTH-8);

  conf->width = width;
  conf->height = height;
  conf->fps = fps;
  conf->vcodec = vcodec;
  conf->vfourcc_str = vcodec_codec_to_fourcc_str(vcodec);
  conf->real_time_flag = 0;
  if (fps == 0)
    conf->real_time_flag = 1;

  conf->acodec = acodec;
  conf->rate = rate;
  conf->channels = channels;
  conf->bitspspl = bitspspl;

  a_buf_size = (int)((double)conf->rate * A_BUF_TIME + 0.5);
  a_buf_size *= conf->channels;
  a_buf_size *= (conf->bitspspl+7)/8;
  conf->audio_buf = (uint8_t*)mem_malloc(a_buf_size, "ffmt_init");
  conf->audio_buf_size = a_buf_size;
  conf->audio_buf_pos = 0;

  conf->avifile = NULL;
  conf->qt = NULL;
  conf->vqtrk = NULL;
  conf->aqtrk = NULL;
  conf->file_opend = 0;
  conf->video_frame_count = 0;
  conf->record_time = 0;
  conf->total_data_size = 0;
  conf->file_num = 0;
  conf->drop_frames = 0;

  vframe_set_dst_wait_cb(ffmt_dst_wait_cb);
  aframe_set_dst_wait_cb(ffmt_dst_wait_cb);

  conf->wait_cb_called = 0;
  conf->state = STATE_STOP;
  pthread_mutex_init (&conf->mutex, NULL);
  pthread_mutex_init (&conf->wait_cb_mutex, NULL);
  pthread_cond_init (&conf->interface_wait_cond, NULL);
  pthread_cond_init (&conf->thread_wait_cond, NULL);

  retcode = pthread_create (&conf->ffmt_th, NULL, ffmt_thread, NULL);
  if (retcode != 0) {
    fprintf (stderr, "ffmt_init() thread create failed %d\n", retcode);
    free(conf->name);
    conf->name = NULL;
    free(conf->audio_buf);
    conf->audio_buf = NULL;
    return FFMT_FAIL;
  }

  pthread_mutex_lock(&conf->mutex);
  if (!ffmt_thread_running) {
    conf->state |= STATE_INTERFACE_WAIT;
    pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
  }
  pthread_mutex_unlock(&conf->mutex);

  ffmt_initialized = 1;

  return FFMT_OK;
}

void
ffmt_print_param(void)
{
  FFMT_CONFIG *conf = &ffmt_conf;
  const char *fmt = NULL;

  fprintf (stderr, "FILE PARAM\n");
  if (conf->format == FFMT_FORMAT_AVI)
    fmt = "AVI";
  else if (conf->format == FFMT_FORMAT_RAW)
    fmt = "RAW";
  else if (conf->format == FFMT_FORMAT_MOV)
    fmt = "MOV";
  else
    fmt = "UNKNOWN";
  fprintf (stderr, "file format:            %s\n", fmt);
  fprintf (stderr, "file name:              %s\n", conf->name);
}

