# -*- coding: utf-8 -*-
#
#  Copyright (C) 2003-2012 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import logging
import os
import sys
import hashlib
import logging

from gi.repository import Gtk
from gi.repository import Gdk
from gi.repository import GdkPixbuf
import cairo

import numpy


class TransparentWindow(Gtk.Window):
    __gsignals__ = {'screen-changed': 'override', }

    def __init__(self, type=Gtk.WindowType.TOPLEVEL):
        Gtk.Window.__init__(self, type=type)
        self.set_app_paintable(True)
        self.screen_changed()
        self.truncated = False
        self.offset = {'left': 0,
                       'top': 0,
                       'right': 0,
                       'bottom': 0}
        fixed = Gtk.Fixed()
        fixed.show()
        Gtk.Window.add(self, fixed) # XXX
        self.__fixed = fixed
        self.__child = None
        self.__child_size = (-1, -1)
        self.__position = (0, 0)
        self.connect_after('size_allocate', self.size_allocate)

    @property
    def fixed(self): # read only
        return self.__fixed

    def add(self, widget):
        self.fixed.put(widget, 0, 0)
        self.__child = widget

    def remove(self, widget):
        self.fixed.remove(widget)
        self.__child = None

    def destroy(self):
        Gtk.Window.remove(self, self.fixed)
        self.fixed.destroy()
        Gtk.Window.destroy(self)

    def screen_changed(self, old_screen=None):
        screen = self.get_screen()
        if self.is_composited():
            visual = screen.get_rgba_visual()
            self.supports_alpha = True
        else:
            visual = screen.get_system_visual()
            logging.debug('screen does NOT support alpha.\n')
            self.supports_alpha = False
        assert visual is not None
        self.set_visual(visual)

    def size_allocate(self, widget, event):
        new_x, new_y = self.__position
        Gtk.Window.move(self, new_x, new_y)

    def reset_truncate(self):
        self.fixed.move(self.__child, 0, 0)
        self.__child.set_size_request(*self.__child_size)
        self.set_size_request(-1, -1)
        self.offset['left'] = 0
        self.offset['top'] = 0
        self.offset['right'] = 0
        self.offset['bottom'] = 0
        self.truncated = False

    def resize_move(self, x, y, xoffset=0, yoffset=0):
        ### XXX: does not work propery for Gtk.WindowType.POPUP
        ##if self.type == Gtk.WindowType.POPUP:
        ##    Gtk.Window.move(self, x, y)
        ##    return
        w, h = self.__child.get_size_request()
        w -= self.offset['right']
        h -= self.offset['bottom']
        left, top, scrn_w, scrn_h = get_workarea()
        new_x = min(max(x + xoffset, left - w + 1), left + scrn_w - 1)
        new_y = min(max(y + yoffset, top - h + 1), top + scrn_h - 1)
        offset_left = 0
        offset_top = 0
        offset_right = 0
        offset_bottom = 0
        truncate = False
        if new_x < left:
            offset_left = new_x - left
            new_x = left
            truncate = True
        if new_x + w > left + scrn_w:
            offset_right = left + scrn_w - (new_x + w)
            truncate = True
        if new_y < top:
            offset_top = new_y - top
            new_y = top
            truncate = True
        if new_y + h > top + scrn_h:
            offset_bottom = top + scrn_h - (new_y + h)
            truncate = True
        if not truncate:
            if self.truncated:
                self.reset_truncate()
            else:
                Gtk.Window.move(self, new_x, new_y) # XXX
        else:
            if not self.truncated:
                self.__child_size = (w, h)
            self.fixed.move(self.__child, offset_left, offset_top)
            self.__child.set_size_request(w + offset_right,
                                          h + offset_bottom)
            self.set_size_request(w + offset_left + offset_right,
                                  h + offset_top + offset_bottom)
            self.truncated = True
            self.offset['left'] = offset_left
            self.offset['top'] = offset_top
            self.offset['right'] = offset_right
            self.offset['bottom'] = offset_bottom
        self.__position = (new_x, new_y)
        self.__child.queue_draw()

    def update_shape(self, surface):
        region = Gdk.cairo_region_create_from_surface(surface)
        self.get_window().input_shape_combine_region(
            region, self.offset['left'], self.offset['top'])

def get_png_size(path):
    if not path or not os.path.exists(path):
        return 0, 0
    head, tail = os.path.split(path)
    basename, ext = os.path.splitext(tail)
    ext = os.fsdecode(ext).lower()
    if ext == '.dgp':
        buf = get_DGP_IHDR(path)
    elif ext == '.ddp':
        buf = get_DDP_IHDR(path)
    else:
        buf = get_png_IHDR(path)
    assert buf[0:8] == b'\x89PNG\r\n\x1a\n' # png format
    assert buf[12:16] == b'IHDR' # name of the first chunk in a PNG datastream
    w = buf[16:20]
    h = buf[20:24]
    width = (w[0] << 24) + (w[1] << 16) + (w[2] << 8) + w[3]
    height = (h[0] << 24) + (h[1] << 16) + (h[2] << 8) + h[3]
    return width, height

def get_DGP_IHDR(path):
    head, tail = os.path.split(os.fsdecode(path)) # XXX
    filename = tail
    m_half = hashlib.md5(filename[:len(filename) // 2]).hexdigest()
    m_full = hashlib.md5(filename).hexdigest()
    tmp = ''.join((m_full, filename))
    key = ''
    j = 0
    for i in range(len(tmp)):
        value = ord(tmp[i]) ^ ord(m_half[j])
        if not value:
            break
        key = ''.join((key, chr(value)))
        j += 1
        if j >= len(m_half):
            j = 0
    key_length = len(key)
    if key_length == 0: # not encrypted
        logging.warning(''.join((filename, ' generates a null key.')))
        return get_png_IHDR(path)
    key = ''.join((key[1:], key[0]))
    key_pos = 0
    buf = b''
    with open(path, 'rb') as f:
        for i in range(24):
            c = f.read(1)
            buf = b''.join(
                (buf, int.to_bytes(c[0] ^ ord(key[key_pos]), 1, 'little')))
            key_pos += 1
            if key_pos >= key_length:
                key_pos = 0
    return buf

def get_DDP_IHDR(path):
    size = os.path.getsize(path)
    key = size << 2    
    buf = b''
    with open(path, 'rb') as f:
        for i in range(24):
            c = f.read(1)
            key = (key * 0x08088405 + 1) & 0xffffffff
            buf = b''.join(
                (buf, int.to_bytes((c[0] ^ key >> 24) & 0xff, 1, 'little')))
    return buf

def get_png_IHDR(path):
    with open(path, 'rb') as f:
        buf = f.read(24)
    return buf

def __pixbuf_new_from_file(path):
    path = os.fsdecode(path) # XXX
    return GdkPixbuf.Pixbuf.new_from_file(path)

def __surface_new_from_file(path):
    path = os.fsdecode(path) # XXX
    return cairo.ImageSurface.create_from_png(path)

def create_icon_pixbuf(path):
    path = os.fsdecode(path) # XXX
    try:
        pixbuf = __pixbuf_new_from_file(path)
    except: # compressed icons are not supported. :-(
        pixbuf = None
    else:
        pixbuf = pixbuf.scale_simple(16, 16, GdkPixbuf.InterpType.BILINEAR)
    return pixbuf

def create_blank_surface(width, height): ## FIXME
    surface = cairo.ImageSurface(cairo.FORMAT_ARGB32, width, height)
    return surface

def create_pixbuf_from_DGP_file(path):
    head, tail = os.path.split(os.fsdeocde(path)) # XXX
    filename = tail
    m_half = hashlib.md5(filename[:len(filename) // 2]).hexdigest()
    m_full = hashlib.md5(filename).hexdigest()
    tmp = ''.join((m_full, filename))
    key = ''
    j = 0
    for i in range(len(tmp)):
        value = ord(tmp[i]) ^ ord(m_half[j])
        if not value:
            break
        key = ''.join((key, chr(value)))
        j += 1
        if j >= len(m_half):
            j = 0
    key_length = len(key)
    if key_length == 0: # not encrypted
        logging.warning(''.join((filename, ' generates a null key.')))
        pixbuf = __pixbuf_new_from_file(filename)
        return pixbuf
    key = ''.join((key[1:], key[0]))
    key_pos = 0
    loader = GdkPixbuf.PixbufLoader.new_with_type('png')
    with open(path, 'rb') as f:
        while 1:
            c = f.read(1)
            if c == b'':
                break
            loader.write(int.to_bytes(c[0] ^ ord(key[key_pos]), 1, 'little'))
            key_pos += 1
            if key_pos >= key_length:
                key_pos = 0
    pixbuf = loader.get_pixbuf()
    loader.close()
    return pixbuf

def create_pixbuf_from_DDP_file(path):
    with open(path, 'rb') as f:
        buf = f.read()
    key = len(buf) << 2
    loader = GdkPixbuf.PixbufLoader.new_with_type('png')
    for i in range(len(buf)):
        key = (key * 0x08088405 + 1) & 0xffffffff
        loader.write(int.to_bytes((buf[i] ^ key >> 24) & 0xff, 1, 'little'))
    pixbuf = loader.get_pixbuf()
    loader.close()
    return pixbuf

def create_surface_from_file(path, is_pnr=True, use_pna=False):
    head, tail = os.path.split(path)
    basename, ext = os.path.splitext(tail)
    pixbuf = create_pixbuf_from_file(path, is_pnr, use_pna)
    surface = cairo.ImageSurface(cairo.FORMAT_ARGB32,
                                 pixbuf.get_width(), pixbuf.get_height())
    cr = cairo.Context(surface)
    Gdk.cairo_set_source_pixbuf(cr, pixbuf, 0, 0)
    cr.set_operator(cairo.OPERATOR_SOURCE)
    cr.paint()
    del cr
    if is_pnr:
        buf = surface.get_data()
        ar = numpy.frombuffer(buf, numpy.uint32)
        argb = ar[0]
        ar[ar == argb] = 0x00000000
    if use_pna:
        path = os.path.join(head, b''.join((basename, b'.pna')))
        if os.path.exists(path):
            pna_surface = __surface_new_from_file(path)
            pna_buf = pna_surface.get_data()
            buf = surface.get_data()
            step = cairo.ImageSurface.format_stride_for_width(
                pna_surface.get_format(), 1)
            pos = 0
            for index in range(0, len(buf), 4):
                if sys.byteorder == 'little':
                    buf[index + 3] = pna_buf[pos]
                else:
                    buf[index + 0] = pna_buf[pos]
                pos += step
    return surface

def create_pixbuf_from_file(path, is_pnr=True, use_pna=False):
    head, tail = os.path.split(path)
    basename, ext = os.path.splitext(tail)
    ext = os.fsdecode(ext).lower()
    if ext == '.dgp':
        pixbuf = create_pixbuf_from_DGP_file(path)
    elif ext == '.ddp':
        pixbuf = create_pixbuf_from_DDP_file(path)
    else:
        pixbuf = __pixbuf_new_from_file(path)
    return pixbuf

def get_workarea():
    scrn = Gdk.Screen.get_default()
    root = scrn.get_root_window()
    left, top, width, height = root.get_geometry()
    return left, top, width, height
