using System;
using System.Collections;
using System.Collections.Generic;
using System.Data;
using NDac.Keys;
using NDac.Modules.DataTransferObjects;
using NDac.Modules.Entities;

namespace NDac.Modules
{
	/// <summary>
	/// テーブルモジュールの列挙子を表します。(エンティティを列挙します。)
	/// </summary>
	class TableModuleEnumerator : TableModuleEnumerator< IKey, IEntity, IDataTransferObject >
	{
		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="module">テーブルモジュール</param>
		public TableModuleEnumerator( TableModule module )
			: base( ( TableModule< IKey, IEntity, IDataTransferObject > )module, FilteringCondition.Default )
		{
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="module">テーブルモジュール</param>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		public TableModuleEnumerator( TableModule module, FilteringCondition filteringCondition )
			: base( ( TableModule< IKey, IEntity, IDataTransferObject > )module, filteringCondition )
		{
		}

		/// <summary>
		/// デストラクタ
		/// </summary>
		~TableModuleEnumerator()
		{
			base.Dispose();
		}
	}

	/// <summary>
	/// テーブルモジュールの列挙子を表します。(エンティティを列挙します。)
	/// </summary>
	/// <typeparam name="TKey">キーの型</typeparam>
	/// <typeparam name="TEntity">エンティティの型</typeparam>
	/// <typeparam name="TDataTransferObject">データ変換オブジェクトの型</typeparam>
	class TableModuleEnumerator< TKey, TEntity, TDataTransferObject > : IEnumerator< TEntity >
		where TKey : class, IKey
		where TEntity : class, IEntity
		where TDataTransferObject : class, IDataTransferObject
	{
		private TableModule< TKey, TEntity, TDataTransferObject >	_module;
		private int					_currentIndex;
		private DataRow				_currentRow;
		private bool				_isDisposed;
		private FilteringCondition	_filteringCondition;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="module">テーブルモジュール</param>
		public TableModuleEnumerator( TableModule< TKey, TEntity, TDataTransferObject > module ) : this( module, FilteringCondition.Default )
		{
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="module">テーブルモジュール</param>
		/// <param name="filteringCondition">エンティティのフィルタ処理条件</param>
		public TableModuleEnumerator( TableModule< TKey, TEntity, TDataTransferObject > module, FilteringCondition filteringCondition )
		{
			this._module = module;

			this._isDisposed = false;

			this._filteringCondition = filteringCondition;

			this.Reset();
		}

		/// <summary>
		/// デストラクタ
		/// </summary>
		~TableModuleEnumerator()
		{
			this.Dispose();
		}

		#region IEnumerator<TEntity> メンバ

		/// <summary>
		/// 現在の要素を表します。
		/// </summary>
		public TEntity Current
		{
			get
			{
				return( this._module.ToEntity( this._currentRow ) );
			}
		}

		#endregion

		#region IDisposable メンバ

		/// <summary>
		/// リソースの解放を行います。
		/// </summary>
		public void Dispose()
		{
			if( !this._isDisposed )
			{
				this._module = null;

				this._isDisposed = true;
			}
		}

		#endregion

		#region IEnumerator メンバ

		/// <summary>
		/// 現在の要素を表します。
		/// </summary>
		object IEnumerator.Current
		{
			get
			{
				return( this.Current );
			}
		}

		/// <summary>
		/// 次の要素に進めます。
		/// </summary>
		/// <returns>コレクションの末尾を過ぎた場合はfalseが返ります。</returns>
		public bool MoveNext()
		{
			while( true )
			{
				if( this._currentIndex < this._module.Table.Rows.Count )
				{
					DataRow row = this._module.Table.Rows[ this._currentIndex ];

					this._currentIndex++;

					if( this._filteringCondition.CanConvert( row ) )
					{
						this._currentRow = row;

						break;
					}
				}
				else
				{
					this._currentRow = null;

					break;
				}
			}

			return( this._currentRow != null ? true : false );
		}

		/// <summary>
		/// 最初の要素に戻します。
		/// </summary>
		public void Reset()
		{
			this._currentIndex = 0;

			this._currentRow = null;

			foreach( DataRow row in this._module.Table.Rows )
			{
				if( this._filteringCondition.CanConvert( row ) )
				{
					this._currentRow = row;

					break;
				}

				this._currentIndex++;
			}
		}

		#endregion
	}
}
