using System;
using System.Data;

namespace NDac.Modules.Extensions
{
	/// <summary>
	/// 削除された行を管理しているテーブルのマネージャを表します。
	/// </summary>
	class DeletedRowTableManager : IDisposable
	{
		private DataTable	_table;
		private DataTable	_deletedTable;
		private bool		_isDisposed;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="table">データテーブル</param>
		public DeletedRowTableManager( DataTable table )
		{
			this._table = table;

			this._deletedTable = this._table.Clone();

			this._isDisposed = false;

			this._table.RowDeleting += new DataRowChangeEventHandler( _table_RowDeleting );

			this._table.RowChanged += new DataRowChangeEventHandler( _table_RowChanged );
		}

		/// <summary>
		/// デストラクタ
		/// </summary>
		~DeletedRowTableManager()
		{
			this.Dispose();
		}

		/// <summary>
		/// 行が削除される直前のイベント処理を表します。
		/// </summary>
		/// <param name="sender">イベント送信者</param>
		/// <param name="e">パラメータ</param>
		private void _table_RowDeleting( object sender, DataRowChangeEventArgs e )
		{
			this._deletedTable.ImportRow( e.Row );
		}

		/// <summary>
		/// 行が変更された後のイベント処理を表します。
		/// </summary>
		/// <param name="sender">イベント送信者</param>
		/// <param name="e">パラメータ</param>
		private void _table_RowChanged( object sender, DataRowChangeEventArgs e )
		{
			if( ( 0 < this._deletedTable.Rows.Count ) && ( this._table.GetChanges( DataRowState.Deleted ) == null ) )
			{
				this._deletedTable.Clear();
			}
		}

		/// <summary>
		/// 削除された行を管理しているテーブルを表します。
		/// </summary>
		public DataTable Table
		{
			get
			{
				return( this._deletedTable );
			}
		}

		#region IDisposable メンバ

		/// <summary>
		/// リソースの解放を行います。
		/// </summary>
		public void Dispose()
		{
			if( !this._isDisposed )
			{
				this._table = null;

				this._deletedTable.Dispose();

				this._deletedTable = null;

				this._isDisposed = true;
			}
		}

		#endregion
	}
}
