using System;
using System.Data;
using System.Reflection;
using NDac.Datas;
using NDac.Modules.Entities;

namespace NDac.Modules.DataTransferObjects
{
	/// <summary>
	/// エンティティのアダプターを表します。
	/// </summary>
	/// <typeparam name="TEntity">エンティティの型</typeparam>
	public class EntityAdapter< TEntity > : EntityAdapter< TEntity, IDataTransferObject >
		where TEntity : IEntity
	{
		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="entity">エンティティ</param>
		/// <param name="fillMethod">データを充填するメソッド</param>
		public EntityAdapter( TEntity entity, FillDelegate fillMethod )
			: base( entity, fillMethod )
		{
		}
	}

	/// <summary>
	/// エンティティのアダプターを表します。
	/// </summary>
	/// <typeparam name="TEntity">エンティティの型</typeparam>
	/// <typeparam name="TDataTransferObject">データ変換オブジェクトの型</typeparam>
	public class EntityAdapter< TEntity, TDataTransferObject >
		where TEntity : IEntity
		where TDataTransferObject : IDataTransferObject
	{
		public delegate void FillDelegate( TEntity entity );
		[ NonSerialized ] private TEntity _entity;
		[ NonSerialized ] private FillDelegate _fillMethod;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="entity">エンティティ</param>
		/// <param name="fillMethod">データを充填するメソッド</param>
		public EntityAdapter( TEntity entity, FillDelegate fillMethod )
		{
			this._entity = entity;

			this._fillMethod = fillMethod;
		}

		/// <summary>
		/// エンティティを表します。
		/// </summary>
		public TEntity Entity
		{
			get
			{
				return( this._entity );
			}
			set
			{
				this._entity = value;
			}
		}

		/// <summary>
		/// エンティティにデータを充填します。
		/// </summary>
		public void Fill()
		{
			this._fillMethod( this._entity );
		}

		/// <summary>
		/// データセットホルダーを取得します。
		/// </summary>
		/// <returns></returns>
		public DataSetHolder GetHolder()
		{
			return( this._entity.Row.Table.DataSet.GetHolder() );
		}

		/// <summary>
		/// ハッシュコードを取得します。
		/// </summary>
		/// <returns>ハッシュコード</returns>
		public override int GetHashCode()
		{
			if( this._entity == null )
			{
				return( 0 );
			}
			else
			{
				return( this._entity.GetHashCode() );
			}
		}

		/// <summary>
		/// エンティティと等価か判定します。
		/// </summary>
		/// <returns>エンティティと等価の場合trueが返ります。</returns>
		public bool IsEquals( TDataTransferObject dto )
		{
			bool isEquals = false;

			if( this._entity == null )
			{
				return( false );
			}

			foreach( PropertyInfo dtoInfo in this.GetType().GetProperties() )
			{
				PropertyInfo entityInfo = this._entity.GetType().GetProperty( dtoInfo.Name );

				if( entityInfo != null )
				{
					object dtoValue		= dtoInfo.GetValue( this, null );

					object entityValue	= entityInfo.GetValue( this._entity, null );

					if( dtoValue == entityValue )
					{
						isEquals = true;
					}
					else
					{
						isEquals = false;

						break;
					}
				}
			}

			return( isEquals );
		}
	}
}
