using System;
using NDac.Datas;
using NDac.Gateways;
using NDac.Keys;
using NDac.Modules;
using NDac.Modules.Attributes;
using NDac.Modules.Entities;

namespace NDac.Modules.DataTransferObjects
{
	/// <summary>
	/// Authorテーブルのデータ変換オブジェクトを表します。
	/// </summary>
	[ Serializable ]
	public partial class AuthorDTO : DataTransferObject< AuthorKey, AuthorEntity >
	{
		private long _id;
		private string _name;
		private int _sex;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		public AuthorDTO()
			: base()
		{
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="entity">エンティティ</param>
		public AuthorDTO( AuthorEntity entity )
			: base( entity )
		{
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		public AuthorDTO( long primaryKey )
		{
			this._id = primaryKey;
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		public AuthorDTO( AuthorKey primaryKey )
		{
			this._id = primaryKey.Id;
		}

		/// <summary>
		/// Idを表します。
		/// </summary>
		[ PrimaryKey( "Id", typeof( long ) ) ]
		public long Id
		{
			get
			{
				return( this._id );
			}
			set
			{
				this._id = value;
			}
		}

		/// <summary>
		/// Nameを表します。
		/// </summary>
		public string Name
		{
			get
			{
				return( StringCutDowner.Variable( this._name, 100 ) );
			}
			set
			{
				this._name = StringCutDowner.Variable( value, 100 );
			}
		}

		/// <summary>
		/// Sexを表します。
		/// </summary>
		public int Sex
		{
			get
			{
				return( this._sex );
			}
			set
			{
				this._sex = value;
			}
		}

		/// <summary>
		/// データ変換オブジェクトの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のデータ変換オブジェクト</param>
		protected override void Copy( DataTransferObject< AuthorKey, AuthorEntity > source )
		{
			this.Copy( ( AuthorDTO )source );
		}

		/// <summary>
		/// データ変換オブジェクトの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のデータ変換オブジェクト</param>
		public void Copy( AuthorDTO source )
		{
			this.Id = source.Id;
			this.Name = source.Name;
			this.Sex = source.Sex;
		}

		/// <summary>
		/// エンティティの内容をコピーします。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		public override void Copy( AuthorEntity source )
		{
			this.FillValue( source );
		}

		/// <summary>
		/// データ変換オブジェクトの内容をエンティティにコピーします。
		/// </summary>
		/// <param name="source">コピー元のデータ変換オブジェクト</param>
		/// <param name="destination">コピー先のエンティティ</param>
		public static void Copy( AuthorDTO source, AuthorEntity destination )
		{
			source.FillTo( destination );
		}

		/// <summary>
		/// エンティティの内容をデータ変換オブジェクトにコピーします。
		/// </summary>
		/// <param name="source">コピー元のエンティティ</param>
		/// <param name="destination">コピー先のデータ変換オブジェクト</param>
		public static void Copy( AuthorEntity source, AuthorDTO destination )
		{
			destination.FillValue( source );
		}

		/// <summary>
		/// エンティティより値を充填します。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		protected override void FillValue( IEntity entity )
		{
			this.FillValue( ( AuthorEntity )entity );
		}

		/// <summary>
		/// エンティティより値を充填します。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		protected override void FillValue( AuthorEntity entity )
		{
			this._id = entity.Id;
			this._name = entity.Name;
			this._sex = entity.Sex;
		}

		/// <summary>
		/// データ変換オブジェクトの内容をエンティティに反映させます。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		protected override void FillTo( IEntity entity )
		{
			this.FillTo( ( AuthorEntity )entity );
		}

		/// <summary>
		/// データ変換オブジェクトの内容をエンティティに反映させます。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		protected override void FillTo( AuthorEntity entity )
		{
			entity.Id = this._id;
			entity.Name = this._name;
			entity.Sex = this._sex;
		}

		/// <summary>
		/// プライマリキーでデータを充填されたテーブルモジュールを取得します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>テーブルモジュール</returns>
		protected override ITableModule LoadModule( IKey primaryKey )
		{
			return( this.LoadModule( ( AuthorKey )primaryKey ) );
		}

		/// <summary>
		/// プライマリキーでデータを充填されたテーブルモジュールを取得します。
		/// </summary>
		/// <param name="primaryKey">プライマリキー</param>
		/// <returns>テーブルモジュール</returns>
		protected override ITableModule LoadModule( AuthorKey primaryKey )
		{
			DataSetHolder holder = new DataSetHolder();

			AuthorGateway gateway = new AuthorGateway( holder );

			AuthorModule module = new AuthorModule( holder );

			gateway.LoadBy( primaryKey );

			return( module );
		}

		/// <summary>
		/// プライマリキーに変換します。
		/// </summary>
		/// <returns>プライマリキー</returns>
		protected override AuthorKey ToPrimaryKey()
		{
			return( new AuthorKey( this.Id ) );
		}
	}

	/// <summary>
	/// Authorテーブルエンティティの拡張メソッド定義クラスを表します。
	/// </summary>
	public static class AuthorEntityExtension
	{
		/// <summary>
		/// データ変換オブジェクトの内容をコピーします。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		/// <param name="source">コピー元のデータ変換オブジェクト</param>
		public static void Copy( this AuthorEntity entity, AuthorDTO source )
		{
			AuthorDTO.Copy( source, entity );
		}

		/// <summary>
		/// データ変換オブジェクトに変換します。
		/// </summary>
		/// <param name="entity">エンティティ</param>
		/// <returns>データ変換オブジェクト</returns>
		public static AuthorDTO ToDataTransferObject( this AuthorEntity entity )
		{
			return( new AuthorDTO( entity ) );
		}
	}
}
