/* $Id: ProcessInspector.java,v 1.2 2007/12/11 09:54:58 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfd.inspector;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.border.Border;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.AbstractTableModel;

import jp.co.argo21.nautica.commons.swing.ByteLimitedDocument;
import jp.co.argo21.nautica.commons.swing.DialogUtilities;
import jp.co.argo21.nautica.commons.swing.NumericDocument;
import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.commons.util.StringUtils;
import jp.co.argo21.nautica.tool.wfd.IconView;
import jp.co.argo21.nautica.tool.wfd.InspectorDialog;
import jp.co.argo21.nautica.tool.wfd.WorkflowDesignerConstants;
import jp.co.argo21.nautica.tool.wfd.WorkflowDesignerManager;
import jp.co.argo21.nautica.tool.wfd.element.Element;
import jp.co.argo21.nautica.tool.wfd.element.ParameterElement;
import jp.co.argo21.nautica.tool.wfd.element.ProcessElement;

/**
 * プロセス用インスペクタ
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class ProcessInspector extends AbstractElementInspector
{
	/** シリアルバージョンID */
	private static final long serialVersionUID = 1L;

	/** テーブルの列名 */
	private static String[] columnNames;
	/** 通知者テーブルの列名 */
	private static String rspColumnName;

	/** 単位 */
	public static String[] DURATION_UNITS;
	/** アクセスレベル */
	private static String[] accessLevels;
	/** 枠線 */
	protected static final Border border = BorderFactory.createEtchedBorder();

	/** アイコン表示エリア */
	private IconView iconView;
	/** ID表示エリア */
	private JLabel idField;
	/** 名称表示エリア */
	private JTextField nameField;
	/** 実行期限エリア */
	private JTextField limitField;
	/** 実行期限単位 */
	private JComboBox unitBox;
	/** アクセスレベル表示エリア */
	private JLabel accessField;
	/** 備考表示エリア */
	private JTextField descriptionField;

	/** パラメータリスト */
	private JTable paramTable;
	/** パラメータ追加ボタン */
	private JButton addParamButton;
	/** パラメータ編集ボタン */
	private JButton editParamButton;
	/** パラメータ削除ボタン */
	private JButton delParamButton;

	/** 通知者リスト */
	private JTable rspTable;
	/** 通知者追加ボタン */
	private JButton addRspButton;
	/** 通知者更新ボタン */
	private JButton updRspButton;
	/** 通知者削除ボタン */
	private JButton delRspButton;
	/** 通知者入力エリア */
	private JTextField rspField;
	
	/** 通知者一覧 */
	private List<String> responsibles = new ArrayList<String>();
	
	/**
	 * 定数の初期化
	 */
	static
	{
		ResourceManager rm = WorkflowDesignerManager.getResourceManager();

		DURATION_UNITS = new String[6];
		DURATION_UNITS[0] = rm.getResource("ProcessInspector.unit.0");
		DURATION_UNITS[1] = rm.getResource("ProcessInspector.unit.1");
		DURATION_UNITS[2] = rm.getResource("ProcessInspector.unit.2");
		DURATION_UNITS[3] = rm.getResource("ProcessInspector.unit.3");
		DURATION_UNITS[4] = rm.getResource("ProcessInspector.unit.4");
		DURATION_UNITS[5] = rm.getResource("ProcessInspector.unit.5");
		
		accessLevels = new String[2];
		accessLevels[0] = rm.getResource("ProcessInspector.access.0");
		accessLevels[1] = rm.getResource("ProcessInspector.access.1");

		columnNames = new String[3];
		columnNames[0] = rm.getResource("ParamTableModel.columnName.0");
		columnNames[1] = rm.getResource("ParamTableModel.columnName.1");
		columnNames[2] = rm.getResource("ParamTableModel.columnName.2");

		rspColumnName = rm.getResource("ResponsibleTableModel.columnName.0");
	}

	/**
	 * インスペクタを生成する。
	 */
	public ProcessInspector()
	{
		ResourceManager rm = WorkflowDesignerManager.getResourceManager();

		JPanel panel0 = createBasicInfoPanel(rm);
		addTab(rm.getResource("ProcessInspector.tab.0"), panel0);

		JPanel panel1 = createParamInfoPanel(rm);
		addTab(rm.getResource("ProcessInspector.tab.1"), panel1);

		JPanel panel2 = createResponsibleInfoPanel(rm);
		addTab(rm.getResource("ProcessInspector.tab.2"), panel2);

		validate();
	}
	
	/**
	 * 基本情報のパネルを生成する。
	 * 
	 * @param rm リソースマネージャ
	 * @return パネル
	 */
	private JPanel createBasicInfoPanel(ResourceManager rm)
	{
		JPanel panel = new JPanel();
		panel.setLayout(null);
		panel.setBorder(border);
		Dimension size = new Dimension(380, 170);
		panel.setSize(size);
		panel.setPreferredSize(size);
		JLabel l0 = new JLabel(rm.getResource("ProcessInspector.label.0"));
		JLabel l1 = new JLabel(rm.getResource("ProcessInspector.label.1"));
		JLabel l2 = new JLabel(rm.getResource("ProcessInspector.label.2"));
		JLabel l3 = new JLabel(rm.getResource("ProcessInspector.label.3"));
		JLabel l4 = new JLabel(rm.getResource("ProcessInspector.label.4"));

		iconView = new IconView();
		iconView.setIconName(WorkflowDesignerConstants.EL_PROCESS);
		idField = new JLabel();
		idField.setBorder(border);
		nameField = new JTextField();
		nameField.setDocument(new ByteLimitedDocument(64));
		descriptionField = new JTextField();
		descriptionField.setDocument(new ByteLimitedDocument(128));
		limitField = new JTextField();
		limitField.setHorizontalAlignment(SwingConstants.RIGHT);
		limitField.setDocument(new NumericDocument(3, true, true));
		unitBox = new JComboBox(DURATION_UNITS);
		accessField = new JLabel();
		accessField.setBorder(border);

		panel.add(locate(iconView,          10,  10,  60, 60));
		panel.add(locate(l0,                80,  10, 100, 25));
		panel.add(locate(l1,                80,  40, 100, 25));
		panel.add(locate(l2,                80,  70, 100, 25));
		panel.add(locate(l3,                80, 100, 100, 25));
		panel.add(locate(l4,                80, 130, 100, 25));
		panel.add(locate(idField,          180,  10, 190, 25));
		panel.add(locate(nameField,        180,  40, 190, 25));
		panel.add(locate(limitField,       180,  70, 110, 25));
		panel.add(locate(unitBox,          290,  70,  80, 25));
		panel.add(locate(accessField,      180, 100, 190, 25));
		panel.add(locate(descriptionField, 180, 130, 190, 25));

		validate();
		
		return panel;
	}
	
	/**
	 * パラメータ情報のパネルを生成する。
	 * 
	 * @param rm リソースマネージャ
	 * @return パネル
	 */
	private JPanel createParamInfoPanel(ResourceManager rm)
	{
		JPanel panel = new JPanel();
		panel.setLayout(null);
		panel.setBorder(border);
		Dimension size = new Dimension(380, 200);
		panel.setSize(size);
		panel.setPreferredSize(size);
		JLabel l0 = new JLabel(rm.getResource("ProcessInspector.label.5"));

		paramTable = new JTable(new ParamTableModel());
		paramTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		JScrollPane sp = new JScrollPane();
		sp.setViewportView(paramTable);
		addParamButton = new JButton(rm.getResource("JButton.label.add"));
		addParamButton.addActionListener(new AddParamAction());
		editParamButton = new JButton(rm.getResource("JButton.label.edit"));
		editParamButton.addActionListener(new EditParamAction());
		delParamButton = new JButton(rm.getResource("JButton.label.delete"));
		delParamButton.addActionListener(new DeleteParamAction());

		panel.add(locate(l0,                10,  10, 100,  25));
		panel.add(locate(sp,                10,  40, 270, 145));
		panel.add(locate(addParamButton,   290,  40,  80,  25));
		panel.add(locate(editParamButton,  290,  70,  80,  25));
		panel.add(locate(delParamButton,   290, 100,  80,  25));

		validate();
		
		return panel;
	}
	
	/**
	 * 通知者情報のパネルを生成する。
	 * 
	 * @param rm リソースマネージャ
	 * @return パネル
	 */
	private JPanel createResponsibleInfoPanel(ResourceManager rm)
	{
		JPanel panel = new JPanel();
		panel.setLayout(null);
		panel.setBorder(border);
		Dimension size = new Dimension(380, 200);
		panel.setSize(size);
		panel.setPreferredSize(size);
		JLabel l0 = new JLabel(rm.getResource("ProcessInspector.label.5"));

		rspField = new JTextField();
		rspField.setDocument(new ByteLimitedDocument(64));
		rspTable = new JTable(new ResponsibleTableModel());
		rspTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		ListSelectionModel lsm = rspTable.getSelectionModel();
		lsm.addListSelectionListener(new ResponsibleSelectAction());
		JScrollPane sp = new JScrollPane();
		sp.setViewportView(rspTable);
		addRspButton = new JButton(rm.getResource("JButton.label.add"));
		addRspButton.addActionListener(new AddResponsibleAction());
		updRspButton = new JButton(rm.getResource("JButton.label.update"));
		updRspButton.addActionListener(new EditResponsibleAction());
		delRspButton = new JButton(rm.getResource("JButton.label.delete"));
		delRspButton.addActionListener(new DeleteResponsibleAction());

		panel.add(locate(l0,            10,  10, 100,  25));
		panel.add(locate(sp,            10,  40, 270, 115));
		panel.add(locate(addRspButton, 290,  40,  80,  25));
		panel.add(locate(updRspButton, 290,  70,  80,  25));
		panel.add(locate(delRspButton, 290, 100,  80,  25));
		panel.add(locate(rspField,      10, 160, 270,  25));

		validate();
		
		return panel;
	}

	/**
	 * インスペクタを初期化する。
	 * 
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#init()
	 */
	public void init()
	{
		super.init();

		nameField.setText("");
		descriptionField.setText("");
		limitField.setText("0");
		unitBox.setSelectedIndex(2);
		responsibles.clear();
		rspField.setText("");
		paramTable.clearSelection();
		rspTable.clearSelection();
	}

	/**
	 * インスペクト対象を設定する。
	 * 
	 * @param elem	対象要素
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#setInspectable(jp.co.argo21.nautica.tool.wfd.element.Element)
	 */	
	public void setInspectable(Element elem)
	{
		super.setInspectable(elem);
		
		if (elem != null) {
			if (elem instanceof ProcessElement) {
				ProcessElement proc = (ProcessElement)elem;
				idField.setText(proc.getID());
				nameField.setText(proc.getName());
				limitField.setText("" + proc.getLimit());
				int u = proc.getDurationUnit();
				if (u == ProcessElement.YEAR) {
					unitBox.setSelectedIndex(0);
				} else if (u == ProcessElement.MONTH) {
					unitBox.setSelectedIndex(1);
				} else if (u == ProcessElement.DAY) {
					unitBox.setSelectedIndex(2);
				} else if (u == ProcessElement.HOUR) {
					unitBox.setSelectedIndex(3);
				} else if (u == ProcessElement.MINUTE) {
					unitBox.setSelectedIndex(4);
				} else if (u == ProcessElement.SECOND) {
					unitBox.setSelectedIndex(5);
				} else {
					unitBox.setSelectedIndex(2);
				}
				int a = proc.getAccessLevel();
				if (a == ProcessElement.PUBLIC) {
					accessField.setText(accessLevels[0]);
				} else {
					accessField.setText(accessLevels[1]);
				}
				descriptionField.setText(proc.getDescription());
				
				for (String r: proc.getResponsibles()) {
					responsibles.add(r);
				}
			}
		}
		setSelectedIndex(0);
	}

	/**
	 * 設定値をチェックする。
	 * 
	 * @return	チェック結果
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#check()
	 */	
	public boolean check()
	{
		return true;
	}

	/**
	 * 設定値をコミットする。
	 * 
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#commit()
	 */	
	public void commit()
	{
		Element e = getInspectable();
		if (e != null) {
			if (e instanceof ProcessElement) {
				ProcessElement proc = (ProcessElement)e;
				proc.setName(nameField.getText());
				
				try {
					int limit = Integer.parseInt(limitField.getText());
					proc.setLimit(limit);
				} catch (Exception ex) {
					proc.setLimit(0);
				}
				int u = unitBox.getSelectedIndex();
				if (u == 0) {
					proc.setDurationUnit(ProcessElement.YEAR);
				} else if (u == 1) {
					proc.setDurationUnit(ProcessElement.MONTH);
				} else if (u == 2) {
					proc.setDurationUnit(ProcessElement.DAY);
				} else if (u == 3) {
					proc.setDurationUnit(ProcessElement.HOUR);
				} else if (u == 4) {
					proc.setDurationUnit(ProcessElement.MINUTE);
				} else if (u == 5) {
					proc.setDurationUnit(ProcessElement.SECOND);
				}
				proc.setDescription(descriptionField.getText());
				
				proc.getResponsibles().clear();
				for (String r : responsibles) {
					proc.addResponsible(r);
				}

				fireElementChanged();
			}
		}		
	}

	/**
	 * 指定されたコンポーネントの属性を指定の大きさと座標で更新する。
	 * 
	 * @param component 対象コンポーネント
	 * @param x 左上のX座標
	 * @param y 左上のY座標
	 * @param w 幅
	 * @param h 高さ
	 * @return 変更されたコンポーネント
	 */
	protected Component locate(Component component, int x, int y, int w, int h)
	{
		component.setLocation(x, y);
		component.setSize(w, h);
		return  component;
	}

	/**
	 * パラメータ追加処理。
	 */	
	class AddParamAction extends AbstractAction
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * パラメータを追加する。
		 * 
		 * @param e	イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			Element newElem = InspectorDialog.showDialog(ProcessInspector.this, WorkflowDesignerConstants.EL_PARAMETER);
			if (newElem == null) return;
			
			Element proc = getInspectable();
			if (proc == null) return;

			((ProcessElement)proc).addParameter((ParameterElement)newElem);
			paramTable.clearSelection();
			paramTable.updateUI();
		}
	}
	
	/**
	 * パラメータ編集処理
	 */	
	class EditParamAction extends AbstractAction
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * パラメータを編集する。
		 * 
		 * @param e	イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			ResourceManager rm = WorkflowDesignerManager.getResourceManager();
			int selected = paramTable.getSelectedRow();
			if (selected < 0) {
				DialogUtilities.showWarning(rm.getResource("message.warn.0002"));
				return;
			}
			Element proc = getInspectable();
			if (proc == null) return;

			List<ParameterElement> list = ((ProcessElement)proc).getParameters();
			ParameterElement p = list.get(selected);
			InspectorDialog.showDialog(ProcessInspector.this, p);
			paramTable.clearSelection();
			paramTable.updateUI();
		}
	}
	
	/**
	 * パラメータ削除処理。
	 */	
	class DeleteParamAction extends AbstractAction
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * パラメータを削除する。
		 * 
		 * @param e	イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			ResourceManager rm = WorkflowDesignerManager.getResourceManager();
			int selected = paramTable.getSelectedRow();
			if (selected < 0) {
				DialogUtilities.showWarning(rm.getResource("message.warn.0002"));
				return;
			}
			Element proc = getInspectable();
			if (proc == null) return;

			List<ParameterElement> list = ((ProcessElement)proc).getParameters();
			@SuppressWarnings("unused")
			ParameterElement p = list.remove(selected);
			paramTable.clearSelection();
			paramTable.updateUI();
		}
	}
	
	/**
	 * パラメータテーブル制御モデル
	 */
	class ParamTableModel extends AbstractTableModel
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * パラメータテーブル制御モデルを生成する。
		 */
		ParamTableModel()
		{
		}

		/**
		 * データソースオブジェクトが管理する列数を返す。
		 * JTableはこのメソッドを使って、初期化時に作成および表示する
		 * 必要のある列数を判断する。
		 *
		 * @return	モデルの列数
		 */
		public int getColumnCount()
		{
			return 3;
		}

		/**
		 * データソースオブジェクトが管理するレコード数を返す。
		 * JTableはこのメソッドを使って、作成および表示する必要のある行数を判断する。
		 *
		 * @return	モデルの行数
		 */
		public int getRowCount()
		{
			Element e = getInspectable();
			if (e == null) return 0;
			
			return ((ProcessElement)e).getParameters().size();
		}

		/**
		 * colにある列の名前を返す。
		 * このメソッドは、テーブルの列ヘッダ名を初期化するために使われる。
		 *
		 * @param	col		列のインデックス
		 * @return	列ヘッダ名
		 */
		public String getColumnName(int col)
		{
			return columnNames[col];
		}

		/**
		 * セルの値を返す。
		 *
		 * @param	row		値が参照される行
		 * @param	col		値が参照される列
		 * @return	指定されたセルにある値
		 */
		public Object getValueAt(int row, int col)
		{
			Element e = getInspectable();
			if (e == null) return "";

			List<ParameterElement> list = ((ProcessElement)e).getParameters();
			ParameterElement p = list.get(row);
			if (col == 0) {
				int m = p.getMode();
				if (m == ParameterElement.IN) {
					return ParameterInspector.PARAM_MODES[0];
				} else if (m == ParameterElement.OUT) {
					return ParameterInspector.PARAM_MODES[1];
				} else if (m == ParameterElement.INOUT) {
					return ParameterInspector.PARAM_MODES[2];
				}
			} else if (col == 1) {
				int t = p.getType();
				if (t == ParameterElement.STRING) {
					return ParameterInspector.PARAM_TYPES[0];
				} else if (t == ParameterElement.INTEGER) {
					return ParameterInspector.PARAM_TYPES[1];
				} else if (t == ParameterElement.BOOLEAN) {
					return ParameterInspector.PARAM_TYPES[2];
				}
			} else if (col == 2) {
				return p.getDescription();
			}
			return "";
		}

		/**
		 * 列のClassを返す。
		 *
		 * @param	col		列
		 * @return	モデルのオブジェクト値の共通の上位クラス
		 */
		public Class<?> getColumnClass(int col)
		{
			return String.class;
		}

		/**
		 * rowとcolにあるセルが編集可能な場合はtrueを返す。
		 * そうでない場合、セルでsetValueAt()を呼び出しても、そのセルの値は変更されない。
		 *
		 * @param	row		値が参照される行
		 * @param	col		値が参照される列
		 * @return	セルが編集可能な場合はtrue
		 */
		public boolean isCellEditable(int row, int col)
		{
			return false;
		}
	}
	
	/**
	 * 通知者追加処理
	 */	
	class AddResponsibleAction extends AbstractAction
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * 通知者を追加する。
		 * 
		 * @param e	イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			ResourceManager rm = WorkflowDesignerManager.getResourceManager();
			String addr = rspField.getText();
			if (StringUtils.checkNull(addr)) {
				DialogUtilities.showError(rm.getResource("message.error.0026"));
			} else if (responsibles.contains(addr)) {
				DialogUtilities.showError(rm.getResource("message.error.0027"));
			} else {
				responsibles.add(addr);
			}
			rspTable.clearSelection();
			rspTable.updateUI();
		}
	}
	
	/**
	 * 通知者編集処理
	 */	
	class EditResponsibleAction extends AbstractAction
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * 通知者を編集する。
		 * 
		 * @param e	イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			ResourceManager rm = WorkflowDesignerManager.getResourceManager();
			int selected = rspTable.getSelectedRow();
			if (selected < 0) {
				DialogUtilities.showWarning(rm.getResource("message.warn.0003"));
				return;
			}
			String addr = rspField.getText();
			if (StringUtils.checkNull(addr)) {
				DialogUtilities.showError(rm.getResource("message.error.0026"));
			} else if (responsibles.contains(addr)) {
				if (addr.equals(responsibles.get(selected)) == false) { 
					DialogUtilities.showError(rm.getResource("message.error.0027"));
				}
			} else {
				responsibles.set(selected, addr);
			}
			responsibles.set(selected, addr);
			rspTable.clearSelection();
			rspTable.updateUI();
		}
	}
	
	/**
	 * 通知者削除処理
	 */	
	class DeleteResponsibleAction extends AbstractAction
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * 通知者を削除する。
		 * 
		 * @param e	イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			ResourceManager rm = WorkflowDesignerManager.getResourceManager();
			int selected = rspTable.getSelectedRow();
			if (selected < 0) {
				DialogUtilities.showWarning(rm.getResource("message.warn.0003"));
				return;
			}
			responsibles.remove(selected);
			rspField.setText("");
			rspTable.clearSelection();
			rspTable.updateUI();
		}
	}

	/**
	 * 通知者選択処理
	 */	
	class ResponsibleSelectAction implements ListSelectionListener
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * 選択された通知者を表示する。
		 * 
		 * @param e	イベント
		 * @see javax.swing.event.ListSelectionListener#valueChanged(javax.swing.event.ListSelectionEvent)
		 */
		public void valueChanged(ListSelectionEvent e)
		{
			if (e.getValueIsAdjusting()) return;

			ListSelectionModel lsm = (ListSelectionModel)e.getSource();
			if (lsm.isSelectionEmpty() == false) {
				int selected = lsm.getMinSelectionIndex();
				rspField.setText((String)responsibles.get(selected));
			}
		}
	}
		
	/**
	 * 通知者テーブル制御モデル
	 */
	class ResponsibleTableModel extends AbstractTableModel
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * 通知者テーブル制御モデルを生成する。
		 */
		ResponsibleTableModel()
		{
		}

		/**
		 * データソースオブジェクトが管理する列数を返す。
		 * JTableはこのメソッドを使って、初期化時に作成および表示する
		 * 必要のある列数を判断する。
		 *
		 * @return	モデルの列数
		 */
		public int getColumnCount()
		{
			return 1;
		}

		/**
		 * データソースオブジェクトが管理するレコード数を返す。
		 * JTableはこのメソッドを使って、作成および表示する必要のある行数を判断する。
		 *
		 * @return	モデルの行数
		 */
		public int getRowCount()
		{
			return responsibles.size();
		}

		/**
		 * colにある列の名前を返す。
		 * このメソッドは、テーブルの列ヘッダ名を初期化するために使われる。
		 *
		 * @param	col		列のインデックス
		 * @return	列ヘッダ名
		 */
		public String getColumnName(int col)
		{
			return rspColumnName;
		}

		/**
		 * セルの値を返す。
		 *
		 * @param	row		値が参照される行
		 * @param	col		値が参照される列
		 * @return	指定されたセルにある値
		 */
		public Object getValueAt(int row, int col)
		{
			return responsibles.get(row);
		}

		/**
		 * 列のClassを返す。
		 *
		 * @param	col		列
		 * @return	モデルのオブジェクト値の共通の上位クラス
		 */
		public Class<?> getColumnClass(int col)
		{
			return String.class;
		}

		/**
		 * rowとcolにあるセルが編集可能な場合はtrueを返す。
		 * そうでない場合、セルでsetValueAt()を呼び出しても、そのセルの値は変更されない。
		 *
		 * @param	row		値が参照される行
		 * @param	col		値が参照される列
		 * @return	セルが編集可能な場合はtrue
		 */
		public boolean isCellEditable(int row, int col)
		{
			return false;
		}
	}
}
