/* $Id: BasicActivityInspector.java,v 1.1 2007/10/19 08:19:34 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfd.inspector;

import java.awt.Component;
import java.awt.Dimension;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.border.Border;

import jp.co.argo21.nautica.commons.swing.ByteLimitedDocument;
import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.tool.wfd.IconView;
import jp.co.argo21.nautica.tool.wfd.WorkflowDesignerManager;
import jp.co.argo21.nautica.tool.wfd.element.ActivityElement;
import jp.co.argo21.nautica.tool.wfd.element.Element;

/**
 * すべてのアクティビティに共通するインスペクタ
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class BasicActivityInspector extends JTabbedPane implements Inspector
{
	/** シリアルバージョンID */
	private static final long serialVersionUID = 1L;

	/** 枠線 */
	protected static final Border border = BorderFactory.createEtchedBorder();

	/** アイコン表示エリア */
	private IconView iconView;
	/** ID表示エリア */
	private JLabel idField;
	/** 名称表示エリア */
	private JTextField nameField;
	/** 備考表示エリア */
	private JTextField descriptionField;
	/** 論理X座標表示エリア */
	private JLabel xField;
	/** 論理Y座標表示エリア */
	private JLabel yField;

	/** 対象要素 */
	private Element inspectable;
	
	/**
	 * 共通アクティビティインスペクタを生成する。
	 */
	public BasicActivityInspector()
	{
		ResourceManager rm = WorkflowDesignerManager.getResourceManager();

		JPanel panel = createBasicInfoPanel(rm);
		addTab(rm.getResource("BasicActivityInspector.tab.0"), panel);

		validate();
	}
	
	/**
	 * 基本情報のパネルを生成する。
	 * 
	 * @param rm リソースマネージャ
	 * @return パネル
	 */
	private JPanel createBasicInfoPanel(ResourceManager rm)
	{
		JPanel panel = new JPanel();
		panel.setLayout(null);
		panel.setBorder(border);
		Dimension size = new Dimension(380, 135);
		panel.setSize(size);
		panel.setPreferredSize(size);
		JLabel l0 = new JLabel(rm.getResource("BasicActivityInspector.label.0"));
		JLabel l1 = new JLabel(rm.getResource("BasicActivityInspector.label.1"));
		JLabel l2 = new JLabel(rm.getResource("BasicActivityInspector.label.2"));
		JLabel l3 = new JLabel(rm.getResource("BasicActivityInspector.label.3"));
		JLabel l4 = new JLabel(rm.getResource("BasicActivityInspector.label.4"));

		iconView = new IconView();
		idField = new JLabel();
		idField.setBorder(border);
		nameField = new JTextField();
		nameField.setDocument(new ByteLimitedDocument(64));
		descriptionField = new JTextField();
		descriptionField.setDocument(new ByteLimitedDocument(128));
		xField = new JLabel();
		xField.setHorizontalAlignment(SwingConstants.RIGHT);
		xField.setBorder(border);
		yField = new JLabel();
		yField.setHorizontalAlignment(SwingConstants.RIGHT);
		yField.setBorder(border);

		panel.add(locate(iconView,          10,  10,  60, 60));
		panel.add(locate(l0,                80,  10, 100, 25));
		panel.add(locate(l1,                80,  40, 100, 25));
		panel.add(locate(l2,                80,  70, 100, 25));
		panel.add(locate(l3,               230,  70, 100, 25));
		panel.add(locate(l4,                80, 100, 100, 25));
		panel.add(locate(idField,          180,  10, 190, 25));
		panel.add(locate(nameField,        180,  40, 190, 25));
		panel.add(locate(xField,           180,  70,  40, 25));
		panel.add(locate(yField,           330,  70,  40, 25));
		panel.add(locate(descriptionField, 180, 100, 190, 25));

		validate();
		
		return panel;
	}

	/**
	 * インスペクト対象を返す。
	 * 
	 * @return	インスペクト対象
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#getInspectable()
	 */
	public Element getInspectable()
	{
		return this.inspectable;
	}

	/**
	 * インスペクト対象を設定する。
	 * 
	 * @param elem	対象要素
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#setInspectable(jp.co.argo21.nautica.tool.wfd.element.Element)
	 */	
	public void setInspectable(Element elem)
	{
		this.inspectable = elem;

		if (elem != null) {
			if (elem instanceof ActivityElement) {
				ActivityElement act = (ActivityElement)elem;
				iconView.setIconName(act.getActivityState().getIconName());
				idField.setText(act.getID());
				nameField.setText(act.getName());
				xField.setText("" + (act.getLogicalX() + 1));
				yField.setText("" + (act.getLogicalY() + 1));
				descriptionField.setText(act.getDescription());
			}
		}
		setSelectedIndex(0);
	}

	/**
	 * インスペクタが自動コミットするかどうかを返す。
	 * 
	 * @return	自動コミットの場合にはtrue
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#isAutoCommit()
	 */	
	public boolean isAutoCommit()
	{
		return false;
	}

	/**
	 * 設定値をチェックする。
	 * 
	 * @return	チェック結果
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#check()
	 */	
	public boolean check()
	{
		return true;
	}

	/**
	 * 設定値をコミットする。
	 * 
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#commit()
	 */	
	public void commit()
	{
		Element e = getInspectable();
		if (e != null) {
			if (e instanceof ActivityElement) {
				ActivityElement act = (ActivityElement)e;
				act.setName(nameField.getText());
				act.setDescription(descriptionField.getText());
			}
		}		
	}

	/**
	 * インスペクタを初期化する。
	 * 
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#init()
	 */
	public void init()
	{
		nameField.setText("");
		descriptionField.setText("");
	}

	/**
	 * インスペクタのアンロードを行う。
	 * 
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#unload()
	 */
	public void unload()
	{
		//NOP
	}

	/**
	 * 指定されたコンポーネントの属性を指定の大きさと座標で更新する。
	 * 
	 * @param component 対象コンポーネント
	 * @param x 左上のX座標
	 * @param y 左上のY座標
	 * @param w 幅
	 * @param h 高さ
	 * @return 変更されたコンポーネント
	 */
	protected Component locate(Component component, int x, int y, int w, int h)
	{
		component.setLocation(x, y);
		component.setSize(w, h);
		return  component;
	}
}
