/* $Id: ResizingCorner.java,v 1.1 2007/10/19 08:19:34 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfd;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Polygon;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.SwingConstants;
import javax.swing.border.BevelBorder;
import javax.swing.border.Border;

/**
 * コーナーのリサイズボタンを実現。
 * 大きさに応じてボタンイメージを自動調整する。
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class ResizingCorner extends JComponent
{
	/** シリアルバージョンID */
	private static final long serialVersionUID = 1L;

	/** 上向き矢印を示す定数 */
	public static final int TOP    = SwingConstants.TOP;
	/** 左向き矢印を示す定数 */
	public static final int LEFT   = SwingConstants.LEFT;
	/** 右向き矢印を示す定数 */
	public static final int RIGHT  = SwingConstants.RIGHT;
	/** 下向き矢印を示す定数 */
	public static final int BOTTOM = SwingConstants.BOTTOM;

	/** 押下時の枠線 */
	private static final Border L_BORDER
		= BorderFactory.createBevelBorder(BevelBorder.LOWERED, Color.CYAN, Color.BLACK);
	/** 通常時の枠線 */
	private static final Border R_BORDER
		= BorderFactory.createBevelBorder(BevelBorder.RAISED, Color.CYAN, Color.BLACK);

	/** リスナー一覧管理用 */
	private List<ActionListener> listeners = new ArrayList<ActionListener>();
	/** 矢印方向 */
	private int orientation = RIGHT;
	/** 押されているかどうかを表すフラグ */
	private boolean pushing;

	/**
	 * リサイズボタンを生成する。
	 * 
	 * @param orientation 矢印方向
	 */
	public ResizingCorner(int orientation)
	{
		this.orientation = orientation;
		addMouseListener(
			new MouseAdapter() {
				public void mouseClicked(MouseEvent e) {
					ActionEvent event = new ActionEvent(e.getSource(), e.getID(), e.paramString());
					int count = listeners.size();
					for (int i = 0; i < count; i++) {
						ActionListener l = (ActionListener)listeners.get(i);
						l.actionPerformed(event);
					}
				}
				public void mousePressed(MouseEvent e) {
					pushing = true;
					repaint();
				}
				public void mouseReleased(MouseEvent e) {
					pushing = false;
					repaint();
				}
				public void mouseExited(MouseEvent e) {
					pushing = false;
					repaint();
				}
			}
		);
	}

	/**
	 * アイコン表示エリアを描画する。
	 *
	 * @param g 描画対象グラフィックス
	 */
	public void paintComponent(Graphics g)
	{
		Dimension size = getSize();

		g.setColor(Color.BLUE);
		g.fillRect(0, 0, size.width, size.height);

		Border border;
		if (pushing) border = L_BORDER;
		else border = R_BORDER;
		border.paintBorder(this, g, 0, 0, size.width, size.height);

		Polygon polygon = null;
		int t = 3;
		int m = size.height / 2;
		int b = size.height - 4;
		int l = 3;
		int c = size.width / 2;
		int r = size.width - 4;
		
		if (orientation == TOP) {
			int[] x = {l, c, r, l};
			int[] y = {b, t, b, b};
			polygon = new Polygon(x, y, 4);
		} else if (orientation == LEFT) {
			int[] x = {r, l, r, r};
			int[] y = {t, m, b, t};
			polygon = new Polygon(x, y, 4);
		} else if (orientation == RIGHT) {
			int[] x = {l, r, l, l};
			int[] y = {t, m, b, t};
			polygon = new Polygon(x, y, 4);
		} else if (orientation == BOTTOM) {
			int[] x = {l, c, r, l};
			int[] y = {t, b, t, t};
			polygon = new Polygon(x, y, 4);
		}

		if (polygon != null) {
			g.setColor(Color.YELLOW);
			g.fillPolygon(polygon);
			g.setColor(Color.GREEN);
			g.drawPolygon(polygon);
		} 

		g.dispose();			
	}

	/**
	 * アクションリスナーを追加する。
	 * 
	 * @param l リスナー
	 */
	public void addActionListener(ActionListener l)
	{
		if (listeners.contains(l) == false) listeners.add(l);
	}

	/**
	 * アクションリスナーを削除する。
	 * 
	 * @param l リスナー
	 */
	public void removeActionListener(ActionListener l)
	{
		if (listeners.contains(l)) listeners.remove(l);
	}
}
