/* $Id: ProcessDesignFrame.java,v 1.1 2007/10/19 08:19:34 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfd;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JInternalFrame;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;

import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.tool.wfd.element.ActivityElement;
import jp.co.argo21.nautica.tool.wfd.element.ProcessElement;

/**
 * プロセスデザイン用のフレームを提供する
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class ProcessDesignFrame extends JInternalFrame
{
	/** シリアルバージョンID */
	private static final long serialVersionUID = 1L;

	/** タブペイン */
	private JTabbedPane tabbedPane;
	
	/** スプリットペイン */
	private JSplitPane splitPane;

	/** フローデザイン用のビュー */
	private ProcessDesignView view;

	/** アクティビティインスペクタ表示用のビュー */
	private ActivityInspectorView inspectorView;
	
	/** プロセス */
	private ProcessElement process;
	
	/**  インスペクタが開かれているかどうかを示すフラグ*/
	private boolean isInfoOpened;

	/**
	 * フレームを生成する。
	 * 
	 * @param proc	プロセス
	 */
	public ProcessDesignFrame(ProcessElement proc)
	{
		super("", true, true, true, true);

		ResourceManager rm = WorkflowDesignerManager.getResourceManager();

		setFrameIcon(IconManager.getIcon("process-frameicon"));
		
		this.process = proc;
		view = new ProcessDesignView(process);
		view.addProcessInfoActionListener(new ProcessInfoAction());
		view.addDividerActionListener(new DividerAction());

		inspectorView = new ActivityInspectorView();
		
		splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
		splitPane.setLeftComponent(view);
		splitPane.setRightComponent(inspectorView);
		
		tabbedPane = new JTabbedPane();
		tabbedPane.addTab(rm.getResource("ProcessDesignFrame.tab.0"), splitPane);
		
		Container contentPane = this.getContentPane();
		BorderLayout layout = new BorderLayout();
		contentPane.setLayout(layout);
		contentPane.add(tabbedPane, BorderLayout.CENTER);

		setTitle(rm.getResource("ProcessDesignFrame.title"));
		setSize(new Dimension(800, 600));
		setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
		addInternalFrameListener(
			new InternalFrameAdapter() {
				public void internalFrameOpened(InternalFrameEvent e) {
					opened();
				}

				public void internalFrameActivated(InternalFrameEvent e) {
					activated();
				}

				public void internalFrameDeactivated(InternalFrameEvent e) {
					deactivated();
				}

				public void internalFrameClosing(InternalFrameEvent e) {
					close();
				}
			}
		);
		this.validate();
	}

	/**
	 * プロセスデザインビューを返す。
	 * 
	 * @return	プロセスデザインビュー
	 */
	public ProcessDesignView getView()
	{
		return view;
	}

	/**
	 * 指定されたプロセスを編集しているかどうかを返す。
	 * 
	 * @param proc	プロセス
	 * @return	編集している場合にはtrue
	 */	
	public boolean isEditingProcess(ProcessElement proc)
	{
		return (this.process == proc);
	}

	/**
	 * 指定されたアクティビティ用のインスペクタをセットする。
	 * 
	 * @param act	アクティビティ
	 */	
	public void setInspector(ActivityElement act)
	{
		inspectorView.inspect(act);
	}

	/**
	 * インスペクタを表示する。
	 */	
	public void showInspector()
	{
		splitPane.setDividerLocation(0.5d);
		isInfoOpened = true;
	}

	/**
	 * プロセスを返す。
	 * 
	 * @return	プロセス
	 */
	public ProcessElement getProcess()
	{
		return process;
	}


	/**
	 * フレームを初期化する。
	 */
	void initialize()
	{
		ActivityElement selected = view.getProcessFlowView().getSelectedActivity();
		
		inspectorView.inspect(selected);
		
		isInfoOpened = false;
		splitPane.setDividerLocation(1.0d);
	}
	
	/**
	 * 開いた時の処理
	 */
	void opened()
	{
		FlowDesignToolBar toolbar = WorkflowDesignerManager.getMainFrame().getToolbox().getToolBar();
		toolbar.addModeChangedListener(view);
	}
	
	/**
	 * アクティブ状態になった時の処理
	 */
	void activated()
	{
		view.getProcessFlowView().notifyFrameIsActivated();
	}
	
	/**
	 * 非アクティブ状態になった時の処理
	 */
	void deactivated()
	{
		view.getProcessFlowView().notifyFrameIsDeactivated();
	}
	
	/**
	 * 閉じる時の処理
	 */
	void close()
	{
		FlowDesignToolBar toolbar = WorkflowDesignerManager.getMainFrame().getToolbox().getToolBar();
		toolbar.removeModeChangedListener(view);
		setVisible(false);
		dispose();
	}

	/**
	 * プロセス情報ボタンが押された場合の処理
	 */
	class ProcessInfoAction implements ActionListener
	{
		/**
		 * プロセス情報設定用のインスペクタを表示する。
		 * 
		 * @param e	イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			InspectorDialog.showDialog(ProcessDesignFrame.this, process);
		}
	}

	/**
	 * インスペクタ表示ボタンが押された場合の処理
	 */
	class DividerAction implements ActionListener
	{
		/**
		 * 状態に応じて、インスペクタの表示と非表示を切り替える。
		 * 
		 * @param e	イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			if (isInfoOpened) splitPane.setDividerLocation(1.0d);
			else splitPane.setDividerLocation(0.5d);
			isInfoOpened = (! isInfoOpened);
		}
	}
}
