/* $Id: VariableInspector.java,v 1.2 2007/12/11 09:54:58 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfd.inspector;

import java.awt.Component;
import java.awt.Dimension;

import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.border.Border;

import jp.co.argo21.nautica.commons.swing.ByteLimitedDocument;
import jp.co.argo21.nautica.commons.swing.DialogUtilities;
import jp.co.argo21.nautica.commons.swing.NumericDocument;
import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.commons.util.StringUtils;
import jp.co.argo21.nautica.tool.wfd.IconView;
import jp.co.argo21.nautica.tool.wfd.WorkflowDesignerConstants;
import jp.co.argo21.nautica.tool.wfd.WorkflowDesignerManager;
import jp.co.argo21.nautica.tool.wfd.element.Element;
import jp.co.argo21.nautica.tool.wfd.element.PackageElement;
import jp.co.argo21.nautica.tool.wfd.element.VariableElement;

/**
 * プロセス変数用インスペクタ。
 * 
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class VariableInspector extends AbstractElementInspector
{
	/** シリアルバージョンID */
	private static final long serialVersionUID = 1L;

	/** 真理値名 */
	private static String[] bools;
	/** 枠線 */
	protected static final Border border = BorderFactory.createEtchedBorder();

	/** アイコン表示エリア */
	private IconView iconView;
	/** ID表示エリア */
	private JLabel idField;
	/** 名称表示エリア */
	private JTextField nameField;
	/** 備考表示エリア */
	private JTextField descriptionField;
	/** 文字列ボタン */
	private JRadioButton strButton;
	/** 数値ボタン */
	private JRadioButton numButton;
	/** 真理値ボタン */
	private JRadioButton boolButton;
	/** 文字列表示エリア */
	private JTextField strField;
	/** 数値表示エリア */
	private JTextField numField;
	/** 真理値表示エリア */
	private JComboBox boolBox;
	
	/**
	 * 定数の初期化
	 */
	static
	{
		ResourceManager rm = WorkflowDesignerManager.getResourceManager();

		bools = new String[4];
		bools[0] = rm.getResource("VariableInspector.bool.0");
		bools[1] = rm.getResource("VariableInspector.bool.1");
	}

	/**
	 * インスペクタを生成する。
	 */
	public VariableInspector()
	{
		ResourceManager rm = WorkflowDesignerManager.getResourceManager();

		JPanel panel = createBasicInfoPanel(rm);
		addTab(rm.getResource("VariableInspector.tab.0"), panel);

		validate();
	}
	
	/**
	 * 基本情報のパネルを生成する。
	 * 
	 * @param rm リソースマネージャ
	 * @return パネル
	 */
	private JPanel createBasicInfoPanel(ResourceManager rm)
	{
		JPanel panel = new JPanel();
		panel.setLayout(null);
		panel.setBorder(border);
		Dimension size = new Dimension(380, 200);
		panel.setSize(size);
		panel.setPreferredSize(size);
		JLabel l0 = new JLabel(rm.getResource("VariableInspector.label.0"));
		JLabel l1 = new JLabel(rm.getResource("VariableInspector.label.1"));
		JLabel l2 = new JLabel(rm.getResource("VariableInspector.label.2"));

		iconView = new IconView();
		iconView.setIconName(WorkflowDesignerConstants.EL_VARIABLE);
		idField = new JLabel();
		idField.setBorder(border);
		nameField = new JTextField();
		nameField.setDocument(new ByteLimitedDocument(64));
		descriptionField = new JTextField();
		descriptionField.setDocument(new ByteLimitedDocument(128));
		strButton = new JRadioButton(rm.getResource("VariableInspector.label.3"));
		numButton = new JRadioButton(rm.getResource("VariableInspector.label.4"));
		boolButton = new JRadioButton(rm.getResource("VariableInspector.label.5"));

		strButton.setSelected(true);

		ButtonGroup group = new ButtonGroup();
		group.add(strButton);
		group.add(numButton);
		group.add(boolButton);
		strField = new JTextField();
		strField.setDocument(new ByteLimitedDocument(128));
		numField = new JTextField();
		numField.setDocument(new NumericDocument(9, true));
		numField.setHorizontalAlignment(SwingConstants.RIGHT);
		boolBox = new JComboBox(bools);

		panel.add(locate(iconView,          10,  10,  60, 60));
		panel.add(locate(l0,                80,  10, 100, 25));
		panel.add(locate(l1,                80,  40, 100, 25));
		panel.add(locate(strButton,         80,  70, 100, 25));
		panel.add(locate(numButton,         80, 100, 100, 25));
		panel.add(locate(boolButton,        80, 130, 100, 25));
		panel.add(locate(l2,                80, 160, 100, 25));
		panel.add(locate(idField,          180,  10, 190, 25));
		panel.add(locate(nameField,        180,  40, 190, 25));
		panel.add(locate(strField,         180,  70, 190, 25));
		panel.add(locate(numField,         180, 100, 190, 25));
		panel.add(locate(boolBox,          180, 130, 190, 25));
		panel.add(locate(descriptionField, 180, 160, 190, 25));

		validate();
		
		return panel;
	}

	/**
	 * インスペクタを初期化する。
	 * 
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#init()
	 */	
	public void init()
	{
		idField.setText("");
		nameField.setEditable(true);
		nameField.setText("");
		descriptionField.setText("");
		nameField.setEditable(true);
		nameField.setText("");
		strButton.setEnabled(true);
		strField.setEditable(true);
		strField.setText("");
		numButton.setEnabled(true);
		numField.setEditable(true);
		numField.setText("0");
		boolButton.setEnabled(true);
		boolBox.setEnabled(true);
		boolBox.setSelectedIndex(0);
	}

	/**
	 * インスペクト対象の要素を設定する。
	 * 
	 * @param elem	対象要素
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#setInspectable(jp.co.argo21.nautica.tool.wfd.element.Element)
	 */	
	public void setInspectable(Element elem)
	{
		super.setInspectable(elem);

		if (elem != null) {
			if (elem instanceof VariableElement) {
				VariableElement var = (VariableElement)elem;
				idField.setText(var.getID());
				nameField.setText(var.getName());
				int t = var.getType();
				boolean initialized = var.isInitialized();
				if (initialized) {
					nameField.setEditable(false);
				}

				if (t == VariableElement.STRING) {
					strButton.setSelected(true);
					strField.setText(var.getInitialValue());
					if (initialized) {
						numButton.setEnabled(false);
						numField.setEditable(false);
						boolButton.setEnabled(false);
						boolBox.setEnabled(false);
					}
				} else if (t == VariableElement.INTEGER) {
					numButton.setSelected(true);
					numField.setText(var.getInitialValue());
					if (initialized) {
						strButton.setEnabled(false);
						strField.setEditable(false);
						boolButton.setEnabled(false);
						boolBox.setEnabled(false);
					}
				} else if (t == VariableElement.BOOLEAN) {
					boolButton.setSelected(true);
					String v = var.getInitialValue();
					if (v != null && v.equals("true")) {
						boolBox.setSelectedIndex(0);
					} else {
						boolBox.setSelectedIndex(1);
					}
					if (initialized) {
						strButton.setEnabled(false);
						strField.setEditable(false);
						numButton.setEnabled(false);
						numField.setEditable(false);
					}
				} else {
					strButton.setSelected(true);
					strField.setText(var.getInitialValue());
				}
				descriptionField.setText(var.getDescription());
			}
		}
		setSelectedIndex(0);
	}

	/**
	 * 設定値をチェックする。
	 * 
	 * @return	チェック結果
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#check()
	 */	
	public boolean check()
	{
		ResourceManager rm = WorkflowDesignerManager.getResourceManager();

		VariableElement var = (VariableElement)getInspectable();
		String name = nameField.getText();
		//既に変数名が使われていないことを確認する。
		PackageElement pkg = WorkflowDesignerManager.getMainFrame().getPackage();
		for (VariableElement v : pkg.getVariables()) {
			if (v != var) {
				String n = v.getName();
			
				if (StringUtils.checkNull(name)) {
					DialogUtilities.showError(rm.getResource("message.error.0030"));
					return false;
				} else if (name.equals(n)) {
					DialogUtilities.showError(rm.getResource("message.error.0031"));
					return false;
				}
			}
		}
		
		return true;
	}

	/**
	 * 設定値をコミットする。
	 * 
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#commit()
	 */	
	public void commit()
	{
		Element e = getInspectable();
		if (e != null) {
			if (e instanceof VariableElement) {
				VariableElement var = (VariableElement)e;
				var.setName(nameField.getText());
				
				int b = boolBox.getSelectedIndex();
				if (numButton.isSelected()) {
					var.setType(VariableElement.INTEGER);
					var.setInitialValue(numField.getText());
				} else if (boolButton.isSelected()) {
					var.setType(VariableElement.BOOLEAN);
					if (b == 0) {
						var.setInitialValue("true");
					} else if (b == 1) {
						var.setInitialValue("false");
					}
				} else {
					var.setType(VariableElement.STRING);
					var.setInitialValue(strField.getText());
				}
				var.setDescription(descriptionField.getText());

				fireElementChanged();
			}
		}		
	}

	/**
	 * 指定されたコンポーネントの属性を指定の大きさと座標で更新する。
	 * 
	 * @param component 対象コンポーネント
	 * @param x 左上のX座標
	 * @param y 左上のY座標
	 * @param w 幅
	 * @param h 高さ
	 * @return 変更されたコンポーネント
	 */
	protected Component locate(Component component, int x, int y, int w, int h)
	{
		component.setLocation(x, y);
		component.setSize(w, h);
		return  component;
	}
}
