/* $Id: ParticipantInspector.java,v 1.1 2007/10/19 08:19:34 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfd.inspector;

import java.awt.Component;
import java.awt.Dimension;

import javax.swing.BorderFactory;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.border.Border;

import jp.co.argo21.nautica.commons.swing.ByteLimitedDocument;
import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.tool.wfd.IconView;
import jp.co.argo21.nautica.tool.wfd.WorkflowDesignerConstants;
import jp.co.argo21.nautica.tool.wfd.WorkflowDesignerManager;
import jp.co.argo21.nautica.tool.wfd.element.Element;
import jp.co.argo21.nautica.tool.wfd.element.ParticipantElement;

/**
 * ワークフロー参加者用インスペクタ
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class ParticipantInspector extends AbstractElementInspector
{
	/** シリアルバージョンID */
	private static final long serialVersionUID = 1L;

	/** 型 */
	private static String[] types;
	/** 枠線 */
	protected static final Border border = BorderFactory.createEtchedBorder();

	/** アイコン表示エリア */
	private IconView iconView;
	/** ID表示エリア */
	private JLabel idField;
	/** 名称表示エリア */
	private JTextField nameField;
	/** タイプ */
	private JComboBox typeBox;
	/** 備考表示エリア */
	private JTextField descriptionField;
	
	/**
	 * 定数初期化
	 */
	static
	{
		ResourceManager rm = WorkflowDesignerManager.getResourceManager();

		types = new String[4];
		types[0] = rm.getResource("ParticipantInspector.type.0");
		types[1] = rm.getResource("ParticipantInspector.type.1");
		types[2] = rm.getResource("ParticipantInspector.type.2");
		types[3] = rm.getResource("ParticipantInspector.type.3");
	}

	/**
	 * ワークフロー参加者用インスペクタを生成する。
	 */
	public ParticipantInspector()
	{
		ResourceManager rm = WorkflowDesignerManager.getResourceManager();

		JPanel panel = createBasicInfoPanel(rm);
		addTab(rm.getResource("ParticipantInspector.tab.0"), panel);

		validate();
	}
	
	/**
	 * 基本情報のパネルを生成する。
	 * 
	 * @param rm リソースマネージャ
	 * @return パネル
	 */
	private JPanel createBasicInfoPanel(ResourceManager rm)
	{
		JPanel panel = new JPanel();
		panel.setLayout(null);
		panel.setBorder(border);
		Dimension size = new Dimension(380, 200);
		panel.setSize(size);
		panel.setPreferredSize(size);
		JLabel l0 = new JLabel(rm.getResource("ParticipantInspector.label.0"));
		JLabel l1 = new JLabel(rm.getResource("ParticipantInspector.label.1"));
		JLabel l2 = new JLabel(rm.getResource("ParticipantInspector.label.2"));
		JLabel l3 = new JLabel(rm.getResource("ParticipantInspector.label.3"));

		iconView = new IconView();
		iconView.setIconName(WorkflowDesignerConstants.EL_PARTICIPANT);
		idField = new JLabel();
		idField.setBorder(border);
		nameField = new JTextField();
		nameField.setDocument(new ByteLimitedDocument(64));
		descriptionField = new JTextField();
		descriptionField.setDocument(new ByteLimitedDocument(128));
		typeBox = new JComboBox(types);

		panel.add(locate(iconView,          10,  10,  60, 60));
		panel.add(locate(l0,                80,  10, 100, 25));
		panel.add(locate(l1,                80,  40, 100, 25));
		panel.add(locate(l2,                80,  70, 100, 25));
		panel.add(locate(l3,                80, 100, 100, 25));
		panel.add(locate(idField,          180,  10, 190, 25));
		panel.add(locate(nameField,        180,  40, 190, 25));
		panel.add(locate(typeBox,          180,  70, 190, 25));
		panel.add(locate(descriptionField, 180, 100, 190, 25));

		validate();
		
		return panel;
	}

	/**
	 * インスペクタを初期化する。
	 * 
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#init()
	 */
	public void init()
	{
		super.init();

		nameField.setText("");
		descriptionField.setText("");
		typeBox.setSelectedIndex(0);
	}

	/**
	 * インスペクト対象を設定する。
	 * 
	 * @param elem	対象要素
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#setInspectable(jp.co.argo21.nautica.tool.wfd.element.Element)
	 */	
	public void setInspectable(Element elem)
	{
		super.setInspectable(elem);

		if (elem != null) {
			if (elem instanceof ParticipantElement) {
				ParticipantElement part = (ParticipantElement)elem;
				idField.setText(part.getID());
				nameField.setText(part.getName());
				int t = part.getType();
				if (t == ParticipantElement.ROLE) {
					typeBox.setSelectedIndex(0);
				} else if (t == ParticipantElement.ORGANIZATIONAL_UNIT) {
					typeBox.setSelectedIndex(1);
				} else if (t == ParticipantElement.HUMAN) {
					typeBox.setSelectedIndex(2);
				} else if (t == ParticipantElement.SYSTEM) {
					typeBox.setSelectedIndex(3);
				} else {
					typeBox.setSelectedIndex(3);
				}
				descriptionField.setText(part.getDescription());
			}
		}
		setSelectedIndex(0);
	}

	/**
	 * 設定値をチェックする。
	 * 
	 * @return	チェック結果
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#check()
	 */	
	public boolean check()
	{
		return true;
	}

	/**
	 * 設定値をコミットする。
	 * 
	 * @see jp.co.argo21.nautica.tool.wfd.inspector.Inspector#commit()
	 */	
	public void commit()
	{
		Element e = getInspectable();
		if (e != null) {
			if (e instanceof ParticipantElement) {
				ParticipantElement part = (ParticipantElement)e;
				part.setName(nameField.getText());
				
				int t = typeBox.getSelectedIndex();
				if (t == 0) {
					part.setType(ParticipantElement.ROLE);
				} else if (t == 1) {
					part.setType(ParticipantElement.ORGANIZATIONAL_UNIT);
				} else if (t == 2) {
					part.setType(ParticipantElement.HUMAN);
				} else if (t == 3) {
					part.setType(ParticipantElement.SYSTEM);
				}
				part.setDescription(descriptionField.getText());

				fireElementChanged();
			}
		}		
	}

	/**
	 * 指定されたコンポーネントの属性を指定の大きさと座標で更新する。
	 * 
	 * @param component 対象コンポーネント
	 * @param x 左上のX座標
	 * @param y 左上のY座標
	 * @param w 幅
	 * @param h 高さ
	 * @return 変更されたコンポーネント
	 */
	protected Component locate(Component component, int x, int y, int w, int h)
	{
		component.setLocation(x, y);
		component.setSize(w, h);
		return  component;
	}
}
