/* $Id: ParamMappingTableModel.java,v 1.2 2007/12/11 09:54:58 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfd.inspector;

import java.util.ArrayList;
import java.util.List;

import javax.swing.table.AbstractTableModel;

import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.tool.wfd.WorkflowDesignerManager;
import jp.co.argo21.nautica.tool.wfd.element.ParameterElement;
import jp.co.argo21.nautica.tool.wfd.element.VariableElement;

/**
 * 実パラメータの割付管理テーブル用制御モデル
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
class ParamMappingTableModel extends AbstractTableModel
{
	/** シリアルバージョンID */
	private static final long serialVersionUID = 1L;

	/** テーブルの列名 */
	private static String[] columnNames;
	
	/** パラメータ */
	private List<MappedParam> params = new ArrayList<MappedParam>();

	/**
	 * 定数初期化
	 */
	static
	{
		ResourceManager rm = WorkflowDesignerManager.getResourceManager();

		columnNames = new String[3];
		columnNames[0] = rm.getResource("ActualParamTableModel.columnName.0");
		columnNames[1] = rm.getResource("ActualParamTableModel.columnName.1");
		columnNames[2] = rm.getResource("ActualParamTableModel.columnName.2");
	}


	/**
	 * パラメータテーブル制御モデルを生成する。
	 */
	ParamMappingTableModel()
	{
	}

	/**
	 * データソースオブジェクトが管理する列数を返す。
	 * JTableはこのメソッドを使って、初期化時に作成および表示する
	 * 必要のある列数を判断する。
	 *
	 * @return	モデルの列数
	 */
	public int getColumnCount()
	{
		return 3;
	}

	/**
	 * データソースオブジェクトが管理するレコード数を返す。
	 * JTableはこのメソッドを使って、作成および表示する必要のある行数を判断する。
	 *
	 * @return	モデルの行数
	 */
	public int getRowCount()
	{
		return params.size();
	}

	/**
	 * colにある列の名前を返す。
	 * このメソッドは、テーブルの列ヘッダ名を初期化するために使われる。
	 *
	 * @param	col		列のインデックス
	 * @return	列ヘッダ名
	 */
	public String getColumnName(int col)
	{
		return columnNames[col];
	}

	/**
	 * セルの値を返す。
	 *
	 * @param	row		値が参照される行
	 * @param	col		値が参照される列
	 * @return	指定されたセルにある値
	 */
	public Object getValueAt(int row, int col)
	{
		MappedParam p = (MappedParam)params.get(row);
		if (col == 0) {
			int m = p.getParam().getMode();
			if (m == ParameterElement.IN) {
				return ParameterInspector.PARAM_MODES[0];
			} else if (m == ParameterElement.OUT) {
				return ParameterInspector.PARAM_MODES[1];
			} else if (m == ParameterElement.INOUT) {
				return ParameterInspector.PARAM_MODES[2];
			}
		} else if (col == 1) {
			int t = p.getParam().getType();
			if (t == ParameterElement.STRING) {
				return ParameterInspector.PARAM_TYPES[0];
			} else if (t == ParameterElement.INTEGER) {
				return ParameterInspector.PARAM_TYPES[1];
			} else if (t == ParameterElement.BOOLEAN) {
				return ParameterInspector.PARAM_TYPES[2];
			}
		} else if (col == 2) {
			VariableElement ap = p.getActualParam();
			if (ap != null) return ap.getName();
		}
		return "";
	}

	/**
	 * 列のClassを返す。
	 *
	 * @param	col		列
	 * @return	モデルのオブジェクト値の共通の上位クラス
	 */
	public Class<?> getColumnClass(int col)
	{
		return String.class;
	}

	/**
	 * rowとcolにあるセルが編集可能な場合はtrueを返す。
	 * そうでない場合、セルでsetValueAt()を呼び出しても、そのセルの値は変更されない。
	 *
	 * @param	row		値が参照される行
	 * @param	col		値が参照される列
	 * @return	セルが編集可能な場合はtrue
	 */
	public boolean isCellEditable(int row, int col)
	{
		return false;
	}
	
	/**
	 * パラメータを追加する。
	 * 
	 * @param p	パラメータ
	 */
	public void addMappedParam(MappedParam p)
	{
		if (p == null) return;

		params.add(p);
		fireTableDataChanged();
	}

	/**
	 * マッピングを削除する。
	 * 
	 * @param ix	インデクス
	 */	
	public void removeMapping(int ix)
	{
		if (ix < 0 || ix >= params.size()) return;
		
		MappedParam mp = (MappedParam)params.get(ix);
		mp.setActualParam(null);
		fireTableDataChanged();
	}

	/**
	 * マッピングを更新する
	 * 
	 * @param ix	インデクス
	 * @param actual	実パラメータ
	 */	
	public void updateMapping(int ix, VariableElement actual)
	{
		if (ix < 0 || ix >= params.size()) return;
		
		MappedParam mp = (MappedParam)params.get(ix);
		mp.setActualParam(actual);
		fireTableDataChanged();
	}

	/**
	 * パラメータをクリアする。
	 */	
	public void clearAllParams()
	{
		params.clear();
		fireTableDataChanged();
	}

	/**
	 * リストを初期化する。
	 * 
	 * @param parameters	パラメータ一覧
	 */	
	void initParams(List<ParameterElement> parameters)
	{
		clearAllParams();
		
		if (parameters == null) return;
		
		for (Object o : parameters) {
			if (o == null) continue;
			
			if (o instanceof ParameterElement) {
				MappedParam mp = new MappedParam((ParameterElement)o);
				addMappedParam(mp);
			}
		}
		fireTableDataChanged();
	}

	/**
	 * インデクスに対応するパラメータを返す。
	 * 
	 * @param ix	インデクス
	 * @return	パラメータ
	 */	
	MappedParam getMappedParam(int ix)
	{
		if (ix < 0 || ix >= params.size()) return null;
		
		return (MappedParam)params.get(ix);
	}

	/**
	 * すべてのパラメータを返す。
	 * 
	 * @return	パラメータ一覧
	 */	
	List<MappedParam> getAllMappedParams()
	{
		return params;
	}
}
