/* $Id: UnaryExpression.java,v 1.2 2007/12/11 09:54:58 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfd.expression;

import java.util.List;

import org.apache.xmlbeans.XmlObject;

import jp.co.argo21.nautica.tool.wfd.element.VariableElement;
import jp.co.argo21.nautica.workflow.xpdl.LogicalExpressionDocument;
import jp.co.argo21.nautica.workflow.xpdl.RelationalExpressionDocument;
import jp.co.argo21.nautica.workflow.xpdl.UnaryExpressionDocument;

/**
 * NOT条件式
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class UnaryExpression implements Expression
{
	/** NOT */
	private static final int NOT = UnaryExpressionDocument.UnaryExpression.Type.INT_NOT;

	/** 演算子 */
	private int operator;
	/** NOT演算対象 */
	private Expression subExpression;
	
	/**
	 * NOT条件式を生成する。
	 * 
	 * @param exp	演算対象の条件式
	 */
	public UnaryExpression(Expression  exp)
	{
		operator = NOT;
		subExpression = exp;
	}
	
	/**
	 * XMLからNOT条件式を生成する。
	 * 
	 * @param exp	XML
	 */
	public UnaryExpression(UnaryExpressionDocument.UnaryExpression  exp)
	{
		operator = NOT;
		if (exp.isSetUnaryExpression()) {
			UnaryExpressionDocument.UnaryExpression unary = exp.getUnaryExpression();
			subExpression = new UnaryExpression(unary);
		} else if (exp.isSetLogicalExpression()) {
			LogicalExpressionDocument.LogicalExpression logic = exp.getLogicalExpression();
			subExpression = new LogicalExpression(logic);
		} else if (exp.isSetRelationalExpression()) {
			RelationalExpressionDocument.RelationalExpression rel = exp.getRelationalExpression();
			subExpression = new RelationalExpression(rel);
		}
	}
	
	/**
	 * 引数のオブジェクトに条件式を格納する。
	 * 
	 * @param o	XMLオブジェクト
	 * @see jp.co.argo21.nautica.tool.wfd.expression.Expression#saveTo(org.apache.xmlbeans.XmlObject)
	 */
	public void saveTo(XmlObject o)
	{
		if (o == null) return;
		if (subExpression == null) return;
		if (!(o instanceof UnaryExpressionDocument.UnaryExpression)) return;

		UnaryExpressionDocument.UnaryExpression ue
			= (UnaryExpressionDocument.UnaryExpression)o;

		ue.setType(UnaryExpressionDocument.UnaryExpression.Type.Enum.forInt(operator));
		XmlObject exp = null;
		if (subExpression instanceof UnaryExpression) {
			exp = ue.addNewUnaryExpression();
		} else if (subExpression instanceof LogicalExpression) {
			exp = ue.addNewLogicalExpression();
		} else if (subExpression instanceof RelationalExpression) {
			exp = ue.addNewRelationalExpression();
		}
		subExpression.saveTo(exp);
	}

	/**
	 * 引数の情報を元に条件を評価する。
	 * 
	 * @param list	変数の現在値
	 * @return	条件が満たされた場合はtrue
	 * @see jp.co.argo21.nautica.tool.wfd.expression.Expression#eval(java.util.List)
	 */
	public boolean eval(List<ConditionalVariable> list)
	{
		if (subExpression == null) return false;
		else return (! subExpression.eval(list));
	}

	/**
	 * 条件式の文字列表現を返す。
	 * 
	 * @return	文字列表現
	 * @see jp.co.argo21.nautica.tool.wfd.expression.Expression#getExpressionString()
	 */
	public String getExpressionString()
	{
		String s = "null";
		if (subExpression != null) s = subExpression.getExpressionString();

		return "! (" + s + ")";
	}

	/**
	 * 変数一覧を利用しながら条件式の文字列表現を返す。
	 * 
	 * @param vars	変数一覧
	 * @return	文字列表現
	 * @see jp.co.argo21.nautica.tool.wfd.expression.Expression#getExpressionString(java.util.List)
	 */
	public String getExpressionString(List<VariableElement> vars)
	{
		String s = "null";
		if (subExpression != null) s = subExpression.getExpressionString(vars);

		return "! (" + s + ")";
	}

	/**
	 * この条件式に含まれるすべての条件式を返す。
	 * 
	 * @param exps	条件式一覧
	 * @see jp.co.argo21.nautica.tool.wfd.expression.Expression#getAllExpressions(java.util.List)
	 */
	public void getAllExpressions(List<Expression> exps)
	{
		if (exps.contains(this) == false) exps.add(this);
		if (subExpression != null) subExpression.getAllExpressions(exps);
	}
}
