/* $Id: LogicalExpression.java,v 1.2 2007/12/11 09:54:58 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfd.expression;

import java.util.List;

import org.apache.xmlbeans.XmlObject;

import jp.co.argo21.nautica.tool.wfd.element.VariableElement;
import jp.co.argo21.nautica.workflow.xpdl.LogicalExpressionDocument;
import jp.co.argo21.nautica.workflow.xpdl.RelationalExpressionDocument;
import jp.co.argo21.nautica.workflow.xpdl.UnaryExpressionDocument;

/**
 * 論理条件式
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class LogicalExpression implements Expression
{
	/** AND */
	public static final int AND = LogicalExpressionDocument.LogicalExpression.Type.INT_AND;
	/** OR */
	public static final int OR  = LogicalExpressionDocument.LogicalExpression.Type.INT_OR;
	/** && */
	private static final String AND_STR = "&&";
	/** || */
	private static final String OR_STR  = "||";

	/** 演算子 */
	private int operator;
	/** 演算対象の条件式 */
	private Expression[] subExpressions;
	
	/**
	 * 論理条件式を生成する。
	 * 
	 * @param expL	左辺条件式
	 * @param ope	演算子
	 * @param expR	右辺条件式
	 */
	public LogicalExpression(Expression expL, int ope, Expression expR)
	{
		operator = ope;
		subExpressions = new Expression[2];
		subExpressions[0] = expL;
		subExpressions[1] = expR;
	}
	
	/**
	 * XMLから論理条件式を生成する。
	 * 
	 * @param exp	XML
	 */
	public LogicalExpression(LogicalExpressionDocument.LogicalExpression exp)
	{
		operator = exp.getType().intValue();
		subExpressions = new Expression[2];

		int i = 0;

		for (int j = 0; j < exp.sizeOfUnaryExpressionArray(); j++) {
			if (i == 2) break;
			UnaryExpressionDocument.UnaryExpression unary = exp.getUnaryExpressionArray(j);
			subExpressions[i] = new UnaryExpression(unary);
			i++;
		}

		for (int j = 0; j < exp.sizeOfLogicalExpressionArray(); j++) {
			if (i == 2) break;
			LogicalExpressionDocument.LogicalExpression logic = exp.getLogicalExpressionArray(j);
			subExpressions[i] = new LogicalExpression(logic);
			i++;
		}

		for (int j = 0; j < exp.sizeOfRelationalExpressionArray(); j++) {
			if (i == 2) break;
			RelationalExpressionDocument.RelationalExpression rel = exp.getRelationalExpressionArray(j);
			subExpressions[i] = new RelationalExpression(rel);
			i++;
		}
	}
	
	/**
	 * 引数のオブジェクトに条件式を格納する。
	 * 
	 * @param o	XMLオブジェクト
	 * @see jp.co.argo21.nautica.tool.wfd.expression.Expression#saveTo(org.apache.xmlbeans.XmlObject)
	 */
	public void saveTo(XmlObject o)
	{
		if (o == null) return;
		if (!(o instanceof LogicalExpressionDocument.LogicalExpression)) return;

		LogicalExpressionDocument.LogicalExpression le
			= (LogicalExpressionDocument.LogicalExpression)o;

		le.setType(LogicalExpressionDocument.LogicalExpression.Type.Enum.forInt(operator));
		for (int i = 0; i < subExpressions.length; i++) {
			if (subExpressions[i] == null) continue;

			XmlObject exp = null;
			if (subExpressions[i] instanceof UnaryExpression) {
				exp = le.addNewUnaryExpression();
			} else if (subExpressions[i] instanceof LogicalExpression) {
				exp = le.addNewLogicalExpression();
			} else if (subExpressions[i] instanceof RelationalExpression) {
				exp = le.addNewRelationalExpression();
			}
			subExpressions[i].saveTo(exp);
		}
	}

	/**
	 * 引数の情報を元に条件を評価する。
	 * 
	 * @param list	変数の現在値
	 * @return	条件が満たされた場合はtrue
	 * @see jp.co.argo21.nautica.tool.wfd.expression.Expression#eval(java.util.List)
	 */
	public boolean eval(List<ConditionalVariable> list)
	{
		boolean left = false;
		boolean right = false;

		if (subExpressions[0] != null) left = subExpressions[0].eval(list);
		if (subExpressions[1] != null) right = subExpressions[1].eval(list);
		
		if (operator == AND) return (left && right);
		else return (left || right);
	}

	/**
	 * 条件式の文字列表現を返す。
	 * 
	 * @return	文字列表現
	 * @see jp.co.argo21.nautica.tool.wfd.expression.Expression#getExpressionString()
	 */
	public String getExpressionString()
	{
		String ls = "null";
		String rs = "null";
		if (subExpressions[0] != null) ls = subExpressions[0].getExpressionString();
		if (subExpressions[1] != null) rs = subExpressions[1].getExpressionString();
		
		return "(" + ls +  ") " + getOpeString() + " (" + rs + ")";
	}

	/**
	 * 変数一覧を利用しながら条件式の文字列表現を返す。
	 * 
	 * @param vars	変数一覧
	 * @return	文字列表現
	 * @see jp.co.argo21.nautica.tool.wfd.expression.Expression#getExpressionString(java.util.List)
	 */
	public String getExpressionString(List<VariableElement> vars)
	{
		String ls = "null";
		String rs = "null";
		if (subExpressions[0] != null) ls = subExpressions[0].getExpressionString(vars);
		if (subExpressions[1] != null) rs = subExpressions[1].getExpressionString(vars);
		
		return "(" + ls +  ") " + getOpeString() + " (" + rs + ")";
	}

	/**
	 * この条件式に含まれるすべての条件式を返す。
	 * 
	 * @param exps	条件式一覧
	 * @see jp.co.argo21.nautica.tool.wfd.expression.Expression#getAllExpressions(java.util.List)
	 */
	public void getAllExpressions(List<Expression> exps)
	{
		if (exps.contains(this) == false) exps.add(this);
		if (subExpressions[0] != null) subExpressions[0].getAllExpressions(exps);
		if (subExpressions[1] != null) subExpressions[1].getAllExpressions(exps);
	}

	/**
	 * 縁z何誌の文字列表現を返す。
	 * 
	 * @return	文字列表現
	 */	
	private String getOpeString()
	{
		if (operator == AND) return AND_STR;
		else return OR_STR;
	}
}
