/* $Id: TransitionElementImpl.java,v 1.3 2007/12/11 09:54:58 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2002.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfd.element.impl;

import java.util.List;

import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlObject;
import org.wfmc.x2002.xpdl10.ConditionDocument;
import org.wfmc.x2002.xpdl10.TransitionDocument;

import jp.co.argo21.nautica.tool.wfd.element.ActivityElement;
import jp.co.argo21.nautica.tool.wfd.element.Element;
import jp.co.argo21.nautica.tool.wfd.element.TransitionElement;
import jp.co.argo21.nautica.tool.wfd.expression.ExceptionExpression;
import jp.co.argo21.nautica.tool.wfd.expression.Expression;
import jp.co.argo21.nautica.tool.wfd.expression.LogicalExpression;
import jp.co.argo21.nautica.tool.wfd.expression.OtherwiseExpression;
import jp.co.argo21.nautica.tool.wfd.expression.RelationalExpression;
import jp.co.argo21.nautica.tool.wfd.expression.UnaryExpression;
import jp.co.argo21.nautica.workflow.xpdl.ConditionalExpressionDocument;
import jp.co.argo21.nautica.commons.util.IDGenerator;
import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.commons.util.StringUtils;

/**
 * 遷移
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.3 $
 * @since   Nautica Workflow 1.0
 */
public class TransitionElementImpl implements TransitionElement
{
	/** ID */
	private String id;
	/** 名称 */
	private String name;
	/** 備考 */
	private String description;
	/** 遷移元アクティビティ */
	private ActivityElement source;
	/** 遷移先アクティビティ */
	private ActivityElement destination;
	/** 遷移条件 */
	private Expression condition;
	/** 遷移型 */
	private int type;

	/**
	 * 遷移を生成する。
	 * 
	 * @param s	遷移元アクティビティ
	 * @param d	遷移先アクティビティ
	 */
	public TransitionElementImpl(ActivityElement s, ActivityElement d)
	{
		setID(IDGenerator.generate(ID_PREFIX));
		setName("");
		setDescription("");
		setSource(s);
		setDestination(d);
		setType(CONDITION);
		setCondition(null);
	}

	/**
	 * 遷移を生成する。
	 * 
	 * @param tr	XmlObject
	 * @param acts	アクティビティ一覧
	 */
	public TransitionElementImpl(TransitionDocument.Transition tr, List<ActivityElement> acts)
	{
		String id = tr.getId();
		if (StringUtils.checkNull(id)) {
			setID(IDGenerator.generate(ID_PREFIX));
		} else {
			setID(id);
		}
		setName(tr.getName());
		setDescription(tr.getDescription());
		
		String sid = tr.getFrom();
		setSource(getActivityElement(acts, sid));
		
		String did = tr.getTo();
		setDestination(getActivityElement(acts, did));
		
		if (tr.isSetCondition()) {
			ConditionDocument.Condition c = tr.getCondition();
			int type = c.getType().intValue();
			try {
				if (type == CONDITION) {
					XmlObject obj = XmlObject.Factory.parse(c.toString());
					//値が設定されている場合のみConditionalExpressionDocument型の
					//オブジェクトが返されてくる。
					if (obj == null) {
						setCondition(null);
					} else {
						ConditionalExpressionDocument ceDoc
							= (ConditionalExpressionDocument)obj.changeType(ConditionalExpressionDocument.type);
						ConditionalExpressionDocument.ConditionalExpression ce
							= ceDoc.getConditionalExpression();
						if (ce.isSetRelationalExpression()) {
							setCondition(new RelationalExpression(ce.getRelationalExpression()));
						} else if (ce.isSetUnaryExpression()) {
							setCondition(new UnaryExpression(ce.getUnaryExpression()));
						} else if (ce.isSetLogicalExpression()) {
							setCondition(new LogicalExpression(ce.getLogicalExpression()));
						}
					}
				} else if (type == OTHERWISE) {
					setCondition(new OtherwiseExpression());
				} else if (type == EXCEPTION) {
					setCondition(new ExceptionExpression());
				}
			} catch (XmlException ex) {
				//NOP
			}
		} else {
			setCondition(null);
		}
	}

	/**
	 * IDを設定する。
	 * 
	 * @param id	ID
	 */	
	private void setID(String id)
	{
		this.id = id;
	}

	/**
	 * 遷移元アクティビティを設定する。
	 * 
	 * @param act	遷移元アクティビティ
	 */
	private void setSource(ActivityElement act)
	{
		this.source = act;
	}

	/**
	 * 遷移先アクティビティを設定する。
	 * 
	 * @param act	遷移先アクティビティ
	 */
	private void setDestination(ActivityElement act)
	{
		this.destination = act;
	}

	/**
	 * 指定されたIDを持つアクティビティを返す。
	 * 
	 * @param acts	アクティビティ一覧
	 * @param id	アクティビティID
	 * @return	アクティビティ
	 */
	private ActivityElement getActivityElement(List<ActivityElement> acts, String id)
	{
		for (ActivityElement act : acts) {
			String aid = act.getID();
			if (id.equals(aid)) return act;
		}
		return null;
	}

	/**
	 * 説明を返す。
	 * 
	 * @return 説明
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getDescription()
	 */
	public String getDescription()
	{
		return description;
	}

	/**
	 * IDを返す。
	 * 
	 * @return ID
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getID()
	 */
	public String getID()
	{
		return id;
	}

	/**
	 * 名称を返す。
	 * 
	 * @return 名称
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getName()
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * 説明を設定する。
	 * 
	 * @param desc 説明
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#setDescription(java.lang.String)
	 */
	public void setDescription(String desc)
	{
		this.description = desc;
	}

	/**
	 * 名称を設定する。
	 * 
	 * @param name 名称
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#setName(java.lang.String)
	 */
	public void setName(String name)
	{
		this.name = name;
	}

	/**
	 * 遷移条件を返す。
	 * 
	 * @return	遷移条件
	 * @see jp.co.argo21.nautica.tool.wfd.element.TransitionElement#getCondition()
	 */
	public Expression getCondition()
	{
		return condition;
	}

	/**
	 * 遷移先アクティビティを返す。
	 * 
	 * @return	遷移先アクティビティ
	 * @see jp.co.argo21.nautica.tool.wfd.element.TransitionElement#getDestination()
	 */
	public ActivityElement getDestination()
	{
		return destination;
	}

	/**
	 * 遷移元アクティビティを返す。
	 * 
	 * @return	遷移元アクティビティ
	 * @see jp.co.argo21.nautica.tool.wfd.element.TransitionElement#getSource()
	 */
	public ActivityElement getSource()
	{
		return source;
	}

	/**
	 * 遷移型を返す。
	 * 
	 * @return	遷移型
	 * @see jp.co.argo21.nautica.tool.wfd.element.TransitionElement#getType()
	 */
	public int getType()
	{
		return type;
	}

	/**
	 * 遷移条件を設定する。
	 * 
	 * @param cond	遷移条件
	 * @see jp.co.argo21.nautica.tool.wfd.element.TransitionElement#setCondition(jp.co.argo21.nautica.tool.wfd.expression.Expression)
	 */
	public void setCondition(Expression cond)
	{
		if (cond == null) {
			type = CONDITION;
		} else if (cond instanceof OtherwiseExpression) {
			type = OTHERWISE;
		} else if (cond instanceof ExceptionExpression) {
			type = EXCEPTION;
		} else {
			type = CONDITION;
		}
		condition = cond;
	}

	/**
	 * 遷移型を設定する。
	 * 
	 * @param i	遷移型
	 * @see jp.co.argo21.nautica.tool.wfd.element.TransitionElement#setType(int)
	 */
	public void setType(int i)
	{
		type = i;
	}

	/**
	 * ツールボックス文字列を返す。
	 * 
	 * @param rm	リソースマネージャ
	 * @return	文字列
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getTooltipText(jp.co.argo21.nautica.commons.util.ResourceManager)
	 */
	public String getTooltipText(ResourceManager rm)
	{
		StringBuffer s = new StringBuffer();
		s = s.append("<html>");
		s = s.append("<b>").append(getName()).append("</b>");
		s = s.append("<table border>");
		s = s.append("<tr><td width=100>")
			.append(rm.getResource("TransitionElement.label.0"))
			.append("</td><td width=150>")
			.append(getID())
			.append("</td></tr>");
		s = s.append("<tr><td width=100>")
			.append(rm.getResource("TransitionElement.label.3"))
			.append("</td><td width=150>")
			.append(getDescription())
			.append("</td></tr>");
		s = s.append("</table>");
		s = s.append("</html>");
		return s.toString();
	}

	/**
	 * 印刷用文字列を返す。
	 * 
	 * @param rm	リソースマネージャ
	 * @return	文字列
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getPrintText(jp.co.argo21.nautica.commons.util.ResourceManager)
	 */
	public String getPrintText(ResourceManager rm)
	{	
		StringBuffer s = new StringBuffer();
		return s.toString();
	}

	/**
	 * 指定された要素を使用中かどうかを返す。
	 *
	 * @param elem 要素
	 * @return 使用中ならtrue
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#isUsed(jp.co.argo21.nautica.tool.wfd.element.Element)
	 */
	public boolean isUsed(Element elem)
	{
		if (elem == null) return false;
		if (elem == source) return true;
		if (elem == destination) return true;
		return false;
	}
}
