/* $Id: ProcessElementImpl.java,v 1.2 2007/12/11 09:54:58 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfd.element.impl;

import java.util.ArrayList;
import java.util.List;

import org.wfmc.x2002.xpdl10.ActivitiesDocument;
import org.wfmc.x2002.xpdl10.ActivityDocument;
import org.wfmc.x2002.xpdl10.ExtendedAttributeDocument;
import org.wfmc.x2002.xpdl10.ExtendedAttributesDocument;
import org.wfmc.x2002.xpdl10.FormalParameterDocument;
import org.wfmc.x2002.xpdl10.FormalParametersDocument;
import org.wfmc.x2002.xpdl10.ProcessHeaderDocument;
import org.wfmc.x2002.xpdl10.RedefinableHeaderDocument;
import org.wfmc.x2002.xpdl10.ResponsiblesDocument;
import org.wfmc.x2002.xpdl10.TransitionDocument;
import org.wfmc.x2002.xpdl10.TransitionsDocument;
import org.wfmc.x2002.xpdl10.WorkflowProcessDocument;

import jp.co.argo21.nautica.commons.util.IDGenerator;
import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.commons.util.StringUtils;
import jp.co.argo21.nautica.tool.wfd.element.ActivityElement;
import jp.co.argo21.nautica.tool.wfd.element.ActivityState;
import jp.co.argo21.nautica.tool.wfd.element.Element;
import jp.co.argo21.nautica.tool.wfd.element.ParameterElement;
import jp.co.argo21.nautica.tool.wfd.element.ProcessElement;
import jp.co.argo21.nautica.tool.wfd.element.TransitionElement;

/**
 * プロセス
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class ProcessElementImpl implements ProcessElement
{
	/** 未初期化時の仮ID */
	private static final String UNINITIALIZED_ID = "--UNINITIALIZED--"; 

	/** ID */
	private String id;
	/** 名称 */
	private String name;
	/** 備考 */
	private String description;
	/** 期間単位 */
	private int durationUnit;
	/** 実行期限 */
	private int limit;
	/** アクセスレベル */
	private int accessLevel;
	/** 論理座標幅 */
	private int logicalWidth;
	/** 論理座標高さ */
	private int logicalHeight;
	/** 通知者一覧 */
	private List<String> responsibles = new ArrayList<String>();
	/** パラメータ一覧 */
	private List<ParameterElement> parameters = new ArrayList<ParameterElement>();
	/** アクティビティ一覧 */
	private List<ActivityElement> activities = new ArrayList<ActivityElement>();
	/** 遷移一覧 */
	private List<TransitionElement> transitions = new ArrayList<TransitionElement>();

	/**
	 * プロセスを生成する。
	 * 
	 * @param n	名称
	 */
	public ProcessElementImpl(String n)
	{
		setID(IDGenerator.generate(ID_PREFIX));
		setName(getID());
		setDescription("");
		setDurationUnit(DAY);
		setLimit(0);
		setAccessLevel(PRIVATE);
		setLogicalWidth(10);
		setLogicalHeight(10);
		
		ActivityState state = new StartActivityState();
		ActivityElement act = new ActivityElementImpl(state);
		act.setName("Start");
		getActivities().add(act);
	}

	/**
	 * プロセスを生成する。
	 * 
	 * @param proc	XmlObject
	 */
	public ProcessElementImpl(WorkflowProcessDocument.WorkflowProcess proc)
	{
		String id = proc.getId();
		if (StringUtils.checkNull(id) || id.equals(UNINITIALIZED_ID)) {
			setID(IDGenerator.generate(ID_PREFIX));
		} else {
			setID(id);
		}
		setName(proc.getName());
		
		ProcessHeaderDocument.ProcessHeader ph = proc.getProcessHeader();
		setDescription(ph.getDescription());
		setDurationUnit(ph.getDurationUnit().intValue());
		try {
			setLimit(Integer.parseInt(ph.getLimit()));
		} catch (Exception ex) {
			setLimit(0);
		}
		setAccessLevel(proc.getAccessLevel().intValue());

		if (proc.isSetRedefinableHeader()) {
			RedefinableHeaderDocument.RedefinableHeader rhd
				= proc.getRedefinableHeader();
			if (rhd.isSetResponsibles()) {
				ResponsiblesDocument.Responsibles rsps = rhd.getResponsibles();
				int count = rsps.sizeOfResponsibleArray();
				for (int i = 0; i < count; i++) {
					String rsp = rsps.getResponsibleArray(i);
					addResponsible(rsp);
				}
			}
		}

		if (proc.isSetExtendedAttributes()) {
			ExtendedAttributesDocument.ExtendedAttributes eas = proc.getExtendedAttributes();
			int count = eas.sizeOfExtendedAttributeArray();
			for (int i = 0; i < count; i++) {
				ExtendedAttributeDocument.ExtendedAttribute ea = eas.getExtendedAttributeArray(i);
				String n = ea.getName();
				String v = ea.getValue();
				if (n.equals(LOGICAL_WIDTH)) {
					setLogicalWidth(Integer.parseInt(v));
				} else if (n.equals(LOGICAL_HEIGHT)) {
					setLogicalHeight(Integer.parseInt(v));
				}
			}
		}
		
		if (proc.isSetFormalParameters()) {
			FormalParametersDocument.FormalParameters pms = proc.getFormalParameters();
			int count = pms.sizeOfFormalParameterArray();
			for (int i = 0; i < count; i++) {
				FormalParameterDocument.FormalParameter pm = pms.getFormalParameterArray(i);
				ParameterElement e = new ParameterElementImpl(pm);
				addParameter(e);
			}
		}
		
		if (proc.isSetActivities()) {
			ActivitiesDocument.Activities acts = proc.getActivities();
			int count = acts.sizeOfActivityArray();
			for (int i = 0; i < count; i++) {
				ActivityDocument.Activity act = acts.getActivityArray(i);
				ActivityElement e = new ActivityElementImpl(act);
				addActivity(e);
			}
		}
		
		if (proc.isSetTransitions()) {
			TransitionsDocument.Transitions trs = proc.getTransitions();
			int count = trs.sizeOfTransitionArray();
			for (int i = 0; i < count; i++) {
				TransitionDocument.Transition tr = trs.getTransitionArray(i);
				TransitionElement e = new TransitionElementImpl(tr, getActivities());
				addTransition(e);
			}
		}
	}

	/**
	 * IDを設定する。
	 * 
	 * @param id	ID
	 */	
	private void setID(String id)
	{
		this.id = id;
	}

	/**
	 * アクセスレベルを設定する。
	 * 
	 * @param i	アクセスレベル
	 */
	private void setAccessLevel(int i)
	{
		accessLevel = i;
	}

	/**
	 * アクティビティを追加する。
	 * 
	 * @param act	アクティビティ
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#addActivity(jp.co.argo21.nautica.tool.wfd.element.ActivityElement)
	 */
	public void addActivity(ActivityElement act)
	{
		this.activities.add(act);
	}

	/**
	 * パラメータを追加する。
	 * 
	 * @param param	パラメータ
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#addParameter(jp.co.argo21.nautica.tool.wfd.element.ParameterElement)
	 */
	public void addParameter(ParameterElement param)
	{
		this.parameters.add(param);
	}

	/**
	 * 通知者を追加する。
	 * 
	 * @param r	通知者
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#addResponsible(java.lang.String)
	 */
	public void addResponsible(String r)
	{
		this.responsibles.add(r);
	}

	/**
	 * 遷移を追加する。
	 * 
	 * @param tr	遷移
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#addTransition(jp.co.argo21.nautica.tool.wfd.element.TransitionElement)
	 */
	public void addTransition(TransitionElement tr)
	{
		this.transitions.add(tr);
	}

	/**
	 * アクティビティを削除する。
	 * 
	 * @param act	アクティビティ
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#removeActivity(jp.co.argo21.nautica.tool.wfd.element.ActivityElement)
	 */
	public void removeActivity(ActivityElement act)
	{
		this.activities.remove(act);
	}

	/**
	 * パラメータを削除する。
	 * 
	 * @param param	パラメータ
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#removeParameter(jp.co.argo21.nautica.tool.wfd.element.ParameterElement)
	 */
	public void removeParameter(ParameterElement param)
	{
		this.parameters.remove(param);
	}

	/**
	 * 通知者を削除する。
	 * 
	 * @param r	通知者
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#removeResponsible(java.lang.String)
	 */
	public void removeResponsible(String r)
	{
		this.responsibles.remove(r);
	}

	/**
	 * 遷移を削除する。
	 * 
	 * @param tr	遷移
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#removeTransition(jp.co.argo21.nautica.tool.wfd.element.TransitionElement)
	 */
	public void removeTransition(TransitionElement tr)
	{
		this.transitions.remove(tr);
	}

	/**
	 * 説明を返す。
	 * 
	 * @return 説明
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getDescription()
	 */
	public String getDescription()
	{
		return description;
	}

	/**
	 * IDを返す。
	 * 
	 * @return ID
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getID()
	 */
	public String getID()
	{
		return id;
	}

	/**
	 * 名称を返す。
	 * 
	 * @return 名称
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getName()
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * 説明を設定する。
	 * 
	 * @param desc 説明
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#setDescription(java.lang.String)
	 */
	public void setDescription(String desc)
	{
		this.description = desc;
	}

	/**
	 * 名称を設定する。
	 * 
	 * @param name 名称
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#setName(java.lang.String)
	 */
	public void setName(String name)
	{
		this.name = name;
	}

	/**
	 * アクティビティ一覧を返す。
	 * 
	 * @return	アクティビティ一覧
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#getActivities()
	 */
	public List<ActivityElement> getActivities() {
		return activities;
	}

	/**
	 * 期間単位を返す。
	 * 
	 * @return	期間単位
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#getDurationUnit()
	 */
	public int getDurationUnit()
	{
		return durationUnit;
	}

	/**
	 * 実行期限を返す。
	 * 
	 * @return	実行期限
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#getLimit()
	 */
	public int getLimit()
	{
		return limit;
	}

	/**
	 * 論理座標高さを返す。
	 * 
	 * @return	論理座標高さ
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#getLogicalHeight()
	 */
	public int getLogicalHeight()
	{
		return logicalHeight;
	}

	/**
	 * 論理座標幅を返す。
	 * 
	 * @return	論理座標幅
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#getLogicalWidth()
	 */
	public int getLogicalWidth()
	{
		return logicalWidth;
	}

	/**
	 * パラメータ一覧を返す。
	 * 
	 * @return	パラメータ一覧
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#getParameters()
	 */
	public List<ParameterElement> getParameters()
	{
		return parameters;
	}

	/**
	 * 遷移一覧を返す。
	 * 
	 * @return	遷移一覧
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#getTransitions()
	 */
	public List<TransitionElement> getTransitions()
	{
		return transitions;
	}

	/**
	 * 期間単位を設定する。
	 * 
	 * @param i	期間単位
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#setDurationUnit(int)
	 */
	public void setDurationUnit(int i)
	{
		durationUnit = i;
	}

	/**
	 * 実行期限を設定する。
	 * 
	 * @param i	実行期限
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#setLimit(int)
	 */
	public void setLimit(int i)
	{
		limit = i;
	}

	/**
	 * 論理座標高さを設定する。
	 * 
	 * @param i	論理座標高さ
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#setLogicalHeight(int)
	 */
	public void setLogicalHeight(int i)
	{
		logicalHeight = i;
	}

	/**
	 * 論理座標幅を設定する。
	 * 
	 * @param i	論理座標幅
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#setLogicalWidth(int)
	 */
	public void setLogicalWidth(int i)
	{
		logicalWidth = i;
	}

	/**
	 * 通知者一覧を返す。
	 * 
	 * @return	通知者一覧
	 * @see jp.co.argo21.nautica.tool.wfd.element.ProcessElement#getResponsibles()
	 */
	public List<String> getResponsibles()
	{
		return responsibles;
	}

	/**
	 * アクセスレベルを返す。
	 * 
	 * @return	アクセスレベル
	 */
	public int getAccessLevel()
	{
		return accessLevel;
	}

	/**
	 * ツールボックス文字列を返す。
	 * 
	 * @param rm	リソースマネージャ
	 * @return	文字列
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getTooltipText(jp.co.argo21.nautica.commons.util.ResourceManager)
	 */
	public String getTooltipText(ResourceManager rm)
	{
		StringBuffer s = new StringBuffer();
		s = s.append("<html>");
		s = s.append("<b>").append(getName()).append("</b>");
		s = s.append("<table border>");
		s = s.append("<tr><td width=100>")
			.append(rm.getResource("ProcessElement.label.0"))
			.append("</td><td width=150>")
			.append(getID())
			.append("</td></tr>");
		s = s.append("<tr><td width=100>")
			.append(rm.getResource("ProcessElement.label.3"))
			.append("</td><td width=150>");
		int a = getAccessLevel();
		if (a == ProcessElement.PUBLIC) {
			s = s.append(rm.getResource("ProcessElement.access.0"));
		} else {
			s = s.append(rm.getResource("ProcessElement.access.1"));
		}
		s = s.append("</td></tr>");
		s = s.append("<tr><td width=100>")
			.append(rm.getResource("ProcessElement.label.2"))
			.append("</td><td width=150>")
			.append(getLimit())
			.append("</td></tr>");
		s = s.append("<tr><td width=100>")
			.append(rm.getResource("ProcessElement.label.6"))
			.append("</td><td width=150>");
		int u = getDurationUnit();
		if (u == ProcessElement.YEAR) {
			s = s.append(rm.getResource("ProcessElement.unit.0"));
		} else if (u == ProcessElement.MONTH) {
			s = s.append(rm.getResource("ProcessElement.unit.1"));
		} else if (u == ProcessElement.DAY) {
			s = s.append(rm.getResource("ProcessElement.unit.2"));
		} else if (u == ProcessElement.HOUR) {
			s = s.append(rm.getResource("ProcessElement.unit.3"));
		} else if (u == ProcessElement.MINUTE) {
			s = s.append(rm.getResource("ProcessElement.unit.4"));
		} else {
			s = s.append(rm.getResource("ProcessElement.unit.2"));
		}
		s = s.append("</td></tr>");
		s = s.append("<tr><td width=100>")
			.append(rm.getResource("ProcessElement.label.4"))
			.append("</td><td width=150>")
			.append(getDescription())
			.append("</td></tr>");
		s = s.append("</table>");
		s = s.append("</html>");
		return s.toString();
	}

	/**
	 * 印刷用文字列を返す。
	 * 
	 * @param rm	リソースマネージャ
	 * @return	文字列
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getPrintText(jp.co.argo21.nautica.commons.util.ResourceManager)
	 */
	public String getPrintText(ResourceManager rm)
	{	
		StringBuffer s = new StringBuffer();
		return s.toString();
	}

	/**
	 * 指定された要素を使用中かどうかを返す。
	 *
	 * @param elem 要素
	 * @return 使用中ならtrue
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#isUsed(jp.co.argo21.nautica.tool.wfd.element.Element)
	 */
	public boolean isUsed(Element elem)
	{
		if (elem == null) return false;
		
		for (Element e : activities) {
			boolean isUsed = e.isUsed(elem);
			if (isUsed) return true;
		}
		
		for (Element e : transitions) {
			boolean isUsed = e.isUsed(elem);
			if (isUsed) return true;
		}
		return false;
	}
}
