/* $Id: PackageElementImpl.java,v 1.2 2007/12/11 09:54:58 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfd.element.impl;

import java.util.ArrayList;
import java.util.List;

import org.wfmc.x2002.xpdl10.ApplicationDocument;
import org.wfmc.x2002.xpdl10.ApplicationsDocument;
import org.wfmc.x2002.xpdl10.DataFieldDocument;
import org.wfmc.x2002.xpdl10.DataFieldsDocument;
import org.wfmc.x2002.xpdl10.PackageDocument;
import org.wfmc.x2002.xpdl10.PackageHeaderDocument;
import org.wfmc.x2002.xpdl10.ParticipantDocument;
import org.wfmc.x2002.xpdl10.ParticipantsDocument;
import org.wfmc.x2002.xpdl10.RedefinableHeaderDocument;
import org.wfmc.x2002.xpdl10.WorkflowProcessDocument;
import org.wfmc.x2002.xpdl10.WorkflowProcessesDocument;

import jp.co.argo21.nautica.commons.util.IDGenerator;
import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.commons.util.StringUtils;
import jp.co.argo21.nautica.tool.wfd.element.ApplicationElement;
import jp.co.argo21.nautica.tool.wfd.element.Element;
import jp.co.argo21.nautica.tool.wfd.element.PackageElement;
import jp.co.argo21.nautica.tool.wfd.element.ParticipantElement;
import jp.co.argo21.nautica.tool.wfd.element.ProcessElement;
import jp.co.argo21.nautica.tool.wfd.element.VariableElement;

/**
 * 
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class PackageElementImpl implements PackageElement
{
	/** 未初期化時の仮ID */
	private static final String UNINITIALIZED_ID = "--UNINITIALIZED--"; 

	/** ID */
	private String id;
	/** 名称 */
	private String name;
	/** 備考 */
	private String description;
	/** 作成日 */
	private String createdDate;
	/** バージョン */
	private String version;
	/** 定義者 */
	private String author;
	/** ワークフロー参加者一覧 */
	private List<ParticipantElement> participants = new ArrayList<ParticipantElement>();
	/** アプリケーション一覧 */
	private List<ApplicationElement> applications = new ArrayList<ApplicationElement>();
	/** プロセス変数一覧 */
	private List<VariableElement> variables = new ArrayList<VariableElement>();
	/** プロセス一覧 */
	private List<ProcessElement> processes = new ArrayList<ProcessElement>();

	/**
	 * パッケージを生成する。
	 * 
	 * @param n	名称
	 */
	public PackageElementImpl(String n)
	{
		setID(IDGenerator.generate(ID_PREFIX));
		setName(n);
		setDescription("");
		setCreatedDate("");
		setVersion("1.0");
		setAuthor(System.getProperty("user.name"));
	}

	/**
	 * パッケージを生成する。
	 * 
	 * @param doc	XmlObject
	 */
	public PackageElementImpl(PackageDocument doc)
	{
		PackageDocument.Package p = doc.getPackage();
		String id = p.getId();
		if (StringUtils.checkNull(id) || id.equals(UNINITIALIZED_ID)) {
			setID(IDGenerator.generate(ID_PREFIX));
		} else {
			setID(id);
		}
		setName(p.getName());
		
		PackageHeaderDocument.PackageHeader phd = p.getPackageHeader();
		setDescription(phd.getDescription());
		setCreatedDate(phd.getCreated());
		
		if (p.isSetRedefinableHeader()) {
			RedefinableHeaderDocument.RedefinableHeader rhd
				= p.getRedefinableHeader();
			setAuthor(rhd.getAuthor());
			setVersion(rhd.getVersion());
		}
		
		if (p.isSetApplications()) {
			ApplicationsDocument.Applications apps = p.getApplications();
			int count = apps.sizeOfApplicationArray();
			for (int i = 0; i < count; i++) {
				ApplicationDocument.Application app = apps.getApplicationArray(i);
				ApplicationElement e = new ApplicationElementImpl(app);
				addApplication(e);
			}
		}
		
		if (p.isSetParticipants()) {
			ParticipantsDocument.Participants pts = p.getParticipants();
			int count = pts.sizeOfParticipantArray();
			for (int i = 0; i < count; i++) {
				ParticipantDocument.Participant pt = pts.getParticipantArray(i);
				ParticipantElement e = new ParticipantElementImpl(pt);
				addParticipant(e);
			}
		}
		
		if (p.isSetDataFields()) {
			DataFieldsDocument.DataFields dfs = p.getDataFields();
			int count = dfs.sizeOfDataFieldArray();
			for (int i = 0; i < count; i++) {
				DataFieldDocument.DataField df = dfs.getDataFieldArray(i);
				VariableElement e = new VariableElementImpl(df);
				addVariable(e);
			}
		}
		
		if (p.isSetWorkflowProcesses()) {
			WorkflowProcessesDocument.WorkflowProcesses wps = p.getWorkflowProcesses();
			int count = wps.sizeOfWorkflowProcessArray();
			for (int i = 0; i < count; i++) {
				WorkflowProcessDocument.WorkflowProcess proc = wps.getWorkflowProcessArray(i);
				ProcessElement e = new ProcessElementImpl(proc);
				addProcess(e);
			}
		}
	}

	/**
	 * IDを設定する。
	 * 
	 * @param id	ID
	 */	
	private void setID(String id)
	{
		this.id = id;
	}

	/**
	 * アプリケーションを追加する。
	 * 
	 * @param app	アプリケーション
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#addApplication(jp.co.argo21.nautica.tool.wfd.element.ApplicationElement)
	 */
	public void addApplication(ApplicationElement app)
	{
		this.applications.add(app);
	}

	/**
	 * ワークフロー参加者を追加する。
	 * 
	 * @param part	ワークフロー参加者
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#addParticipant(jp.co.argo21.nautica.tool.wfd.element.ParticipantElement)
	 */
	public void addParticipant(ParticipantElement part)
	{
		this.participants.add(part);
	}

	/**
	 * プロセスを追加する。
	 * 
	 * @param proc	プロセス
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#addProcess(jp.co.argo21.nautica.tool.wfd.element.ProcessElement)
	 */
	public void addProcess(ProcessElement proc)
	{
		this.processes.add(proc);
	}

	/**
	 * プロセス変数を追加する。
	 * 
	 * @param var	プロセス変数
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#addVariable(jp.co.argo21.nautica.tool.wfd.element.VariableElement)
	 */
	public void addVariable(VariableElement var)
	{
		this.variables.add(var);
	}

	/**
	 * アプリケーションを削除する。
	 * 
	 * @param app	アプリケーション
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#removeApplication(jp.co.argo21.nautica.tool.wfd.element.ApplicationElement)
	 */
	public void removeApplication(ApplicationElement app)
	{
		this.applications.remove(app);
	}

	/**
	 * ワークフロー参加者を削除する。
	 * 
	 * @param part	ワークフロー参加者
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#removeParticipant(jp.co.argo21.nautica.tool.wfd.element.ParticipantElement)
	 */
	public void removeParticipant(ParticipantElement part)
	{
		this.participants.remove(part);
	}

	/**
	 * プロセスを削除する。
	 * 
	 * @param proc	プロセス
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#removeProcess(jp.co.argo21.nautica.tool.wfd.element.ProcessElement)
	 */
	public void removeProcess(ProcessElement proc)
	{
		this.processes.remove(proc);
	}

	/**
	 * プロセス変数を削除する。
	 * 
	 * @param var	プロセス変数
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#removeVariable(jp.co.argo21.nautica.tool.wfd.element.VariableElement)
	 */
	public void removeVariable(VariableElement var)
	{
		this.variables.remove(var);
	}

	/**
	 * 説明を返す。
	 * 
	 * @return 説明
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getDescription()
	 */
	public String getDescription()
	{
		return description;
	}

	/**
	 * IDを返す。
	 * 
	 * @return ID
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getID()
	 */
	public String getID()
	{
		return id;
	}

	/**
	 * 名称を返す。
	 * 
	 * @return 名称
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getName()
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * 説明を設定する。
	 * 
	 * @param desc 説明
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#setDescription(java.lang.String)
	 */
	public void setDescription(String desc)
	{
		this.description = desc;
	}

	/**
	 * 名称を設定する。
	 * 
	 * @param name 名称
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#setName(java.lang.String)
	 */
	public void setName(String name)
	{
		this.name = name;
	}

	/**
	 * アプリケーション一覧を返す。
	 * 
	 * @return	アプリケーション一覧
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#getApplications()
	 */
	public List<ApplicationElement> getApplications()
	{
		return applications;
	}

	/**
	 * 作成日を返す。
	 * 
	 * @return	作成日
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#getCreatedDate()
	 */
	public String getCreatedDate()
	{
		return createdDate;
	}

	/**
	 * ワークフロー参加者一覧を返す。
	 * 
	 * @return	ワークフロー参加者一覧
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#getParticipants()
	 */
	public List<ParticipantElement> getParticipants()
	{
		return participants;
	}

	/**
	 * プロセス一覧を返す。
	 * 
	 * @return	プロセス一覧
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#getProcesses()
	 */
	public List<ProcessElement> getProcesses()
	{
		return processes;
	}

	/**
	 * プロセス変数一覧を返す。
	 * 
	 * @return	プロセス変数一覧
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#getVariables()
	 */
	public List<VariableElement> getVariables()
	{
		return variables;
	}

	/**
	 * バージョンを返す。
	 * 
	 * @return	バージョン
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#getVersion()
	 */
	public String getVersion()
	{
		return version;
	}

	/**
	 * 作成日を設定する。
	 * 
	 * @param s	作成日
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#setCreatedDate(java.lang.String)
	 */
	public void setCreatedDate(String s)
	{
		createdDate = s;
	}

	/**
	 * バージョンを設定する。
	 * 
	 * @param s	バージョン
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#setVersion(java.lang.String)
	 */
	public void setVersion(String s)
	{
		version = s;
	}

	/**
	 * 定義者を返す。
	 * 
	 * @return	定義者
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#getAuthor()
	 */
	public String getAuthor()
	{
		return author;
	}

	/**
	 * 定義者を設定する。
	 * 
	 * @param s	定義者
	 * @see jp.co.argo21.nautica.tool.wfd.element.PackageElement#setAuthor(java.lang.String)
	 */
	public void setAuthor(String s)
	{
		author = s;
	}

	/**
	 * ツールボックス文字列を返す。
	 * 
	 * @param rm	リソースマネージャ
	 * @return	文字列
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getTooltipText(jp.co.argo21.nautica.commons.util.ResourceManager)
	 */
	public String getTooltipText(ResourceManager rm)
	{
		StringBuffer s = new StringBuffer();
		s = s.append("<html>");
		s = s.append("<b>").append(getName()).append("</b>");
		s = s.append("<table border>");
		s = s.append("<tr><td width=100>")
			.append(rm.getResource("PackageElement.label.0"))
			.append("</td><td width=150>")
			.append(getID())
			.append("</td></tr>");
		s = s.append("<tr><td width=100>")
			.append(rm.getResource("PackageElement.label.5"))
			.append("</td><td width=150>")
			.append(getDescription())
			.append("</td></tr>");
		s = s.append("</table>");
		s = s.append("</html>");
		return s.toString();
	}

	/**
	 * 印刷用文字列を返す。
	 * 
	 * @param rm	リソースマネージャ
	 * @return	文字列
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getPrintText(jp.co.argo21.nautica.commons.util.ResourceManager)
	 */
	public String getPrintText(ResourceManager rm)
	{	
		StringBuffer s = new StringBuffer();
		return s.toString();
	}

	/**
	 * 指定された要素を使用中かどうかを返す。
	 *
	 * @param elem 要素
	 * @return 使用中ならtrue
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#isUsed(jp.co.argo21.nautica.tool.wfd.element.Element)
	 */
	public boolean isUsed(Element elem)
	{
		if (elem == null) return false;
		
		for (Element e : processes) {
			boolean isUsed = e.isUsed(elem);
			if (isUsed) return true;
		}
		return false;
	}
}
