/* $Id: ActivityElementImpl.java,v 1.2 2007/12/11 09:54:58 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfd.element.impl;

import java.util.ArrayList;
import java.util.List;

import org.wfmc.x2002.xpdl10.ActivityDocument;
import org.wfmc.x2002.xpdl10.ActualParametersDocument;
import org.wfmc.x2002.xpdl10.BlockActivityDocument;
import org.wfmc.x2002.xpdl10.ExtendedAttributeDocument;
import org.wfmc.x2002.xpdl10.ExtendedAttributesDocument;
import org.wfmc.x2002.xpdl10.ImplementationDocument;
import org.wfmc.x2002.xpdl10.SubFlowDocument;
import org.wfmc.x2002.xpdl10.ToolDocument;

import jp.co.argo21.nautica.commons.util.IDGenerator;
import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.commons.util.StringUtils;
import jp.co.argo21.nautica.tool.wfd.ActivityStateFactory;
import jp.co.argo21.nautica.tool.wfd.element.ActivityElement;
import jp.co.argo21.nautica.tool.wfd.element.ActivityState;
import jp.co.argo21.nautica.tool.wfd.element.Element;

/**
 * アクティビティ
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class ActivityElementImpl implements ActivityElement
{
	/** 未初期化時の仮ID */
	private static final String UNINITIALIZED_ID = "--UNINITIALIZED--"; 

	/** ID */
	private String id;
	/** 名称 */
	private String name;
	/** 備考 */
	private String description;
	/** 実行期限 */
	private int limit;
	/** 論理X座標 */
	private int logicalX;
	/** 論理Y座標 */
	private int logicalY;
	/** ブロックID */
	private String blockID;
	/** 所属ブロックID */
	private String affiliateBID;
	/** 親ブロックID */
	private String parentBID;
	/** アプリケーションID */
	private String applicationID;
	/** サブプロセスID */
	private String subprocessID;
	/** 実行許可者ID */
	private String performerID;
	/** サブプロセス実行時の同期状態 */
	private int sync;
	/** アプリケーションパラメータ一覧 */
	private List<String> applicationParameters = new ArrayList<String>();
	/** サブプロセスパラメータ一覧 */
	private List<String> subprocessParameters = new ArrayList<String>();
	/** エンジン名 */
	private String engineName;
	/** アクティビティ状態 */
	private ActivityState state;

	/**
	 * アクティビティを生成する。
	 * 
	 * @param state	アクティビティ状態
	 */
	public ActivityElementImpl(ActivityState state)
	{
		setID(IDGenerator.generate(ID_PREFIX));
		setName(getID());
		setDescription("");
		setLimit(0);
		setLogicalX(0);
		setLogicalY(0);
		setExecution(SYNCHR);
		setActivityState(state);
	}

	/**
	 * アクティビティを生成する。
	 * 
	 * @param act	XmlObject
	 */
	public ActivityElementImpl(ActivityDocument.Activity act)
	{
		String id = act.getId();
		if (StringUtils.checkNull(id) || id.equals(UNINITIALIZED_ID)) {
			setID(IDGenerator.generate(ID_PREFIX));
		} else {
			setID(id);
		}
		setName(act.getName());
		setDescription(act.getDescription());

		try {
			setLimit(Integer.parseInt(act.getLimit()));
		} catch (Exception ex) {
			setLimit(0);
		}
	
		if (act.isSetBlockActivity()) {
			BlockActivityDocument.BlockActivity ba = act.getBlockActivity();
			setBlockID(ba.getBlockId());
		}
		
		if (act.isSetPerformer()) {
			setPerformerID(act.getPerformer());
		}
		
		if (act.isSetImplementation()) {
			ImplementationDocument.Implementation imp = act.getImplementation();
			if (imp.isSetSubFlow()) {
				SubFlowDocument.SubFlow sf = imp.getSubFlow();
				setSubprocessID(sf.getId());
				setExecution(sf.getExecution().intValue());
				if (sf.isSetActualParameters()) {
					ActualParametersDocument.ActualParameters aps = sf.getActualParameters();
					int count = aps.sizeOfActualParameterArray();
					for (int i = 0; i < count; i++) {
						String pid = aps.getActualParameterArray(i);
						addSubprocessParameterID(pid);
					}
				}
			}

			if (imp.sizeOfToolArray() == 1) {
				ToolDocument.Tool tool = imp.getToolArray(0);
				setApplicationID(tool.getId());
				if (tool.isSetActualParameters()) {
					ActualParametersDocument.ActualParameters aps = tool.getActualParameters();
					int count = aps.sizeOfActualParameterArray();
					for (int i = 0; i < count; i++) {
						String aid = aps.getActualParameterArray(i);
						addApplicationParameterID(aid);
					}
				}
			}
		}

		if (act.isSetExtendedAttributes()) {
			ExtendedAttributesDocument.ExtendedAttributes eas = act.getExtendedAttributes();
			int count = eas.sizeOfExtendedAttributeArray();
			for (int i = 0; i < count; i++) {
				ExtendedAttributeDocument.ExtendedAttribute ea = eas.getExtendedAttributeArray(i);
				String n = ea.getName();
				String v = ea.getValue();
				if (n.equals(LOGICAL_X)) {
					setLogicalX(Integer.parseInt(v));
				} else if (n.equals(LOGICAL_Y)) {
					setLogicalY(Integer.parseInt(v));
				} else if (n.equals(AFFILIATE_BID)) {
					setAffiliateBID(v);
				} else if (n.equals(PARENT_BID)) {
					setParentBID(v);
				} else if (n.equals(ACTIVITY_TYPE)) {
					ActivityStateFactory factory = ActivityStateFactory.getInstance();
					state = factory.createElement(v);
				} else if (n.equals(ENGINE_NANE)) {
					setEngineName(v);
				}
			}
		}
	}

	/**
	 * IDを設定する。
	 * 
	 * @param id	ID
	 */	
	private void setID(String id)
	{
		this.id = id;
	}

	/**
	 * アクティビティ状態を設定する。
	 * 
	 * @param state	アクティビティ状態
	 */
	private void setActivityState(ActivityState state)
	{
		this.state = state;
	}


	/**
	 * アクティビティ状態を返す。
	 * 
	 * @return	アクティビティ状態
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#getActivityState()
	 */
	public ActivityState getActivityState()
	{
		return state;
	}

	/**
	 * ブロックIDを返す。
	 * 
	 * @return ブロックID
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#getBlockID()
	 */
	public String getBlockID()
	{
		return blockID;
	}

	/**
	 * 実行期限を返す。
	 * 
	 * @return	実行期限
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#getLimit()
	 */
	public int getLimit()
	{
		return limit;
	}

	/**
	 * 論理X座標を返す。
	 * 
	 * @return 論理X座標
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#getLogicalX()
	 */
	public int getLogicalX()
	{
		return logicalX;
	}

	/**
	 * 論理Y座標を返す。
	 * 
	 * @return 論理Y座標
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#getLogicalY()
	 */
	public int getLogicalY()
	{
		return logicalY;
	}

	/**
	 * アクティビティ型を返す。
	 * 
	 * @return	アクティビティ型
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#getType()
	 */
	public String getType()
	{
		return state.getType();
	}

	/**
	 * ブロックIDを設定する。
	 * 
	 * @param id ブロックID
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#setBlockID(java.lang.String)
	 */
	public void setBlockID(String id)
	{
		this.blockID = id;
	}

	/**
	 * 実行期限を設定する。単位は所属するプロセスのDurationUnitを使用する。
	 * 
	 * @param limit 待機時間の上限値
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#setLimit(int)
	 */
	public void setLimit(int limit)
	{
		this.limit = limit;
	}

	/**
	 * 論理X座標を設定する。
	 * 
	 * @param x 論理X座標
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#setLogicalX(int)
	 */
	public void setLogicalX(int x)
	{
		this.logicalX = x;
	}

	/**
	 * 論理Y座標を設定する。
	 * 
	 * @param y 論理Y座標
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#setLogicalY(int)
	 */
	public void setLogicalY(int y)
	{
		this.logicalY = y;
	}

	/**
	 * 説明を返す。
	 * 
	 * @return 説明
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getDescription()
	 */
	public String getDescription()
	{
		return description;
	}

	/**
	 * IDを返す。
	 * 
	 * @return ID
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getID()
	 */
	public String getID()
	{
		return id;
	}

	/**
	 * アクティビティ名称を返す。
	 * 
	 * @return アクティビティ名称
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getName()
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * 説明を設定する。
	 * 
	 * @param desc 説明
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#setDescription(java.lang.String)
	 */
	public void setDescription(String desc)
	{
		this.description = desc;
	}

	/**
	 * アクティビティ名称を設定する。
	 * 
	 * @param name アクティビティ名称
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#setName(java.lang.String)
	 */
	public void setName(String name)
	{
		this.name = name;
	}

	/**
	 * アプリケーションパラメータを追加する。
	 * 
	 * @param id	アプリケーションパラメータ
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#addApplicationParameterID(java.lang.String)
	 */
	public void addApplicationParameterID(String id)
	{
		this.applicationParameters.add(id);
	}

	/**
	 * サブプロセスパラメータを追加する。
	 * 
	 * @param id	サブプロセスパラメータ
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#addSubprocessParameterID(java.lang.String)
	 */
	public void addSubprocessParameterID(String id)
	{
		this.subprocessParameters.add(id);
	}

	/**
	 * アプリケーションIDを返す。
	 * 
	 * @return	アプリケーションID
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#getApplicationID()
	 */
	public String getApplicationID()
	{
		return applicationID;
	}

	/**
	 * アプリケーションパラメータ一覧を返す。
	 * 
	 * @return	アプリケーションパラメータ一覧
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#getApplicationParameters()
	 */
	public List<String> getApplicationParameters()
	{
		return applicationParameters;
	}

	/**
	 * サブプロセスIDを返す。
	 * 
	 * @return	サブプロセスID
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#getSubprocessID()
	 */
	public String getSubprocessID()
	{
		return subprocessID;
	}

	/**
	 * サブプロセスパラメータ一覧を返す。
	 * 
	 * @return	サブプロセスパラメータ一覧
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#getSubprocessParameters()
	 */
	public List<String> getSubprocessParameters()
	{
		return subprocessParameters;
	}

	/**
	 * アプリケーションパラメータを削除する。
	 * 
	 * @param id	アプリケーションパラメータ
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#removeApplicationParameterID(java.lang.String)
	 */
	public void removeApplicationParameterID(String id)
	{
		this.applicationParameters.remove(id);
	}

	/**
	 * サブプロセスパラメータを削除する。
	 * 
	 * @param id	サブプロセスパラメータ
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#removeSubprocessParameterID(java.lang.String)
	 */
	public void removeSubprocessParameterID(String id)
	{
		this.subprocessParameters.remove(id);
	}

	/**
	 * アプリケーションIDを設定する。
	 * 
	 * @param id	アプリケーションID
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#setApplicationID(java.lang.String)
	 */
	public void setApplicationID(String id)
	{
		this.applicationID = id;
	}

	/**
	 * サブプロセスIDを設定する。
	 * 
	 * @param id	サブプロセスID
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#setSubprocessID(java.lang.String)
	 */
	public void setSubprocessID(String id)
	{
		this.subprocessID = id;
	}

	/**
	 * サブプロセス実行時の同期状態を返す。
	 * 
	 * @return 同期状態
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#getExecution()
	 */
	public int getExecution()
	{
		return sync;
	}

	/**
	 * サブプロセス実行時の同期状態を設定する。
	 * 
	 * @param sync 同期状態
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#setExecution(int)
	 */
	public void setExecution(int sync)
	{
		this.sync = sync;
	}
	
	/**
	 * 複製する。
	 * 
	 * @return	複製したアクティビティ
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#copy()
	 */
	public ActivityElement copy()
	{
		ActivityElement copy = new ActivityElementImpl(getActivityState());
		copy.setName(getName());
		copy.setDescription(getDescription());
		copy.setLimit(getLimit());
		copy.setLogicalX(getLogicalX());
		copy.setLogicalY(getLogicalY());
		copy.setExecution(getExecution());
		return copy;
	}

	/**
	 * 所属ブロックIDを返す。
	 * 
	 * @return	所属ブロックID
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#getAffiliateBID()
	 */
	public String getAffiliateBID()
	{
		return affiliateBID;
	}

	/**
	 * 親ブロックIDを返す。
	 * 
	 * @return	親ブロックID
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#getParentBID()
	 */
	public String getParentBID()
	{
		return parentBID;
	}

	/**
	 * 所属ブロックIDを設定する。
	 * 
	 * @param id	所属ブロックID
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#setAffiliateBID(java.lang.String)
	 */
	public void setAffiliateBID(String id)
	{
		affiliateBID = id;
	}

	/**
	 * 親ブロックIDを設定する。
	 * 
	 * @param id	親ブロックID
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#setParentBID(java.lang.String)
	 */
	public void setParentBID(String id)
	{
		parentBID = id;
	}

	/**
	 * 実行許可者IDを返す。
	 * 
	 * @return	実行許可者ID
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#getPerformerID()
	 */
	public String getPerformerID()
	{
		return performerID;
	}

	/**
	 * 実行許可者IDを設定する。
	 * 
	 * @param id	実行許可者ID
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#setPerformerID(java.lang.String)
	 */
	public void setPerformerID(String id)
	{
		performerID = id;
	}

	/**
	 * エンジン名を返す。
	 *
	 * @return エンジン名
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#getEngineName()
	 */
	public String getEngineName()
	{
		return engineName;
	}

	/**
	 * エンジン名を設定する。
	 * 
	 * @param engineName エンジン名
	 * @see jp.co.argo21.nautica.tool.wfd.element.ActivityElement#setEngineName(java.lang.String)
	 */
	public void setEngineName(String engineName)
	{
		this.engineName = engineName;
	}

	/**
	 * ツールボックス文字列を返す。
	 * 
	 * @param rm	リソースマネージャ
	 * @return	文字列
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getTooltipText(jp.co.argo21.nautica.commons.util.ResourceManager)
	 */
	public String getTooltipText(ResourceManager rm)
	{
		StringBuffer s = new StringBuffer();
		s = s.append("<html>");
		s = s.append("<b>").append(getName()).append("</b>");
		s = s.append("<table border>");
		s = s.append("<tr><td width=100>")
			.append(rm.getResource("ActivityElement.label.0"))
			.append("</td><td width=150>")
			.append(getID())
			.append("</td></tr>");
		s = s.append("<tr><td width=100>")
			.append(rm.getResource("ActivityElement.label.4"))
			.append("</td><td width=150>")
			.append(getDescription())
			.append("</td></tr>");
		s = s.append("</table>");
		s = s.append("</html>");
		return s.toString();
	}

	/**
	 * 印刷用文字列を返す。
	 * 
	 * @param rm	リソースマネージャ
	 * @return	文字列
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#getPrintText(jp.co.argo21.nautica.commons.util.ResourceManager)
	 */
	public String getPrintText(ResourceManager rm)
	{	
		StringBuffer s = new StringBuffer();
		return s.toString();
	}

	/**
	 * 指定された要素を使用中かどうかを返す。
	 *
	 * @param elem 要素
	 * @return 使用中ならtrue
	 * @see jp.co.argo21.nautica.tool.wfd.element.Element#isUsed(jp.co.argo21.nautica.tool.wfd.element.Element)
	 */
	public boolean isUsed(Element elem)
	{
		if (elem == null) return false;
		
		String eid = elem.getID();
		ActivityState as = getActivityState();
		if (as instanceof InteractiveActivityState) {
			if (performerID != null) {
				if (performerID.equals(eid)) return true;
			}
		} else if (as instanceof AutomaticActivityState) {
			if (performerID != null) {
				if (performerID.equals(eid)) return true;
			}
			if (applicationID != null) {
				if (applicationID.equals(eid)) return true;
				
				for (String id : applicationParameters) {
					if (id != null) {
						if (id.equals(eid)) return true;
					}
				}
			}
		} else if (as instanceof SubprocessActivityState) {
			if (subprocessID != null) {
				if (subprocessID.equals(eid)) return true;

				for (String id : subprocessParameters) {
					if (id != null) {
						if (id.equals(eid)) return true;
					}
				}
			}
		}
		return false;
	}
}
