/* $Id: WorkflowEngineHandler.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.wfmc;

/**
 * ワークフローエンジンハンドラは、ワークフローエンジンに対する
 * クライアントインタフェースである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public interface WorkflowEngineHandler {

	/**
	 * プロセスインスタンスの生成を行う。
	 *
	 * @param session セッションID
	 * @param pdid プロセス定義ID
	 * @param name プロセスインスタンス名称
	 * @return プロセスインスタンスID
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessDefinitionException 指定されたプロセス定義IDが有効でない場合
	 */
	public String createProcess(String session, String pdid, String name)
	throws InvalidSessionException, InvalidProcessDefinitionException;

	/**
	 * プロセスインスタンスの開始を行う。
	 * 開始した場合に、プロセスインスタンスIDを更新するような
	 * 実装も可能である。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @return 更新されたプロセスインスタンスID
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 */
	public String startProcess(String session, String pid)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidAttributeException;

	/**
	 * プロセスインスタンスの停止を行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 */
	public void terminateProcess(String session, String pid)
	throws InvalidSessionException, InvalidProcessInstanceException;

	/**
	 * 指定されたプロセスインスタンスが利用可能な状態一覧の取得を行う。
	 * 利用可能なフィルターはStateFilterのみとする。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param filter 検索用フィルター
	 * @return プロセスインスタンス状態一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 */
	public ProcessState[] getProcessStates(String session, 
			String pid, Filter filter)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidFilterException;


	/**
	 * プロセスインスタンス状態の変更を行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param state プロセスインスタンス状態
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidStateException 指定された状態が有効でない場合
	 * @throws TransitionNotAllowedException 状態の変更が許可されていない場合
	 */
	public void changeProcessState(String session,
			String pid, ProcessState state)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidStateException, TransitionNotAllowedException;

	/**
	 * プロセスインスタンス属性一覧の取得を行う。
	 * 利用可能なフィルターはNameFilter, AttributeFilter, GroupFilterとする。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param filter 検索用フィルター
	 * @return 属性一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 */
	public Attribute[] getProcessAttributes(String session,
			String pid, Filter filter)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidFilterException;

	/**
	 * プロセスインスタンス属性の属性値の取得を行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param name 属性名
	 * @return 属性
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 */
	public Attribute getProcessAttributeValue(String session,
			String pid, String name)
	throws InvalidSessionException, InvalidAttributeException;

	/**
	 * プロセスインスタンス属性の割り当てを行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param attr 属性
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 * @throws AttributeAssignmentFailedException 属性の割り当てに失敗した場合
	 */
	public void assignProcessAttribute(String session,
			String pid, Attribute attr)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidAttributeException, AttributeAssignmentFailedException;

	/**
	 * 指定されたアクティビティインスタンスが利用可能な状態一覧の取得を行う。
	 * 利用可能なフィルターはStateFilterのみとする。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param aid アクティビティインスタンスID
	 * @param filter 検索用フィルター
	 * @return アクティビティインスタンス状態一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidActivityInstanceException アクティビティインスタンスIDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 */
	public ActivityState[] getActivityStates(String session,
			String pid, String aid, Filter filter)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidActivityInstanceException, InvalidFilterException;

	/**
	 * アクティビティインスタンス状態の変更を行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param aid アクティビティインスタンスID
	 * @param state アクティビティインスタンス状態
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidActivityInstanceException アクティビティインスタンスIDが有効でない場合
	 * @throws InvalidStateException 指定された状態が有効でない場合
	 * @throws TransitionNotAllowedException 状態の変更が許可されていない場合
	 */
	public void changeActivityState(String session,
			String pid, String aid, ActivityState state)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidActivityInstanceException, InvalidStateException,
		TransitionNotAllowedException;

	/**
	 * アクティビティインスタンス属性一覧の取得を行う。
	 * 利用可能なフィルターはNameFilter, AttributeFilter, GroupFilterとする。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param aid アクティビティインスタンスID
	 * @param filter 検索用フィルター
	 * @return アクティビティインスタンス属性一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidActivityInstanceException アクティビティインスタンスIDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 */
	public Attribute[] getActivityAttributes(String session,
			String pid, String aid, Filter filter)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidActivityInstanceException, InvalidFilterException;

	/**
	 * アクティビティインスタンス属性の属性値の取得を行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param aid アクティビティインスタンスID
	 * @param name 属性名
	 * @return 属性
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 */
	public Attribute getActivityAttributeValue(String session,
			String pid, String aid, String name)
	throws InvalidSessionException, InvalidAttributeException;

	/**
	 * アクティビティインスタンス属性の割り当てを行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param aid アクティビティインスタンスID
	 * @param attr 属性
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidActivityInstanceException アクティビティインスタンスIDが有効でない場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 * @throws AttributeAssignmentFailedException 属性の割り当てに失敗した場合
	 */
	public void assignActivityAttribute(String session,
			String pid, String aid, Attribute attr)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidActivityInstanceException, InvalidAttributeException,
		AttributeAssignmentFailedException;

	/**
	 * プロセスインスタンス一覧の取得を行う。
	 * 利用可能なフィルターはNameFilter, StateFilter、GroupFilterとする。
	 *
	 * @param session セッションID
	 * @param filter 検索用フィルター
	 * @return プロセスの一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 */
	public Process[] getProcesses(String session, Filter filter)
	throws InvalidSessionException, InvalidFilterException;

	/**
	 * プロセスインスタンスの取得を行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @return プロセスインスタンス
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 */
	public Process getProcess(String session, String pid)
	throws InvalidSessionException, InvalidProcessInstanceException;

	/**
	 * アクティビティインスタンス一覧の取得を行う。
	 * 利用可能なフィルターはAttributeFilter, StateFilter、GroupFilterとする。
	 *
	 * @param session セッションID
	 * @param filter 検索用フィルター
	 * @return アクティビティインスタンス一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 */
	public Activity[] getActivities(String session, Filter filter)
	throws InvalidSessionException, InvalidFilterException;

	/**
	 * アクティビティインスタンス一覧の取得を行う。
	 * 利用可能なフィルターはAttributeFilter, StateFilter、GroupFilterとする。
	 * WfMC仕様ではpidはないが、これは誤りであろうと判断し、
	 * pidを引数に追加した。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param filter 検索用フィルター
	 * @return アクティビティインスタンス一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 */
	public Activity[] getActivities(String session, String pid, Filter filter)
	throws InvalidSessionException, InvalidFilterException;

	/**
	 * アクティビティインスタンスの取得を行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param aid アクティビティインスタンスID
	 * @return アクティビティインスタンス
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidActivityInstanceException アクティビティインスタンスIDが有効でない場合
	 */
	public Activity getActivity(String session, String pid, String aid)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidActivityInstanceException;
}
