/* $Id: WorkItemState.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.wfmc;

import java.util.ArrayList;
import java.util.List;

import jp.co.argo21.nautica.workflow.omg.WfState;

/**
 * 作業項目の状態を表すオブジェクトである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public final class WorkItemState implements WfState
{
	/** シリアルバージョンUID */
	private static final long serialVersionUID = 20050000000000001L;

	/** open.not_running.not_started */
	public static final WorkItemState OPEN_NOT_RUNNING_NOT_STARTED
		= new WorkItemState("open.not_running.not_started", 1);

	/** open.not_running.not_started */
	public static final WorkItemState OPEN_NOT_RUNNING_SUSPENDED
		= new WorkItemState("open.not_running.suspended", 2);

	/** open.running */
	public static final WorkItemState OPEN_RUNNING
		= new WorkItemState("open.running", 3);

	/** closed.completed */
	public static final WorkItemState CLOSED_COMPLETED
		= new WorkItemState("closed.completed", 4);

	/** closed.terminated */
	public static final WorkItemState CLOSED_TERMINATED
		= new WorkItemState("closed.terminated", 5);

	/** closed.aborted */
	public static final WorkItemState CLOSED_ABORTED
		= new WorkItemState("closed.aborted", 6);
	
	/** 状態文字列 */
	private String stateString;
	
	/** 状態値 */
	private int stateNum;
	
	/**
	 * 作業項目状態を返す。
	 *
	 * @param n 状態番号
	 * @return 作業項目状態
	 */
	public static WorkItemState getState(int n)
	{
		switch (n) {
		case 1: return OPEN_NOT_RUNNING_NOT_STARTED;
		case 2: return OPEN_NOT_RUNNING_SUSPENDED;
		case 3: return OPEN_RUNNING;
		case 4: return CLOSED_COMPLETED;
		case 5: return CLOSED_TERMINATED;
		case 6: return CLOSED_ABORTED;
		default: return null;
		}
	}
	
	/**
	 * 遷移可能な作業項目状態を返す。
	 *
	 * @param n 状態番号
	 * @return 作業項目状態
	 */
	public static WorkItemState[] getTransitableStates(int n)
	{
		List<WorkItemState> states = new ArrayList<WorkItemState>();
		switch (n) {
		case 1:
		case 2:
			states.add(OPEN_RUNNING);
			states.add(CLOSED_COMPLETED);
			states.add(CLOSED_TERMINATED);
			states.add(CLOSED_ABORTED);
			break;
		case 3:
			states.add(OPEN_NOT_RUNNING_SUSPENDED);
			states.add(CLOSED_COMPLETED);
			states.add(CLOSED_TERMINATED);
			states.add(CLOSED_ABORTED);
			break;
		}
		return (WorkItemState[])states.toArray(new WorkItemState[0]);
	}
	
	/**
	 * 作業項目の状態を生成する。
	 * 
	 * @param s 状態文字列
	 * @param i 状態値
	 */
	private WorkItemState(String s, int i)
	{
		stateString = s;
		stateNum = i;
	}

	/**
	 * 状態の文字列表現を返す。
	 * 
	 * @return 状態の文字列表現
	 */
	public String toString()
	{
		return stateString;
	}

	/**
	 * 状態の数値表現を返す。
	 * 
	 * @return 状態の数値表現
	 */
	public int toInt()
	{
		return stateNum;
	}
	
	/**
	 * 引数に指定されたオブジェクトとの比較を行う
	 *
	 * @param obj 比較対象オブジェクト
	 * @return 比較結果
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object obj)
	{
		if (obj == null || obj instanceof WorkItemState == false) {
			return false;
		}
		
		WorkItemState s = (WorkItemState)obj;
		return (this.stateNum == s.stateNum && this.stateString.equals(s.stateString));
	}

	/**
	 * 一意のハッシュコードを返す。
	 *
	 * @return ハッシュコード
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode()
	{
		return stateString.hashCode();
	}
}
