/* $Id: WorkItemHandler.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.wfmc;


/**
 * 作業項目ハンドラは、ワークフローに対するクライアントインタフェースである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public interface WorkItemHandler {

	/**
	 * 作業項目一覧の取得を行う。
	 *
	 * @param session セッションID
	 * @param filter 検索用フィルター
	 * @return 作業項目一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 */
	public WorkItem[] getWorkItems(String session, Filter filter)
	throws InvalidSessionException, InvalidFilterException;

	/**
	 * 作業項目の取得を行う。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @return 作業項目
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 */
	public WorkItem getWorkItem(String session, String wid)
	throws InvalidSessionException, InvalidWorkItemException;

	/**
	 * 作業項目の完了を行う。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 */
	public void completeWorkItem(String session, String wid)
	throws InvalidSessionException, InvalidWorkItemException;

	/**
	 * 現在の作業項目が遷移可能な状態の一覧を返す。
	 * 状態がない場合は、空の配列を返す。
	 * 利用可能なフィルターはStateFilterのみとする。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @param filter 検索用フィルター
	 * @return 作業項目状態一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 */
	public WorkItemState[] getWorkItemStates(String session,
			String wid, Filter filter)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidFilterException;
	
	/**
	 * 作業項目状態の変更を行う。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @param state 作業項目状態
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidStateException 指定された状態が有効でない場合
	 * @throws TransitionNotAllowedException 状態の変更が許可されていない場合
	 */
	public void changeWorkItemState(String session,
			String wid, WorkItemState state)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidStateException, TransitionNotAllowedException;
	
	/**
	 * 作業項目の再割り当てを行う。
	 *
	 * @param session セッションID
	 * @param sourceUser 割り当て前のユーザ
	 * @param targetUser 割り当て後のユーザ
	 * @param wid 作業項目ID
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidSourceUserException 割り当て前ユーザが有効でない場合
	 * @throws InvalidTargetUserException 割り当て先ユーザが有効でない場合
	 */
	public void reassignWorkItem(String session,
			Participant sourceUser, Participant targetUser, String wid)
	throws InvalidSessionException, InvalidWorkItemException,
	InvalidSourceUserException, InvalidTargetUserException;
	
	/**
	 * 作業項目属性一覧の取得を行う。
	 * 利用可能なフィルターはNameFilter, AttributeFilter, GroupFilterとする。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @param filter 検索用フィルター
	 * @return 作業項目属性一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 */
	public Attribute[] getWorkItemAttributes(String session,
			String wid, Filter filter)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidFilterException;
	
	/**
	 * 作業項目属性の取得を行う。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @param name 属性名
	 * @return 属性
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 */
	public Attribute getWorkItemAttributeValue(String session,
			String wid, String name)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidAttributeException;

	/**
	 * 作業項目属性の割り当てを行う。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @param attr 属性
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 * @throws AttributeAssignmentFailedException 属性の割り当てに失敗した場合
	 */
	public void assignWorkItemAttribute(String session,
			String wid, Attribute attr)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidAttributeException, AttributeAssignmentFailedException;
}
