/* $Id: ProcessState.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.wfmc;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import jp.co.argo21.nautica.workflow.omg.WfState;

/**
 * プロセスインスタンスの状態を表すオブジェクトである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public final class ProcessState implements WfState, Serializable
{
	/** シリアルバージョンUID */
	private static final long serialVersionUID = 20050000000000001L;

	/** open.not_running.not_started */
	public static final ProcessState OPEN_NOT_RUNNING_NOT_STARTED
		= new ProcessState("open.not_running.not_started", 1);

	/** open.not_running.not_started */
	public static final ProcessState OPEN_NOT_RUNNING_SUSPENDED
		= new ProcessState("open.not_running.suspended", 2);

	/** open.running */
	public static final ProcessState OPEN_RUNNING
		= new ProcessState("open.running", 3);

	/** closed.completed */
	public static final ProcessState CLOSED_COMPLETED
		= new ProcessState("closed.completed", 4);

	/** closed.terminated */
	public static final ProcessState CLOSED_TERMINATED
		= new ProcessState("closed.terminated", 5);

	/** closed.aborted */
	public static final ProcessState CLOSED_ABORTED
		= new ProcessState("closed.aborted", 6);
	
	/** 状態文字列 */
	private String stateString;
	
	/** 状態値 */
	private int stateNum;
	
	/**
	 * 引数の状態番号に該当するプロセス状態を返す。
	 * 
	 * @param n　状態番号
	 * @return　プロセス状態
	 */
	public static ProcessState getState(int n)
	{
		switch (n) {
		case 1: return OPEN_NOT_RUNNING_NOT_STARTED;
		case 2: return OPEN_NOT_RUNNING_SUSPENDED;
		case 3: return OPEN_RUNNING;
		case 4: return CLOSED_COMPLETED;
		case 5: return CLOSED_TERMINATED;
		case 6: return CLOSED_ABORTED;
		default: return null;
		}
	}
	
	/**
	 * 引数の状態番号に該当する遷移可能なプロセス状態を返す。
	 * 
	 * @param n　状態番号
	 * @return　遷移可能なプロセス状態
	 */
	public static ProcessState[] getTransitableStates(int n)
	{
		List<ProcessState> states = new ArrayList<ProcessState>();
		switch (n) {
		case 1:
		case 2:
			states.add(OPEN_RUNNING);
			states.add(CLOSED_COMPLETED);
			states.add(CLOSED_TERMINATED);
			states.add(CLOSED_ABORTED);
			break;
		case 3:
			states.add(OPEN_NOT_RUNNING_SUSPENDED);
			states.add(CLOSED_COMPLETED);
			states.add(CLOSED_TERMINATED);
			states.add(CLOSED_ABORTED);
			break;
		}
		return (ProcessState[])states.toArray(new ProcessState[0]);
	}
	
	/**
	 * プロセスインスタンスの状態を生成する。
	 * 
	 * @param s 状態文字列
	 * @param i 状態値
	 */
	private ProcessState(String s, int i)
	{
		stateString = s;
		stateNum = i;
	}

	/**
	 * 状態の文字列表現を返す。
	 * 
	 * @return 状態の文字列表現
	 */
	public String toString()
	{
		return stateString;
	}

	/**
	 * 状態の数値表現を返す。
	 * 
	 * @return 状態の数値表現
	 */
	public int toInt()
	{
		return stateNum;
	}
	
	/**
	 * 引数に指定されたオブジェクトとの比較を行う
	 *
	 * @param obj 比較対象オブジェクト
	 * @return 比較結果
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object obj)
	{
		if (obj == null || obj instanceof ProcessState == false) {
			return false;
		}
		
		ProcessState s = (ProcessState)obj;
		return (this.stateNum == s.stateNum && this.stateString.equals(s.stateString));
	}

	/**
	 * 一意のハッシュコードを返す。
	 *
	 * @return ハッシュコード
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode()
	{
		return stateString.hashCode();
	}
}
