/* $Id: ProcessDefinitionState.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.wfmc;

import java.util.ArrayList;
import java.util.List;

import jp.co.argo21.nautica.workflow.omg.WfState;
import jp.co.argo21.nautica.workflow.util.StringManager;

/**
 * プロセス定義の状態を表すオブジェクトである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public final class ProcessDefinitionState implements WfState
{
	/** シリアルバージョンUID */
	private static final long serialVersionUID = 20050000000000001L;

	/** enable */
	public static final ProcessDefinitionState ENABLE
		= new ProcessDefinitionState("enable", 1001);

	/** disable */
	public static final ProcessDefinitionState DISABLE
		= new ProcessDefinitionState("disable", 1002);
	
	/** 状態文字列 */
	private String stateString;
	
	/** 状態値 */
	private int stateNum;

	/**
	 * 引数の状態番号に該当するプロセス定義状態を返す。
	 * 
	 * @param n　状態番号
	 * @return　プロセス定義状態
	 */
	public static ProcessDefinitionState getState(int n)
	{
		switch (n) {
		case 1001: return ENABLE;
		case 1002: return DISABLE;
		default: return null;
		}
	}
	
	/**
	 * 引数の状態番号に該当する遷移可能なプロセス定義状態を返す。
	 * 
	 * @param n　状態番号
	 * @return　遷移可能なプロセス定義状態
	 */
	public static ProcessDefinitionState[] getTransitableStates(int n)
	{
		List<ProcessDefinitionState> states = new ArrayList<ProcessDefinitionState>();
		switch (n) {
		case 1001:
			states.add(DISABLE);
			break;
		case 1002:
			states.add(ENABLE);
			break;
		}
		return (ProcessDefinitionState[])states.toArray(new ProcessDefinitionState[0]);
	}
	
	/**
	 * 遷移のチェックを行う。
	 *
	 * @param src ソース状態
	 * @param dest　ターゲット状態
	 * @throws TransitionNotAllowedException　遷移が許されない場合に投げられる例外
	 */
	public static void checkTransition(ProcessDefinitionState src,
		ProcessDefinitionState dest)
	throws TransitionNotAllowedException
	{
		if (src.equals(ProcessDefinitionState.ENABLE)) {
			if (dest.equals(ProcessDefinitionState.DISABLE) == false) {
			    String errMsg = StringManager.get("E0050")
				    + "(SourceState=" + src + ", DestinationState=" + dest + ")";
				throw new TransitionNotAllowedException(errMsg);
			}
		} else if (src.equals(ProcessDefinitionState.DISABLE)) {
			if (dest.equals(ProcessDefinitionState.ENABLE) == false) {
			    String errMsg = StringManager.get("E0050")
			        + "(SourceState=" + src + ", DestinationState=" + dest + ")";
				throw new TransitionNotAllowedException(errMsg);
			}
		} else {
		    String errMsg = StringManager.get("E0050")
		        + "(SourceState=" + src + ", DestinationState=" + dest + ")";
			throw new TransitionNotAllowedException(errMsg);
		}
	}
	
	
	/**
	 * プロセス定義の状態を生成する。
	 * 
	 * @param s 状態文字列
	 * @param i 状態値
	 */
	private ProcessDefinitionState(String s, int i)
	{
		stateString = s;
		stateNum = i;
	}
	
	/**
	 * 状態の文字列表現を返す。
	 * 
	 * @return 状態の文字列表現
	 */
	public String toString()
	{
		return stateString;
	}

	/**
	 * 状態の数値表現を返す。
	 * 
	 * @return 状態の数値表現
	 */
	public int toInt()
	{
		return stateNum;
	}
	
	/**
	 * 引数に指定されたオブジェクトとの比較を行う
	 *
	 * @param obj 比較対象オブジェクト
	 * @return 比較結果
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object obj)
	{
		if (obj == null || obj instanceof ProcessDefinitionState == false) {
			return false;
		}
		
		ProcessDefinitionState s = (ProcessDefinitionState)obj;
		return (this.stateNum == s.stateNum && this.stateString.equals(s.stateString));
	}

	/**
	 * 一意のハッシュコードを返す。
	 *
	 * @return ハッシュコード
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode()
	{
		return stateString.hashCode();
	}
}
