/* $Id: ScriptingConfig.java,v 1.1 2007/10/29 02:55:37 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.ta;

import java.util.HashMap;
import java.util.Map;

import jp.co.argo21.nautica.workflow.ta.config.App;
import jp.co.argo21.nautica.workflow.ta.config.Script;
import jp.co.argo21.nautica.workflow.ta.config.Scripts;
import jp.co.argo21.nautica.workflow.ta.config.ScriptingDocument.Scripting;

/**
 * スクリプト実行用設定オブジェクト
 *
 * @author  kiyoda(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class ScriptingConfig {
    
    /**
     * ユーザID
     */
    private String user = null;
    
    /**
     * パスワード
     */
    private String password = null;
    
    /**
     * アプリケーションとスクリプトの対応表
     */
    private Map<String, Script> applications = new HashMap<String, Script>();
    
    /**
     * 引数付コンストラクタ
     *
     * @param scripting Scriptingオブジェクト
     */
    ScriptingConfig(Scripting scripting) {
        // ツールエージェント個々の認証情報の設定
        this.user = scripting.getAuthentication().getUser();
        this.password = scripting.getAuthentication().getPassword();
        
        // アプリケーション名とスクリプトのマッピング
        Scripts scriptlist = scripting.getScripts();
        if (scriptlist == null) {
            return;
        }
        Script[] scripts = scriptlist.getScriptArray();
        for (int i = 0; i < scripts.length; i++) {
            Script script = scripts[i];
            App[] apps = scripts[i].getAppArray();
            for (int j = 0; j < apps.length; j++) {
                this.applications.put(apps[j].getName(), script);
            }
        }    
    }

    /**
     * パラメータのアプリケーション名が設定情報に存在するかどうかチェック
     *
     * @param appName　アプリケーション名
     * @return	設定情報に指定したアプリケーションが存在する場合、true
     */
    public boolean existApplication(String appName)
    {
        return this.applications.containsKey(appName);
    }
    
    /**
     * ユーザIDを取得する
     *
     * @return	ユーザID
     */
    public String getUser() {
        return this.user;
    }
    
    /**
     * パスワードを取得する
     *
     * @return	パスワード
     */
    public String getPassword() {
        return this.password;
    }
    
    /**
     * アプリケーション名に対応したスクリプト設定を取得する。
     * 
     * アプリケーション名に該当するスクリプト設定が存在しない場合は null が返る。
     *
     * @param appName	アプリケーション名
     * @return	スクリプト設定
     */
    private Script getScript(String appName) {
        return (Script)this.applications.get(appName);
    }
    
    /**
     * スクリプトエンジン名を取得する。
     * 
     * アプリケーション名に該当するスクリプト設定が存在しない場合は null が返る。
     *
     * @param appName	アプリケーション名
     * @return	スクリプトエンジン名
     */
    public String getEngineName(String appName) {
        Script script = getScript(appName);
        if (script == null) {
            return null;
        }
        return script.getEngine().getName();
    }
    
    /**
     * スクリプトの種類を取得する。
     * 
     * 実行するスクリプトが、コマンド実行なのか
     * ファイルに記述されたスクリプトを実行するのかを
     * 指定しているスクリプトの種類を取得する。
     * コマンド実行の場合、command が、
     * ファイル実行の場合、file が返る。
     * アプリケーション名に該当するスクリプト設定が存在しない場合は null が返る。
     *
     * @param appName	アプリケーション名
     * @return	スクリプトの種類
     */
    public String getType(String appName) {
        Script script = getScript(appName);
        if (script == null) {
            return null;
        }
        return script.getType();
    }
    
    /**
     * スクリプトファイル名を取得する。
     * 
     * 実行するスクリプトのファイル名を取得する。
     * スクリプトファイル名が設定されていない場合は、nullが返る。
     * アプリケーション名に該当するスクリプト設定が存在しない場合は null が返る。
     *
     * @param appName	アプリケーション名
     * @return	スクリプトファイル名
     */
    public String getFilename(String appName) {
        Script script = getScript(appName);
        if (script == null) {
            return null;
        }
        return script.getFilename();
    }
    
    /**
     * スクリプトコマンドを取得する。
     * 
     * 実行するスクリプトのコマンド文字列を取得する。
     * スクリプトコマンドが設定されていない場合は、nullが返る。
     * アプリケーション名に該当するスクリプト設定が存在しない場合は null が返る。
     *
     * @param appName	アプリケーション名
     * @return	スクリプトコマンド
     */
    public String getCommand(String appName) {
        Script script = getScript(appName);
        if (script == null) {
            return null;
        }
        return script.getCommand();
    }
}
