/* $Id: MailToolAgent.java,v 1.1 2007/10/29 02:55:37 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.ta;

import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.ApplicationBusyException;
import jp.co.argo21.nautica.workflow.wfmc.ApplicationNotDefinedException;
import jp.co.argo21.nautica.workflow.wfmc.ApplicationNotStoppedException;
import jp.co.argo21.nautica.workflow.wfmc.Attribute;
import jp.co.argo21.nautica.workflow.wfmc.InvalidToolAgentException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidWorkItemException;

/**
 * メール送信用ツールエージェントクラス
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class MailToolAgent extends AbstractToolAgent {

    /**
     * コンストラクタ
     *
     * @param userId	ユーザID
     * @param passwd	パスワード
     */
    MailToolAgent(String userId, String passwd) {
        this.userId = userId;
        this.passwd = passwd;
    }

    /**
	 * アプリケーションの停止を行う。
	 *
	 * @param toolAgentHandle ツールエージェントハンドル
	 * @param wid 作業項目ID
	 * @throws ApplicationNotStoppedException アプリケーションを停止できなかった場合
	 * @throws InvalidToolAgentException 有効でないツールエージェントハンドルを指定した場合
	 * @throws InvalidWorkItemException 有効でない作業項目IDを指定した場合
	 * @throws ApplicationBusyException アプリケーションがビジーで要求を受けられない場合
     */
    public void terminateApplication(int toolAgentHandle, String wid)
            throws ApplicationNotStoppedException, InvalidToolAgentException,
            InvalidWorkItemException, ApplicationBusyException {
        
        // メール送信は送信中の停止は出来ないので、
        // 必ずApplicationNotStoppedException を投げる
        String errMsg = StringManager.get("E2002")
        	+ "(ToolAgentHandler=" + toolAgentHandle + ", "
        	+ "WorkItemID=" + wid + ")";
        log.error(errMsg);
        throw new ApplicationNotStoppedException(errMsg);
    }
    
    /**
     * アプリケーションを起動するツールエージェントのクラス名を取得する。
     * 
     * @return	アプリケーションを起動するツールエージェントのクラス名
     */
    protected String getToolAgentClassName() {
        return this.getClass().getName();
    }
    
    /**
     * アプリケーション実行情報を送信するキュー名を取得する。
     * 
     * ツールエージェント毎にキューが必要で、そのキューに名前をつけなくてはならない。
     * このキュー名はツールエージェント固有の文字列である。
     * 基本的にはツールエージェント名を使用する。
     * 各ツールエージェントで重複しないように注意すること。
     *
     * @return	キュー名
     */
    protected String getQueueName() {
        return MailToolAgentFactory.QUEUE_NAME;
    }
    
    /**
     * キューに送信するアプリケーション実行情報を取得する。
     *
     * アプリケーション実行クラスでは、この情報を基にして、
     * アプリケーションを実行する。 
     *
     * @param appName	定義に指定したアプリケーション名
     * @param wid		作業項目ID
     * @param handler	ツールエージェントハンドラ
     * @param attrList	属性一覧
     * @return	アプリケーション実行情報
     */
    protected AppExecutionInfo getExecInfo(String appName, String wid,
            int handler, Attribute[] attrList) 
    {
        return new MailExecutionInfo(appName, getToolAgentClassName(), 
                wid, handler, attrList);
    }
    
    /**
     * アプリケーションが定義されているかどうかチェックする。
     * 
     * アプリケーションが定義されていない場合、例外が発生する。
     *
     * @param appName	アプリケーション名
     * @throws	ApplicationNotDefinedException	アプリケーションが定義されていない場合
     * @see jp.co.argo21.nautica.workflow.ta.AbstractToolAgent#checkApplicationDefined(java.lang.String)
     */
    protected void checkApplicationDefined(String appName)
    	throws ApplicationNotDefinedException {
        
        MailConfig config = null;
        try {
	        config = MailConfigFactory.getConfig();
        } catch (Exception ex) {
            // エラー
            String errMsg = StringManager.get("E2004")
            	+ "(ApplicationName=" + appName + ")";
            throw new ApplicationNotDefinedException(errMsg, ex);
        }
        
        if (config == null || !config.existApplication(appName)) {
            // エラー
            String errMsg = StringManager.get("E2004")
        	+ "(ApplicationName=" + appName + ")";
            throw new ApplicationNotDefinedException(errMsg);
        }
    }
}
