/* $Id: MailConfig.java,v 1.1 2007/10/29 02:55:37 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.ta;

import java.util.HashMap;
import java.util.Map;

import jp.co.argo21.nautica.workflow.ta.config.App;
import jp.co.argo21.nautica.workflow.ta.config.Auth;
import jp.co.argo21.nautica.workflow.ta.config.Receiver;
import jp.co.argo21.nautica.workflow.ta.config.Server;
import jp.co.argo21.nautica.workflow.ta.config.Servers;
import jp.co.argo21.nautica.workflow.ta.config.MailDocument.Mail;

/**
 * メール送信用設定オブジェクト
 *
 * @author  kiyoda(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class MailConfig {
    
    /**
     * ユーザID
     */
    private String user = null;
    
    /**
     * パスワード
     */
    private String password = null;
    
    /**
     * アプリケーションとメールサーバの対応表
     */
    private Map<String, Server> applications = new HashMap<String, Server>();
    
    /**
     * 引数付コンストラクタ
     *
     * @param mali
     */
    MailConfig(Mail mali) {
        // ツールエージェント個々の認証情報の設定
        this.user = mali.getAuthentication().getUser();
        this.password = mali.getAuthentication().getPassword();
        
        // アプリケーション名とサーバ設定のマッピング
        Servers serverlist = mali.getServers();
        if (serverlist == null) {
            return;
        }
        Server[] servers = serverlist.getServerArray();
        for (int i = 0; i < servers.length; i++) {
            Server server = servers[i];
            App[] apps = servers[i].getAppArray();
            for (int j = 0; j < apps.length; j++) {
                this.applications.put(apps[j].getName(), server);
            }
        }
    }
    
    /**
     * アプリケーション名に対応したメールサーバの設定を取得
     *
     * @param appName
     * @return	コマンド文字列
     */
    private Server getServer(String appName)
    {
        return (Server)this.applications.get(appName);
    }
    
    /**
     * メールサーバの認証タイプを取得する。
     * 
     * デフォルトでは normal が返る。<br>
     * アプリケーション名に該当するメールサーバの設定が存在しない場合は null が返る。
     *
     * @param appName	アプリケーション名
     * @return	メールサーバの認証タイプ
     */
    public String getAuthType(String appName) {
        Server server = getServer(appName);
        if (server == null) {
            return null;
        }
        return server.getAuthType();
    }
   
    /**
     * 送信サーバのプロトコルを取得する。
     * 
     * デフォルトでは smtp が返る。<br>
     * アプリケーション名に該当するメールサーバの設定が存在しない場合は null が返る。
     *
     * @param appName	アプリケーション名
     * @return	送信サーバのプロトコル
     */
    public String getSenderProtocol(String appName) {
        Server server = getServer(appName);
        if (server == null) {
            return null;
        }
        return server.getSender().getProtocol();
    }
    
    /**
     * メール送信のエンコード名を取得する。
     * 
     * デフォルトでは iso-2022-jp が返る。<br>
     * アプリケーション名に該当するメールサーバの設定が存在しない場合は null が返る。
     *
     * @param appName	アプリケーション名
     * @return	メール送信のエンコード名
     */
    public String getSenderEncode(String appName) {
        Server server = getServer(appName);
        if (server == null) {
            return null;
        }
        return server.getSender().getEncode();
    }
    
    /**
     * メール送信のMIME-Typeを取得する。
     * 
     * デフォルトでは "text/plain" が返る。<br>
     * アプリケーション名に該当するメールサーバの設定が存在しない場合は null が返る。
     *
     * @param appName	アプリケーション名
     * @return	メール送信のMIME-Type
     */
    public String getSenderMimeType(String appName) {
        Server server = getServer(appName);
        if (server == null) {
            return null;
        }
        return server.getSender().getMimeType();
    }
    
    /**
     * 送信サーバのホストアドレスを取得する。
     * 
     * アプリケーション名に該当するメールサーバの設定が存在しない場合は null が返る。
     *
     * @param appName	アプリケーション名
     * @return	送信サーバのホストアドレス。
     */
    public String getSenderHostAddress(String appName) {
        Server server = getServer(appName);
        if (server == null) {
            return null;
        }
        return server.getSender().getHost().getAddress();
    }
    
    /**
     * 送信サーバのポート番号を取得する。
     * 
     * アプリケーション名に該当するメールサーバの設定が存在しない場合は -1 が返る。
     *
     * @param appName	アプリケーション名
     * @return	送信サーバのポート番号。
     */
    public int getSenderHostPort(String appName) {
        Server server = getServer(appName);
        if (server == null) {
            return -1;
        }
        return server.getSender().getHost().getPort();
    }
    
    /**
     * 送信サーバのユーザ名を取得する。
     * 
     * アプリケーション名に該当するメールサーバの設定が存在しない場合、
     * 送信サーバで認証を行わない場合は null が返る。
     *
     * @param appName	アプリケーション名
     * @return	送信サーバのユーザ名。
     */
    public String getSenderHostUser(String appName) {
        Server server = getServer(appName);
        if (server == null) {
            return null;
        }
        Auth auth = server.getSender().getAuthentication();
        if (auth == null) {
            return null;
        }
        return auth.getUser();
    }
    
    /**
     * 送信サーバのパスワードを取得する。
     * 
     * アプリケーション名に該当するメールサーバの設定が存在しない場合、
     * 送信サーバで認証を行わない場合は null が返る。
     *
     * @param appName	アプリケーション名
     * @return	送信サーバのユーザ名
     */
    public String getSenderHostPassword(String appName) {
        Server server = getServer(appName);
        if (server == null) {
            return null;
        }
        Auth auth = server.getSender().getAuthentication();
        if (auth == null) {
            return null;
        }
        return auth.getPassword();
    }
    
    /**
     * 受信サーバのプロトコルを取得する。
     * 
     * アプリケーション名に該当するメールサーバの設定が存在しない場合、
     * 受信サーバを使用しない場合は、null が返る。
     *
     * @param appName	アプリケーション名
     * @return	受信サーバのプロトコル
     */
    public String getReceiverProtocol(String appName) {
        Server server = getServer(appName);
        if (server == null) {
            return null;
        }
        Receiver receiver = server.getReceiver();
        if (receiver == null) {
            return null;
        }
        return receiver.getProtocol();
    }
    
    /**
     * 受信サーバのホストアドレスを取得する。
     * 
     * アプリケーション名に該当するメールサーバの設定が存在しない場合、
     * 受信サーバを使用しない場合は、null が返る。
     *
     * @param appName	アプリケーション名
     * @return	受信サーバのホストアドレス。
     */
    public String getReceiverHostAddress(String appName) {
        Server server = getServer(appName);
        if (server == null) {
            return null;
        }
        Receiver receiver = server.getReceiver();
        if (receiver == null) {
            return null;
        }
        return receiver.getHost().getAddress();
    }
    
    /**
     * 受信サーバのポート番号を取得する。
     * 
     * アプリケーション名に該当するメールサーバの設定が存在しない場合、
     * 受信サーバを使用しない場合は、-1 が返る。
     *
     * @param appName	アプリケーション名
     * @return	受信サーバのポート番号
     */
    public int getReceiverHostPort(String appName) {
        Server server = getServer(appName);
        if (server == null) {
            return -1;
        }
        Receiver receiver = server.getReceiver();
        if (receiver == null) {
            return -1;
        }
        return receiver.getHost().getPort();
    }
    
    /**
     * 受信サーバのユーザ名を取得する。
     * 
     * アプリケーション名に該当するメールサーバの設定が存在しない場合、
     * 受信サーバを使用しない場合は、null が返る。
     *
     * @param appName	アプリケーション名
     * @return	受信サーバのユーザ名
     */
    public String getReceiverHostUser(String appName) {
        Server server = getServer(appName);
        if (server == null) {
            return null;
        }
        Receiver receiver = server.getReceiver();
        if (receiver == null) {
            return null;
        }
        return receiver.getAuthentication().getUser();
    }
    
    /**
     * 受信サーバのパスワードを取得する。
     * 
     * アプリケーション名に該当するメールサーバの設定が存在しない場合、
     * 受信サーバを使用しない場合は、null が返る。
     *
     * @param appName	アプリケーション名
     * @return	受信サーバのパスワード
     */
    public String getReceiverHostPassword(String appName) {
        Server server = getServer(appName);
        if (server == null) {
            return null;
        }
        Receiver receiver = server.getReceiver();
        if (receiver == null) {
            return null;
        }
        return receiver.getAuthentication().getPassword();
    }
    
    /**
     * パラメータのアプリケーション名が設定情報に存在するかどうかチェック
     *
     * @param appName
     * @return	設定情報に指定したアプリケーションが存在する場合、true
     */
    public boolean existApplication(String appName)
    {
        return this.applications.containsKey(appName);
    }
    
    /**
     * ユーザIDを取得する
     *
     * @return	ユーザID
     */
    public String getUser() {
        return this.user;
    }
    
    /**
     * パスワードを取得する
     *
     * @return	パスワード
     */
    public String getPassword() {
        return this.password;
    }
}
