/* $Id: AbstractToolAgentMessageAdapter.java,v 1.2 2007/11/28 07:09:06 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.ta;

import java.sql.Timestamp;

import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.AppExecutionBean;
import jp.co.argo21.nautica.workflow.dataaccess.AppExecutionDAO;
import jp.co.argo21.nautica.workflow.dataaccess.WorkflowDAOFactory;
import jp.co.argo21.nautica.workflow.engine.DataAccessManager;
import jp.co.argo21.nautica.workflow.engine.WorkflowEngine;
import jp.co.argo21.nautica.workflow.jms.WorkflowMessageAdapter;
import jp.co.argo21.nautica.workflow.omg.WorkflowException;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler;
import jp.co.argo21.nautica.workflow.wfmc.WorkItemState;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowServiceManager;

/**
 * ツールエージェント用メッセージアダプタの抽象クラス
 *
 * @author  kiyoda(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public abstract class AbstractToolAgentMessageAdapter extends WorkflowMessageAdapter {
    
    /* ログオブジェクト */
    protected Logger log = LogManager.getLogger(this.getClass());
    
    /**
     * コンストラクタ
     */
    AbstractToolAgentMessageAdapter() {
        super();
    }

    /**
     * アプリケーションの状態を取得する。
     * 
     * アプリケーションの状態をアプリケーション実行テーブルから取得する。<br>
     * アプリケーション実行テーブルに該当のレコードが存在しない場合は、
     * １秒間隔で、更に４回（計５回）検索を行う。<br>
     * それでもアプリケーション実行テーブルにレコードが存在しない場合は、
     * 例外WorkflowExceptionを投げる。
     *
     * @param info	アプリケーション実行情報
     * @return	アプリケーション情報
     * @throws Exception	状態の取得に失敗した場合
     */
    protected AppExecutionBean getApplicationState(AppExecutionInfo info) 
    	throws Exception {

        AppExecutionBean bean = null;
        AppExecutionDAO appExecutionDao = null;
        
        // アプリケーション実行テーブル(APPEXEC)テーブルの
        // アプリケーションの状態を取得する。
        WorkflowDAOFactory factory = DataAccessManager.getDAOFactory();
        appExecutionDao = factory.getAppExecutionDAO();
        // アプリケーション実行テーブルにINSERTされるまで待機
        // 1秒間隔で5回再試行する
        for (int cnt = 0; cnt < 5 && bean == null; cnt++) {
            // 初回以外は1秒(1000ミリ秒)待機
            if (cnt != 0) {
                Thread.sleep(1000L);
            }
            // アプリケーション実行テーブル検索
            bean = appExecutionDao.findByPrimaryKey(info.getAppName(), info.getToolAgent(), info.getHandler());
        }
        if (bean == null) {
            // 5回再検索してもアプリケーション実行テーブルにレコードがない場合は、
            // ログにエラーメッセージを出力して、実行をやめる。
            String errMsg = StringManager.get("E2009")
            	+ "APPNAME=" + info.getAppName() + ", "
            	+ "TOOL_AGENT=" + info.getToolAgent() + ", "
            	+ "EXEC_ID=" + info.getHandler();
            log.error(errMsg);
            throw new WorkflowException(errMsg);
        }

        return bean;
    }
    
    /**
     * アプリケーション実行テーブルの状態を変更する。
     * 
     * @param info	アプリケーション実行情報
     * @param oldState	旧状態
     * @param newState	新状態
     * @throws Exception	アプリケーション実行テーブルの状態変更に失敗した場合
     */
    protected void updateState(AppExecutionInfo info, 
            ApplicationState oldState, ApplicationState newState) throws Exception {
        
        WorkflowDAOFactory factory = DataAccessManager.getDAOFactory();
        AppExecutionDAO appExecutionDao = factory.getAppExecutionDAO();
        appExecutionDao.updateState(info.getAppName(), info.getToolAgent(), info.getHandler(), oldState, newState);
    }
    
    /**
     * アプリケーションの終了処理またはリトライ処理を行う。
     * 
     * アプリケーションが異常終了しており、なおかつ試行回数が正の値の場合は、
     * アプリケーションのリトライ処理を行う。<br>
     * それ以外の場合は、アプリケーション実行テーブルを終了の状態に変更し、
     * 作業項目に、アプリケーションの実行完了を通知する。
     *
     * @param info	アプリケーション実行情報
     * @param oldState	旧状態
     * @param result	アプリケーション実行結果。0は正常終了。0以外は異常終了。
     * @param agent		ツールエージェント
     * @throws Exception	アプリケーションのリトライ処理か終了処理に失敗した場合
     */
    protected void finalizeApplication(AppExecutionInfo info, 
            ApplicationState oldState, int result, AbstractToolAgent agent)
    throws Exception {        
        WorkflowDAOFactory factory = DataAccessManager.getDAOFactory();
        AppExecutionDAO appExecutionDao = factory.getAppExecutionDAO();

        ApplicationState resultAppState = ApplicationState.TERMINATED;
        WorkItemState resultWorkState = WorkItemState.CLOSED_TERMINATED;
        if (result == 0) {
            resultAppState = ApplicationState.COMPLETED;
            resultWorkState = WorkItemState.CLOSED_COMPLETED;
        }
        
        appExecutionDao.updateState(info.getAppName(), info.getToolAgent(), info.getHandler(), 
                oldState, resultAppState, new Timestamp(System.currentTimeMillis()));
        
        // 作業項目にアプリケーションが終了したことを通知する
        WorkflowServiceManager manager = WorkflowEngine.getInstance();
        String session = null;
        try {
            session = agent.connect();
            WorkItemHandler handler = manager.getWorkItemHandler(session);
            handler.changeWorkItemState(session, info.getWid(), resultWorkState);
        } finally {
            if (session != null) {
                agent.disconnect(session);
            }
        }
    }
}
