/* $Id: RoleManager.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.security;

import jp.co.argo21.nautica.workflow.wfmc.WorkflowSecurityException;

/**
 * 本ワークフローシステム運用基盤利用されるインタフェースを規定する。
 * RoleManagerは、権限管理関連の各種APIを定義する。
 * 実際には、任意の権限管理システムを利用するために、
 * このインタフェースを利用した具体実装を用意しなければならない。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public interface RoleManager {

	/**
	 * ridに対応する権限を返す。
	 * ridに対応する権限がない場合は null を返す。
	 *
	 * @param rid 権限ID
	 * @return 権限
	 * @throws InvalidRoleException 指定された権限が有効でない場合
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	public Role getRole(String rid)
	throws InvalidRoleException, WorkflowSecurityException;

	/**
	 * 組織に付与されている全権限を返す。
	 * 権限が存在しない場合は空の配列を返す。
	 *
	 * @param org 組織単位
	 * @return 権限の一覧
	 * @throws InvalidOrganizationException
	 * @throws WorkflowSecurityException
	 * @throws InvalidRoleException 指定された組織が有効でない場合
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	public Role[] getAttachedRoles(Organization org)
	throws InvalidOrganizationException, WorkflowSecurityException;

	/**
	 * ユーザに付与されている全権限を返す。
	 * 権限が存在しない場合は空の配列を返す。
	 *
	 * @param user ユーザ
	 * @return 権限の一覧
	 * @throws InvalidRoleException 指定されたユーザが有効でない場合
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	public Role[] getAttachedRoles(User user)
	throws InvalidUserException, WorkflowSecurityException;

	/**
	 * 権限を持つ全組織単位を返す。
	 * 組織が存在しない場合は空の配列を返す。
	 *
	 * @param role 権限
	 * @return 組織単位の一覧
	 * @throws InvalidRoleException 指定された権限が有効でない場合
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	public Organization[] getOrganizationsByRole(Role role)
	throws InvalidRoleException, WorkflowSecurityException;

	/**
	 * 権限を持つ全ユーザを返す。
	 * ユーザが存在しない場合は空の配列を返す。
	 *
	 * @param role 権限
	 * @return ユーザの一覧
	 * @throws InvalidRoleException 指定された権限が有効でない場合
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	public User[] getUsersByRole(Role role)
	throws InvalidRoleException, WorkflowSecurityException;
}
