/* $Id: WfProcess.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.omg;

/**
 * WfProcessは、ひとつのビジネスプロセスの管理実体を表すインタフェースである。
 * 本書では、この管理実体を「プロセス」と表記する。
 * WfProcessは、プロセスの実行を開始する操作、
 * プロセスによって生成された結果を取得する操作、
 * WfRequesterやWfActivityとの関連を追加することで、
 * WfExecutionObjectを特化している。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public interface WfProcess extends WfExecutionObject {

	/**
	 * プロセス要求者を返す。
	 *
	 * @return プロセス要求者
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public WfRequester getRequester() throws WorkflowException;

	/**
	 * プロセス要求者をプロセスに関連付ける。
	 * プロセスが生成された時点で、この関連付けが行われる。
	 * プロセスは、WfRequesterのインタフェースを通じて、状態の変更を通知する。
	 *
	 * @param requester プロセス要求者
	 * @throws CannotChangeRequesterException プロセス要求者を更新できない場合
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void setRequester(WfRequester requester)
	throws CannotChangeRequesterException, WorkflowException;

	/**
	 * プロセスは、WfProcessMgrに関連付けられる。
	 * WfProcessが生成される時点で関連付けが行われ、変更はできない。
	 *
	 * @return プロセスマネージャ
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public WfProcessManager getManager() throws WorkflowException;

	/**
	 * 0以上のWfActiviyがプロセスに関連付けられる。
	 * ステップは、現時点でプロセス内でopenな状態であるアクティビティを表すものとする。
	 * ここでは、ステップの個数を返す。
	 *
	 * @return ステップの個数
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public int getStepCount() throws WorkflowException;

	/**
	 * ステップからIDで指定されたアクティビティを返す。
	 *
	 * @param aid アクティビティID
	 * @return アクティビティ
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public WfActivity getStep(String aid) throws WorkflowException;

	/**
	 * 現時点の全ステップを返す。
	 *
	 * @return 全ステップ
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public WfActivity[] getAllSteps() throws WorkflowException;

	/**
	 * 引数で指定されたアクティビティが、そのプロセスのステップに所属しているかどうかを返す。
	 *
	 * @param act アクティビティ
	 * @return 所属している場合はtrue
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public boolean isMemberOfStep(WfActivity act) throws WorkflowException;

	/**
	 * 引数で指定された状態を持つアクティビティを返す。
	 *
	 * @param state 状態
	 * @return アクティビティ
	 * @throws InvalidStateException 不正な状態
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public WfActivity[] getActivitiesInState(String state)
	throws InvalidStateException, WorkflowException;

	/**
	 * WfProcessの実行を開始するために使用される。
	 * プロセスの状態は、open.not_runningからopen.runningに変化する。
	 *
	 * @throws CannotStartException 開始できない場合
	 * @throws AlreadyRunningException すでに開始している場合
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void start()
	throws CannotStartException, AlreadyRunningException, WorkflowException;

	/**
	 * WfProcessによって生成される結果を返す。
	 * プロセスが完了するまでは結果は定義されない。
	 * 結果の取得がまだ取得できない場合は、例外が発生する。
	 *
	 * @return 結果
	 * @throws ResultNotAvailableException 結果を取得できない場合
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public WfNameValue[] getResult()
	throws ResultNotAvailableException, WorkflowException;
}
