/* $Id: WfExecutionObject.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.omg;

/**
 * WfExecutionObjectは、WfProcessとWfActivityの共通の属性や状態、
 * 操作を定義する管理実体の基底インタフェースである。
 * 本書では、この管理実体を「実行オブジェクト」と表記する。
 * WfExecutionObjectは、内部状態の取得および設定機能を提供する。
 * 現時点での状態を取得する操作、別の状態に遷移させる操作、
 * ある特定の状態遷移に対する操作が提供されている。
 * これらの操作は保留(suspend)、再開(resume)、停止(terminate)、中断(abort)である。
 * インタフェースには、名前、説明、キーとなる属性を含む。
 * また、実行履歴を表す監査情報を、指定されたフィルタに基づいて返すことによって、
 * WfExecutionObjectの実行の監視操作も提供する。
 * 他にも、コンテキストの取得と設定のメソッドがある。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public interface WfExecutionObject extends WfObject {

	/**
	 * IDを返す。
	 *
	 * @return ID
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public String getID() throws WorkflowException;

	/**
	 * 名称を返す。
	 *
	 * @return 名称
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public String getName() throws WorkflowException;

	/**
	 * 名称を設定する。
	 *
	 * @param name 名称
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void setName(String name) throws WorkflowException;

	/**
	 * 詳細情報を返す。
	 *
	 * @return 詳細情報
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public String getDescription() throws WorkflowException;

	/**
	 * 詳細情報を設定する。
	 *
	 * @param desc 詳細情報
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void setDescription(String desc) throws WorkflowException;

	/**
	 * 現在の状態を返す。
	 *
	 * @return 状態
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public WfState getState() throws WorkflowException;

	/**
	 * 現在の状態から遷移可能な状態の一覧を返す。
	 *
	 * @return 遷移可能な状態の一覧
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public WfState[] getTransitableStates() throws WorkflowException;

	/**
	 * 指定された状態に遷移を試みる。
	 *
	 * @param state 状態
	 * @throws InvalidStateException 無効な状態の場合
	 * @throws TransitionNotAllowedException 指定された状態に遷移できない場合
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void changeState(WfState state)
	throws InvalidStateException, TransitionNotAllowedException, WorkflowException;

	/**
	 * 最後に状態が遷移した時間を返す。
	 *
	 * @return 最終状態変更時間
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public String getLastStateTime() throws WorkflowException;

	/**
	 * 保留中の実行オブジェクトの再開を要求する。
	 * 要求が受け入れられた場合、状態は、open.not_running.suspendedから、
	 * open.runningに設定される。
	 *
	 * @throws CannotResumeException 再開できない場合 
	 * @throws NotRunningException 走行中でない場合
	 * @throws NotSuspendedException 保留されていない場合
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void resume()
	throws CannotResumeException, NotRunningException, 
		NotSuspendedException, WorkflowException;

	/**
	 * 実行オブジェクトの保留を要求する。
	 * 要求が受け入れられた場合、状態は、open.not_running.suspendedに設定される。
	 *
	 * @throws CannotSuspendException 保留できない場合
	 * @throws NotRunningException 走行中でない場合
	 * @throws AlreadySuspendedException すでに保留されている場合
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void suspend()
	throws CannotSuspendException, NotRunningException,
		AlreadySuspendedException, WorkflowException;

	/**
	 * 実行中の実行オブジェクトの停止を要求する。
	 * 要求が受け入れられた場合、状態はclosed.terminatedに設定される。
	 *
	 * @throws CannotStopException 停止できない場合
	 * @throws NotRunningException 走行中でない場合
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void terminate()
	throws CannotStopException, NotRunningException, WorkflowException;

	/**
	 * 保留された実行オブジェクトの中断を要求する。
	 * 要求が受け入れられた場合、状態はclosed.abortedに設定される。
	 *
	 * @throws CannotStopException 停止できない場合
	 * @throws NotRunningException 走行中でない場合
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void abort()
	throws CannotStopException, NotRunningException, WorkflowException;

	/**
	 * 実行オブジェクトにかかわる属性一覧を返す。
	 *
	 * @return 属性一覧
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public WfNameValue[] getContext() throws WorkflowException;

	/**
	 * 実行オブジェクトにかかわる属性を返す。
	 *
	 * @param name 属性名
	 * @return 属性
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public WfNameValue getContext(String name) throws WorkflowException;

	/**
	 * 実行オブジェクトにかかわる属性一覧を設定する。
	 *
	 * @param context 属性一覧
	 * @throws InvalidDataException 無効な属性の場合 
	 * @throws UpdateNotAllowedException 更新ができない場合
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void setContext(WfNameValue[] context)
	throws InvalidDataException, UpdateNotAllowedException, WorkflowException;

	/**
	 * 実行オブジェクトにかかわる属性を設定する。
	 *
	 * @param context 属性
	 * @throws InvalidDataException 無効な属性の場合 
	 * @throws UpdateNotAllowedException 更新ができない場合
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void setContext(WfNameValue context)
	throws InvalidDataException, UpdateNotAllowedException, WorkflowException;

	/**
	 * 変更履歴の数を返す。
	 *
	 * @return 変更履歴の数
	 * @throws HistoryNotAvailableException 変更履歴が取得可能でない場合
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public int getHistoryCount()
	throws HistoryNotAvailableException, WorkflowException;

	/**
	 * インデクスを指定して、履歴を取得する。
	 *
	 * @param index インデクス
	 * @return 変更履歴
	 * @throws HistoryNotAvailableException 変更履歴が取得可能でない場合
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public WfEvent getHistory(int index)
	throws HistoryNotAvailableException, WorkflowException;

	/**
	 * この実行オブジェクトに関わる変更履歴を取得する。
	 *
	 * @return 変更履歴
	 * @throws HistoryNotAvailableException 変更履歴が取得可能でない場合
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public WfEvent[] getAllHistories()
	throws HistoryNotAvailableException, WorkflowException;
}
