/* $Id: ToolAgentManager.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.engine;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Hashtable;
import java.util.Map;

import jp.co.argo21.nautica.workflow.ta.config.Agent;
import jp.co.argo21.nautica.workflow.ta.config.App;
import jp.co.argo21.nautica.workflow.ta.config.ToolAgentsDocument;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.InvalidApplicationException;
import jp.co.argo21.nautica.workflow.wfmc.ToolAgent;
import jp.co.argo21.nautica.workflow.wfmc.ToolAgentFactory;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlOptions;

/**
 * 登録されているツールエージェントを管理する。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class ToolAgentManager
{
	/** エンジンログ */
	private static Logger eLog = LogManager.getEngineLogger();
	
	/** ToolAgentManagerのシングルトン */
	private static ToolAgentManager singleton;

	/** 設定ファイル名 */
	private static final String CONFIG = "toolagent.xml";
	
	/** アプリケーション情報 */
	private Map<String, ToolAgentFactory> applications = new Hashtable<String, ToolAgentFactory>();
	
	
	/**
	 * ToolAgentManagerの生成と初期化を行う。
	 *
	 * @throws Exception 任意の例外
	 */
	static void create() throws Exception
	{
		singleton = new ToolAgentManager();
		try {
			singleton.init();
		} catch (Exception ex) {
			eLog.error(StringManager.get("E1005"));
			throw ex;
		}
	}

	/**
	 * ToolAgentManagerのインスタンスを返す。
	 *
	 * @return ToolAgentManager
	 */
	static ToolAgentManager getInstance()
	{
		return singleton;
	}
	
	/**
	 * ToolAgentManagerを生成する。
	 */
	private ToolAgentManager()
	{
	}
	
	/**
	 * ResourceManagerFactoryを初期化する。
	 *
	 * @throws Exception 任意の例外
	 */
	private void init() throws Exception
	{
		String home = SystemChecker.getProperty(WorkflowEngineConstants.NAUTICA_HOME);
		String separator = SystemChecker.getProperty("file.separator");
		String path = home
		            + separator + "config"
		            + separator + CONFIG;
		File confFile = new File(path);
		InputStream in = new FileInputStream(confFile);
		XmlOptions opts = new XmlOptions();
		opts.setCharacterEncoding("utf-8");

		ToolAgentsDocument doc = ToolAgentsDocument.Factory.parse(in, opts);
		in.close();
		
		load(doc);
	}
	
	/**
	 * 指定された論理アプリケーション名に対応するツールエージェントを返す。
	 *
	 * @param appName 論理アプリケーション名
	 * @return ツールエージェント
	 * @throws InvalidApplicationException 論理アプリケーション名が正しくない場合
	 */
	ToolAgent getToolAgent(String appName)
	throws InvalidApplicationException
	{
		if (appName == null) {
			// 論理アプリケーション名が設定されていません。
			String E0147 = StringManager.get("E0147");
			eLog.fatal(E0147);
			throw new InvalidApplicationException(E0147);
		}
		ToolAgentFactory factory = (ToolAgentFactory)applications.get(appName);
		if (factory == null) {
			// 対応するツールエージェントのファクトリが取得できません。
			String E0148 = StringManager.get("E0148") + "(ApplicationName = " + appName + ")";
			eLog.fatal(E0148);
			throw new InvalidApplicationException(E0148);
		}
		return factory.getToolAgent();
	}

	/**
	 * ツールエージェント定義ドキュメントをロードする。
	 * 
	 * @param doc XPDLドキュメント
	 * @throws XmlException XML例外
	 * @throws IOException 入出力例外
	 */
	private void load(ToolAgentsDocument doc)
	throws Exception
	{
		ToolAgentsDocument.ToolAgents config = doc.getToolAgents();
		
		Agent[] agents = config.getAgentArray();
		for (int i = 0; i < agents.length; i++) {
			String cn = agents[i].getClassname();
			int retry = 1;
			if (agents[i].isSetRetry()) retry = agents[i].getRetry().intValue();
			ToolAgentFactory factory = (ToolAgentFactory)Class.forName(cn).newInstance();
			factory.setRetryCount(retry);
			
			App[] apps = agents[i].getAppArray();
			for (int j = 0; j < apps.length; j++) {
				applications.put(apps[j].getName(), factory);
			}
		}
	}
}
