/* $Id: SystemMessageTransporter.java,v 1.2 2007/11/05 09:50:54 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.engine;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jp.co.argo21.nautica.workflow.jms.MessagingException;
import jp.co.argo21.nautica.workflow.jms.WorkflowMQ;
import jp.co.argo21.nautica.workflow.jms.WorkflowMessage;

import org.apache.log4j.Logger;

/**
 * エンジン用のメッセージ配送オブジェクト。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class SystemMessageTransporter
{
	/** スレッドローカルな遅延メッセージ格納管理 */
	private static final ThreadLocal<Map<String, List<WorkflowMessage>>> manager = new ThreadLocal<Map<String, List<WorkflowMessage>>>();

	/** エンジンログ */
	@SuppressWarnings("unused")
	private static Logger eLog = LogManager.getEngineLogger();
	
	/** キュー名 */
	private String queueName;
	
	/**
	 * 同一スレッドで投函されたメッセージをまとめて配送する。
	 *
	 * @throws MessagingException 送信時に例外が発生した場合
	 */
	static synchronized void execute()
	throws MessagingException
	{
		Map<String, List<WorkflowMessage>> holders = manager.get();
		if (holders == null) return;

		MessageBroker broker = MessageBroker.getInstance();

		Set<String> keys = holders.keySet();
		for (String key : keys) {
			WorkflowMQ mq = broker.getWorkflowMQ(key);
			List<WorkflowMessage> holder = holders.get(key);
			for (WorkflowMessage msg : holder) {
				mq.send(msg);
			}
			holder.clear();
		}
	}
	
	/**
	 * 指定されたキューに対するメッセージ配送機能を生成する。
	 *
	 * @param queueName キュー名
	 */
	SystemMessageTransporter(String queueName)
	{
		this.queueName = queueName;
		Map<String, List<WorkflowMessage>> holders = manager.get();
		if (holders == null) {
			holders = new Hashtable<String, List<WorkflowMessage>>();
			manager.set(holders);
		}
		if (holders.containsKey(queueName) == false) {
			List<WorkflowMessage> holder = new ArrayList<WorkflowMessage>();
			holders.put(queueName, holder);
		}
	}
	
	/**
	 * メッセージキューにメッセージを送る。
	 * 実際にはまだ送信されず、内部キューに格納される。
	 *
	 * @param msg メッセージ
	 * @throws MessagingException 送信時に例外が発生した場合
	 */
	synchronized void sendMessage(WorkflowMessage msg)
	throws MessagingException
	{
		Map<String, List<WorkflowMessage>> holders = manager.get();
		List<WorkflowMessage> holder = holders.get(queueName);
		holder.add(msg);
	}
}
