/* $Id: SystemChecker.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.engine;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import jp.co.argo21.nautica.workflow.omg.WorkflowException;

/**
 * 起動時のシステムチェックを行う。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class SystemChecker
{
	/** 設定ファイル名 */
	private static final String PROP_NAME = "nautica.properties";

	private static final Properties prop = new Properties();
	
	/**
	 * 
	 *
	 * @param key
	 * @return
	 */
	public static String getProperty(String key)
	{
		return prop.getProperty(key);
	}
		
	/**
	 * システム起動パスと構成をチェックする。
	 *
	 * @throws WorkflowException チェック違反があった場合
	 */
	static void checkBootPath()
	throws WorkflowException
	{
		String home = System.getProperty(WorkflowEngineConstants.NAUTICA_HOME);
		
		checkHome(home);
		checkDir(home, "config", "Configuration path");
		checkDir(home, "repository", "Repository path");
		checkDir(home, "logs", "Logging path");
		checkDir(home, "temp", "Temporary path");
		checkConfig(home);
	}

	/**
	 * 管理者キーをチェックする。
	 *
	 * @param adminkey 管理者キー
	 * @throws WorkflowException チェック違反があった場合
	 */
	static void checkAdmin(String adminkey)
	throws WorkflowException
	{
		//adminkeyの判定
		String admin = SystemChecker.getProperty(WorkflowEngineConstants.NAUTICA_ADMIN);
		if (adminkey == null) {
			throw new WorkflowException("Administrator key is not set.");
		}
		if (adminkey.equals(admin) == false) {
			throw new WorkflowException("Administrator key is not valid.");
		}
	}

	/**
	 * ワークフローエンジンホームをチェックする。
	 *
	 * @param home ワークフローエンジンホーム
	 * @throws WorkflowException チェック違反があった場合
	 */
	private static void checkHome(String home)
	throws WorkflowException
	{
		if (home == null || home.trim().equals("")) {
			throw new WorkflowException(WorkflowEngineConstants.NAUTICA_HOME
					+ " is not set.");
		}
		File dir = new File(home);
		if (! dir.exists()) {
			throw new WorkflowException(WorkflowEngineConstants.NAUTICA_HOME
					+ " is invalid path.: " + home);
		}
		if (! dir.isDirectory()) {
			throw new WorkflowException(WorkflowEngineConstants.NAUTICA_HOME
					+ " is not directory.: " + home);
		}
	}

	/**
	 * サブディレクトリをチェックする。
	 *
	 * @param home ワークフローエンジンホーム
	 * @param name ディレクトリ名
	 * @param label ラベル名
	 * @throws WorkflowException チェック違反があった場合
	 */
	private static void checkDir(String home, String name, String label)
	throws WorkflowException
	{
		String separator = System.getProperty("file.separator");
		String dirPath = home + separator + name;

		File dir = new File(dirPath);
		if (! dir.exists()) {
			throw new WorkflowException(label + " is not found.");
		}
		if (! dir.isDirectory()) {
			throw new WorkflowException(label + " is not directory.");
		}
	}

	/**
	 * 設定ファイルををチェックする。
	 *
	 * @param home ワークフローエンジンホーム
	 * @throws WorkflowException チェック違反があった場合
	 */
	private static void checkConfig(String home)
	throws WorkflowException
	{
		String separator = System.getProperty("file.separator");
		String confDirPath = home + separator + "config";
		String propFilePath = confDirPath + separator + PROP_NAME;
		File propFile = new File(propFilePath);
		if (! propFile.exists()) {
			throw new WorkflowException(PROP_NAME + " is not found.");
		}
		if (! propFile.isFile()) {
			throw new WorkflowException(PROP_NAME + " is not file.");
		}
		try {
			InputStream is = new FileInputStream(propFile);
			prop.load(is);
			is.close();
			prop.putAll(System.getProperties());
		} catch (IOException ex) {
			throw new WorkflowException(PROP_NAME + " cannot loaded.", ex);
		}

		String key = WorkflowEngineConstants.NAUTICA_ADMIN;
		String value = getProperty(key);
		if (value == null || value.trim().equals("")) {
			throw new WorkflowException(key + " property is not set.");
		}
	}
}
