/* $Id: InterWorkflowConfig.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.engine;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Hashtable;
import java.util.Map;

import jp.co.argo21.nautica.workflow.util.StringManager;

import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlOptions;

/**
 * エンジン間接続用の設定情報管理である。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
class InterWorkflowConfig
{
	/** 設定情報ファイル名 */
	private static final String CONFIG = "interworkflow.xml";
	
	/** 自エンジンID */
	private String engineID;
	/** 自エンジンパスワード */
	private String enginePassword;
	/** 登録されているリモートエンジン情報 */
	private Map<String, EngineInfo> remoteEngines = new Hashtable<String, EngineInfo>();
	
	/**
	 * エンジン間接続用の設定情報管理を生成する。
	 *
	 * @throws XmlException 設定ファイルに誤りがある場合
	 * @throws IOException 入力処理に失敗した場合
	 */
	public InterWorkflowConfig()
	throws XmlException, IOException
	{
		String home = SystemChecker.getProperty(WorkflowEngineConstants.NAUTICA_HOME);
		String separator = SystemChecker.getProperty("file.separator");
		String path = home
		            + separator + "config"
		            + separator + CONFIG;
		File confFile = new File(path);
		InputStream in = new FileInputStream(confFile);
		XmlOptions opts = new XmlOptions();
		opts.setCharacterEncoding("utf-8");

		EngineConfigDocument doc = EngineConfigDocument.Factory.parse(in, opts);
		in.close();
		
		load(doc);
	}

	/**
	 * 設定ファイルをロードする。
	 *
	 * @param doc XMLドキュメント
	 * @throws XmlException 設定ファイルに誤りがある場合
	 * @throws IOException 入力処理に失敗した場合
	 */
	private void load(EngineConfigDocument doc)
	throws XmlException, IOException
	{
		EngineConfigDocument.EngineConfig config = doc.getEngineConfig();
		EngineDocument.Engine local = config.getEngine();
		engineID = local.getId();
		enginePassword = local.getPassword();
		
		RemoteEnginesDocument.RemoteEngines remotes = config.getRemoteEngines();
		EngineDocument.Engine[] engines = remotes.getEngineArray();
		for (int i = 0; i < engines.length; i++) {
			String id = engines[i].getId();
			String pass = engines[i].getPassword();
			String url = engines[i].getUrl();
			if (id == null || id.trim().equals("")) {
				// エンジンIDが未設定です。
				String E0162 = StringManager.get("E0162");
				throw new XmlException(E0162);
			}
			
			EngineInfo info = new EngineInfo(id, pass, url);
			remoteEngines.put(id, info);
		}
	}
	
	/**
	 * engineIDを返す。
	 *
	 * @return engineIDの値
	 */
	String getLocalEngineID()
	{
		return engineID;
	}

	/**
	 * enginePasswordを返す。
	 *
	 * @return enginePasswordの値
	 */
	String getPassword()
	{
		return enginePassword;
	}

	/**
	 * エンジンIDに対応したリモートエンジン情報を返す。
	 *
	 * @param eid エンジンID 
	 * @return エンジン情報
	 */
	EngineInfo getEngineInfo(String eid)
	{
		return (EngineInfo)remoteEngines.get(eid);
	}
	
	/**
	 * エンジン情報
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.1 $
	 * @since   Nautica Workflow 1.0
	 */
	class EngineInfo
	{
		/** エンジンID */
		private String engineID;
		/** エンジンパスワード */
		private String enginePassword;
		/** 接続URL */
		private String engineUrl;

		/**
		 * エンジン情報を生成する。
		 *
		 * @param id エンジンID
		 * @param pass エンジンパスワード
		 * @param url 接続URL
		 */
		public EngineInfo(String id, String pass, String url)
		{
			this.engineID = id;
			this.enginePassword = pass;
			this.engineUrl = url;
		}

		/**
		 * エンジンIDを返す。
		 *
		 * @return エンジンID
		 */
		String getID()
		{
			return engineID;
		}

		/**
		 * エンジンパスワードを返す。
		 *
		 * @return エンジンパスワード
		 */
		String getPassword()
		{
			if (enginePassword == null) return "";
			return enginePassword;
		}

		/**
		 * 接続URLを返す。
		 *
		 * @return 接続URL
		 */
		String getURL()
		{
			if (engineUrl == null) return "";
			return engineUrl;
		}
	}
}
