/* $Id: EndActivityBehavior.java,v 1.4 2008/01/11 08:28:07 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.engine;

import java.sql.Timestamp;

import jp.co.argo21.nautica.workflow.dataaccess.ActivityBean;
import jp.co.argo21.nautica.workflow.dataaccess.ActivityDAO;
import jp.co.argo21.nautica.workflow.dataaccess.ConcurrencyBlockDAO;
import jp.co.argo21.nautica.workflow.dataaccess.WorkflowDAOFactory;
import jp.co.argo21.nautica.workflow.definition.ActivityDefinition;
import jp.co.argo21.nautica.workflow.definition.ProcessDefinition;
import jp.co.argo21.nautica.workflow.jms.WorkflowMessage;
import jp.co.argo21.nautica.workflow.wfmc.ActivityState;

/**
 * 終了アクティビティの状態ごとの動作を定義するクラスである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.4 $
 * @since   Nautica Workflow 1.0
 */
public class EndActivityBehavior
extends AbstractActivityBehavior
{
	private static final String CONCURRENCY_LOCK = "__CONCURRENCY_LOCK__";

	/**
	 * 終了アクティビティの状態ごとの動作を生成する。
	 */
	public EndActivityBehavior()
	{
	}

	/**
	 * 固有の中止処理を行う。
	 *
	 * @param pd プロセス定義
	 * @param ad アクティビティ定義
	 * @param proc プロセス
	 * @param act アクティビティ
	 * @throws Exception 任意の例外
	 * @see jp.co.argo21.nautica.workflow.engine.ActivityBehavior#abort(jp.co.argo21.nautica.workflow.definition.ProcessDefinition, jp.co.argo21.nautica.workflow.definition.ActivityDefinition, jp.co.argo21.nautica.workflow.engine.ProcessInternal, jp.co.argo21.nautica.workflow.engine.ActivityInternal, boolean)
	 */
	public void abort(ProcessDefinition pd, ActivityDefinition ad,
			ProcessInternal proc, ActivityInternal act, boolean checkChild)
	throws Exception
	{
		WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
		
		//該当アクティビティを中断させる。
		ActivityDAO actDAO = daoFactory.getActivityDAO();
		Timestamp ts = DataAccessManager.getAccessTime();
		actDAO.updateState(act.getBean(), ActivityState.CLOSED_ABORTED, ts);
	}

	/**
	 * 固有の完了処理を行う。
	 *
	 * @param pd プロセス定義
	 * @param ad アクティビティ定義
	 * @param proc プロセス
	 * @param act アクティビティ
	 * @throws Exception 任意の例外
	 * @see jp.co.argo21.nautica.workflow.engine.ActivityBehavior#complete(jp.co.argo21.nautica.workflow.definition.ProcessDefinition, jp.co.argo21.nautica.workflow.definition.ActivityDefinition, jp.co.argo21.nautica.workflow.engine.ProcessInternal, jp.co.argo21.nautica.workflow.engine.ActivityInternal)
	 */
	public void complete(ProcessDefinition pd, ActivityDefinition ad,
			ProcessInternal proc, ActivityInternal act)
	throws Exception
	{
		ActivityBean abean = act.getBean();
		String pid = abean.getProcessID();

		//該当アクティビティを完了させる。
		WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
		ActivityDAO actDAO = daoFactory.getActivityDAO();
		Timestamp ts = DataAccessManager.getAccessTime();
		actDAO.updateState(act.getBean(), ActivityState.CLOSED_COMPLETED, ts);

		//メッセージキューにプロセス完了要求登録
		WorkflowMessage eprmsg = new EndProcessRequestMessage(pid);
		SystemMessageTransporter mt = new SystemMessageTransporter(EndProcessRequestWatcher.NAME);
		mt.sendMessage(eprmsg);
	}

	/**
	 * 固有の開始処理を行う。
	 *
	 * @param pd プロセス定義
	 * @param ad アクティビティ定義
	 * @param proc プロセス
	 * @param act アクティビティ
	 * @throws Exception 任意の例外
	 * @see jp.co.argo21.nautica.workflow.engine.ActivityBehavior#start(jp.co.argo21.nautica.workflow.definition.ProcessDefinition, jp.co.argo21.nautica.workflow.definition.ActivityDefinition, jp.co.argo21.nautica.workflow.engine.ProcessInternal, jp.co.argo21.nautica.workflow.engine.ActivityInternal)
	 */
	public void start(ProcessDefinition pd, ActivityDefinition ad,
			ProcessInternal proc, ActivityInternal act)
	throws Exception
	{
		ActivityBean abean = act.getBean();
		String pid = abean.getProcessID();
		String aid = abean.getActivityID();

		//該当アクティビティを開始させる。
		WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
		ActivityDAO actDAO = daoFactory.getActivityDAO();
		Timestamp ts = DataAccessManager.getAccessTime();
		actDAO.updateState(act.getBean(), ActivityState.OPEN_RUNNING, ts);
		
		LockManager lockManager = LockManager.getInstance();
		LockManager.Lock lock = lockManager.getLock(CONCURRENCY_LOCK);
		synchronized(lock) {
			ConcurrencyBlockDAO cbDAO = daoFactory.getConcurrencyBlockDAO();
			int actcount = actDAO.findOpeningActivityCountByProcessID(pid);
			int sbcount = cbDAO.findSubblockExistenceByProcessID(pid);
			
			//プロセス内のアクティビティがすべて完了した場合
			if (actcount == 1 && sbcount == 0) {
				//自分自身しかオープンされていない
				//メッセージキューに完了要求登録
				WorkflowMessage earmsg = new EndActivityRequestMessage(pid, aid);
				SystemMessageTransporter mt = new SystemMessageTransporter(EndActivityRequestWatcher.NAME);
				mt.sendMessage(earmsg);
			}
		}
	}

	/**
	 * 固有の停止処理を行う。
	 *
	 * @param pd プロセス定義
	 * @param ad アクティビティ定義
	 * @param proc プロセス
	 * @param act アクティビティ
	 * @throws Exception 任意の例外
	 * @see jp.co.argo21.nautica.workflow.engine.ActivityBehavior#terminate(jp.co.argo21.nautica.workflow.definition.ProcessDefinition, jp.co.argo21.nautica.workflow.definition.ActivityDefinition, jp.co.argo21.nautica.workflow.engine.ProcessInternal, jp.co.argo21.nautica.workflow.engine.ActivityInternal, boolean)
	 */
	public void terminate(ProcessDefinition pd, ActivityDefinition ad,
			ProcessInternal proc, ActivityInternal act, boolean checkChild)
	throws Exception
	{
		ActivityBean abean = act.getBean();
		String pid = abean.getProcessID();

		//該当アクティビティを中止させる。
		WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
		ActivityDAO actDAO = daoFactory.getActivityDAO();
		Timestamp ts = DataAccessManager.getAccessTime();
		actDAO.updateState(act.getBean(), ActivityState.CLOSED_TERMINATED, ts);

		//メッセージキューにプロセス完了要求登録
		WorkflowMessage eprmsg = new EndProcessRequestMessage(pid, ActivityState.CLOSED_TERMINATED);
		SystemMessageTransporter mt = new SystemMessageTransporter(EndProcessRequestWatcher.NAME);
		mt.sendMessage(eprmsg);
	}
}
