/* $Id: DefinitionRepositoryImpl.java,v 1.3 2007/11/26 08:51:40 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.engine;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.naming.NamingException;

import jp.co.argo21.nautica.workflow.dataaccess.DefinitionBean;
import jp.co.argo21.nautica.workflow.dataaccess.DefinitionDAO;
import jp.co.argo21.nautica.workflow.dataaccess.WorkflowDAOFactory;
import jp.co.argo21.nautica.workflow.definition.PackageDefinition;
import jp.co.argo21.nautica.workflow.definition.ProcessDefinition;
import jp.co.argo21.nautica.workflow.definition.impl.DefinitionLoader;
import jp.co.argo21.nautica.workflow.omg.WorkflowException;
import jp.co.argo21.nautica.workflow.security.SessionManager;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.DefinitionRepository;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidProcessDefinitionException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidSessionException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidStateException;
import jp.co.argo21.nautica.workflow.wfmc.ProcDef;
import jp.co.argo21.nautica.workflow.wfmc.ProcessDefinitionState;
import jp.co.argo21.nautica.workflow.wfmc.TransitionNotAllowedException;

import org.apache.log4j.Logger;

/**
 * ローカルのJavaVMで動作する定義リポジトリ実装クラスである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.3 $
 * @since   Nautica Workflow 1.0
 */
public class DefinitionRepositoryImpl 
implements DefinitionRepository, RepositoryLoader
{
	/** 定義リポジトリパス名 */
	private static final String REPOSITORY_DIR = "repository";
	
	/** リポジトリローダの公開名 */
	private static final String WORKFLOW_REPOSITORY_LOADER = "RepositoryLoader";

	/** エンジンログ */
	private static Logger eLog = LogManager.getEngineLogger();
	
	/** 監査ログ */
	private static Logger audit = LogManager.getAuditLogger();

	/** 定義ローダ */
	private DefinitionLoader loader;

	/** 定義キャッシュ */
	private DefinitionCache cache;
	
	/**
	 * リポジトリパスを取得する。
	 *
	 * @return リポジトリパス
	 * @throws WorkflowException 取得中に例外が起こった場合
	 */
	private static String getRepositoryPath()
	throws WorkflowException
	{
		String home = SystemChecker.getProperty(WorkflowEngineConstants.NAUTICA_HOME);
		String separator = SystemChecker.getProperty("file.separator");
		String path = home + separator + REPOSITORY_DIR;
		File confFile = new File(path);
	
		if (! confFile.exists()) {
			// 定義リポジトリパスが存在しません。
			String msg = StringManager.get("E0052") + "(path=" + path + ")";
			eLog.error(msg);
			throw new WorkflowException(msg);
		}
		if (! confFile.isDirectory()) {
			// 定義リポジトリパスはディレクトリではありません。
			String msg = StringManager.get("E0053") + "(path=" + path + ")";
			eLog.error(msg);
			throw new WorkflowException(msg);
		}
		return path;
	}

	/**
	 * DefinitionRepositoryImplを生成する。
	 */
	DefinitionRepositoryImpl()
	throws WorkflowException
	{
		loader = new DefinitionLoader(getRepositoryPath());
		cache = new DefinitionCache();
		
		// リポジトリローダを公開
		try {
		    String scope = SystemChecker.getProperty(WorkflowEngineConstants.NAUTICA_SCOPE);
			if (scope == null) scope = "";
			scope = scope.trim();

			eLog.info(StringManager.get("I0045")
			        + "(" + scope + WORKFLOW_REPOSITORY_LOADER + ")");
			
		} catch (Exception e) {
		    String errMsg = StringManager.get("E0072");
		    eLog.error(errMsg);
		    throw new WorkflowException(errMsg, e);
		}
	}


	/**
	 * 指定された定義の登録処理を行い、エンジンに認識させる。
	 *
	 * @param session セッションID
	 * @param pdid ワークフロー定義ID
	 * @param path ワークフロー定義の相対パス
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws WorkflowException プロセスインスタンスIDが有効でない場合
	 * @see jp.co.argo21.nautica.workflow.wfmc.DefinitionRepository#registerDefinition(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void registerDefinition(String session, String pdid, String path)
	throws InvalidSessionException, WorkflowException
	{
		String argInfo = "(NAUTICA API = registerDefinition,"
		 + "session = [" + session + "],"
		 + "pdid = [" + pdid + "],"
		 + "path = [" + path + "])";

		PackageDefinition packageDefinition = null;
		try {
			DataAccessManager.begin(false);
				
			validateSession(session);
				
			if (pdid == null || pdid.trim().equals("")) {
				// ワークフロー定義名が設定されていません。
				String E0105 = StringManager.get("E0105");
				throw new WorkflowException(E0105);
			}

			if (path == null || path.trim().equals("")) {
				// ワークフロー定義の相対パスが設定されていません。
				String E0106 = StringManager.get("E0106");
				throw new WorkflowException(E0106);
			}
		
			packageDefinition = loader.load(path);
	
			WorkflowDAOFactory daoFactory
				= DataAccessManager.getDAOFactory();
			DefinitionDAO dao = daoFactory.getDefinitionDAO();
	
			Iterator<ProcessDefinition> it = packageDefinition.getProcesses().iterator();
			for (int i = 0; it.hasNext(); i++) {
				ProcessDefinition def = it.next();
				String ipdid = def.getID();
				
				DefinitionBean bean = new DefinitionBean();
				bean.setProcessDefinitionID(pdid + "-" + i);
				bean.setPath(path);
				bean.setInternalPDID(ipdid);
				bean.setRegisterDate(new Timestamp(System.currentTimeMillis()));
				bean.setState(ProcessDefinitionState.ENABLE.toInt());
							
				dao.insert(bean);
			}
				
			DataAccessManager.commit();
	
		} catch (InvalidSessionException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (WorkflowException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// 定義の登録処理でエラーが起こりました。
			String E0107 = StringManager.get("E0107") + argInfo;
			eLog.error(E0107, ex);
			throw new WorkflowException(E0107, ex);
		} finally {
			try { DataAccessManager.close(); } catch (Exception ex) { /* Ignore */ }
		}
	}

	/**
	 * 条件にマッチしたプロセス定義一覧を返す。
	 *
	 * @param session セッションID
	 * @param filter 検索用フィルター
	 * @return 検索されたプロセス定義概要の配列
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @see jp.co.argo21.nautica.workflow.wfmc.DefinitionRepository#getProcessDefinitions(java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public ProcDef[] getProcessDefinitions(
			String session, Filter filter)
	throws InvalidSessionException, InvalidFilterException
	{		
		String argInfo = "(WfMC API = getProcessDefinitions,"
			 + "session = [" + session + "],"
			 + "filter = [" + filter + "])";

		try {
			DataAccessManager.begin(false);
			
			validateSession(session);

			WorkflowDAOFactory daoFactory
				= DataAccessManager.getDAOFactory();
			DefinitionDAO dao = daoFactory.getDefinitionDAO();
			List<DefinitionBean> list = dao.findByFilter(filter);

			List<ProcDef> procDefs = new ArrayList<ProcDef>();
			Iterator<DefinitionBean> it = list.iterator();
			for (int i = 0; it.hasNext(); i++) {
				DefinitionBean bean = (DefinitionBean)it.next();
				ProcessDefinition def = cache.getProcessDefinition(bean.getProcessDefinitionID());
				ProcDefImpl impl = new ProcDefImpl();
				impl.setID(bean.getProcessDefinitionID());
				impl.setName(def.getName());
				impl.setDescription(def.getDescription());
				impl.setState(ProcessDefinitionState.getState(bean.getState()));
				procDefs.add(impl);
			}
			
			DataAccessManager.commit();

			return (ProcDef[])procDefs.toArray(new ProcDef[procDefs.size()]);
		} catch (InvalidSessionException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (InvalidFilterException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (WorkflowException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// プロセス定義一覧の取得処理でエラーが起こりました。
			String E0108 = StringManager.get("E0108") + argInfo;
			eLog.error(E0108, ex);
			throw new InvalidFilterException(E0108, ex);
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// プロセス定義一覧の取得処理でエラーが起こりました。
			String E0108 = StringManager.get("E0108") + argInfo;
			eLog.error(E0108, ex);
			throw new InvalidFilterException(E0108, ex);
		} finally {
			try { DataAccessManager.close(); } catch (Exception ex) { /* Ignore */ }
		}	
	}

	/**
	 * プロセス定義IDで特定された定義の現在の状態から遷移可能な
	 * プロセス定義状態一覧を返す。
	 *
	 * @param session セッションID
	 * @param pdid プロセス定義ID
	 * @return プロセス定義状態の配列
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessDefinitionException 指定されたプロセス定義IDが有効でない場合
	 * @see jp.co.argo21.nautica.workflow.wfmc.DefinitionRepository#getProcessDefinitionStates(java.lang.String, java.lang.String)
	 */
	public ProcessDefinitionState[] getProcessDefinitionStates(
			String session,String pdid)
	throws InvalidSessionException, InvalidProcessDefinitionException
	{
		String argInfo = "(WfMC API = getProcessDefinitionStates,"
			 + "session = [" + session + "],"
			 + "pdid = [" + pdid + "])";
	
		try {
			DataAccessManager.begin(false);
			
			validateSession(session);

			checkProcessDefinitionID(pdid);
	
			DefinitionBean bean = cache.getDefinitionBean(pdid);
				
			ProcessDefinitionState[] states = new ProcessDefinitionState[1];
				
			ProcessDefinitionState currentState
				= ProcessDefinitionState.getState(bean.getState());
				
			if (ProcessDefinitionState.ENABLE.equals(currentState)) {
				states[0] = ProcessDefinitionState.DISABLE;
			} else {
				states[0] = ProcessDefinitionState.ENABLE;
			}
				
			DataAccessManager.commit();
	
			return states;
		} catch (InvalidSessionException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (InvalidProcessDefinitionException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (WorkflowException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// プロセス定義状態の取得処理でエラーが起こりました。
			String E0109 = StringManager.get("E0109") + argInfo;
			eLog.error(E0109, ex);
			throw new InvalidProcessDefinitionException(E0109, ex);
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// プロセス定義状態の取得処理でエラーが起こりました。
			String E0109 = StringManager.get("E0109") + argInfo;
			eLog.error(E0109, ex);
			throw new InvalidProcessDefinitionException(E0109, ex);
		} finally {
			try { DataAccessManager.close(); } catch (Exception ex) { /* Ignore */ }
		}	
	}

	/**
	 * プロセス定義状態を変更する。
	 *
	 * @param session セッションID
	 * @param pdid プロセス定義ID
	 * @param state
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessDefinitionException 指定されたプロセス定義IDが有効でない場合
	 * @throws InvalidStateException 指定された状態が有効でない場合
	 * @throws TransitionNotAllowedException 状態の変更が許可されていない場合
	 * @see jp.co.argo21.nautica.workflow.wfmc.DefinitionRepository#changeProcessDefinitionState(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.ProcessDefinitionState)
	 */
	public void changeProcessDefinitionState(String session,
		String pdid, ProcessDefinitionState state)
	throws InvalidSessionException, InvalidProcessDefinitionException,
		InvalidStateException, TransitionNotAllowedException
	{
		String argInfo = "(WfMC API = changeProcessDefinitionState,"
			 + "session = [" + session + "],"
			 + "pdid = [" + pdid + "],"
			 + "state = [" + state.toString() + "])";
	
		try {
			DataAccessManager.begin(false);
			
			validateSession(session);

			checkProcessDefinitionID(pdid);
		
			if (state == null) {
				// プロセス定義状態が設定されていません。
				String E0110 = StringManager.get("E0110");
				throw new InvalidStateException(E0110);
			}

			setProcessDefinitionState(pdid, state);
				
			DataAccessManager.commit();
			
			// 監査ログ出力
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix("", "WMChangedProcessDifinitionState", session),
		            new AuditSuffix());
		    data.setDataElement("ProcessDefinitionID", pdid);
		    data.setDataElement("NewProcessDefinitionState", state);		    
			audit.info(data);
	
		} catch (InvalidSessionException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (InvalidProcessDefinitionException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (InvalidStateException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (TransitionNotAllowedException ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			String msg = ex.getMessage() + argInfo;
			eLog.error(msg, ex);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// プロセス定義状態の変更に失敗しました。
			String E0186 = StringManager.get("E0186") + argInfo;
			eLog.error(E0186, ex);
			throw new TransitionNotAllowedException(E0186, ex);
		} finally {
			try { DataAccessManager.close(); } catch (Exception ex) { /* Ignore */ }
		}
	}
	
	/**
     * 指定されたパスの定義ファイルを文字列で返す。
     * なければ null を返す。
     * 何らかの例外が発生した場合も null を返す。
     * 
     * @param session      セッションID
     * @param adminKey 管理用パスワード
     * @param path     パス名
     * @return 定義ファイルの文字列
     */
	public String getDefinition(String session, String adminKey, String path) 
	throws InvalidSessionException
	{
		String argInfo = "(NAUTICA API = getDefinition,"
			 + "session = [" + session + "],"
			 + "adminKey = [" + adminKey + "],"
			 + "path = [" + path + "])";
	
	    String separator = SystemChecker.getProperty("file.separator");
	    String filepath = SystemChecker.getProperty(WorkflowEngineConstants.NAUTICA_HOME) + separator
                        + REPOSITORY_DIR + separator
                        + path;
	    
	    BufferedReader br = null;
	    try {
			validateSession(session);
			SystemChecker.checkAdmin(adminKey);
			
			// ファイルを取得し、読み込んで文字列を返す
			String lineseparator = SystemChecker.getProperty("line.separator");
			br = new BufferedReader(
					new InputStreamReader(
							new FileInputStream(filepath), "utf-8"));
			StringBuffer definition = new StringBuffer();
			while (br.ready()) {
			    definition.append(br.readLine());
			    definition.append(lineseparator);
			}
			return definition.toString();
	    } catch (InvalidSessionException ex) {
			String W0000 = ex.getMessage() + argInfo;
			eLog.warn(W0000, ex);
			throw ex;
	    } catch (WorkflowException ex) {
			String W0000 = ex.getMessage() + argInfo;
			eLog.warn(W0000, ex);
			return null;
	    } catch (IOException ex) {
			// 定義ファイル名検索中に例外を捕捉しましたが、処理を継続します。
			String W0003 = StringManager.get("W0003") + argInfo;
			eLog.warn(W0003, ex);
			return null;
	    } finally {
	    	if (br != null) {
	    		try { br.close(); } catch (IOException e) { /* ignore */ }
	        }
	    }
	}
	
	/**
     * 指定されたプロセス定義名に対応するパス名を返す。
     * なければ null を返す。
     * 何らかの例外が発生した場合も null を返す。
     * 
     * @param session    セッションID
     * @param pdname プロセス定義名
     * @return パス名
	 */
	public String getDefinitionPath(String session, String pdname)
	throws InvalidSessionException
	{
		String argInfo = "(NAUTICA API = getDefinitionPath,"
			 + "session = [" + session + "],"
			 + "pdname = [" + pdname + "])";
	
	    try {
			validateSession(session);
			
			WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
			DefinitionDAO defDAO = daoFactory.getDefinitionDAO();
			DefinitionBean defbean = defDAO.findByPrimaryKey(pdname);
			
			if (defbean != null) {
				return defbean.getPath();
			}
			
			return null;
	    } catch (InvalidSessionException ex) {
			String W0000 = ex.getMessage() + argInfo;
			eLog.warn(W0000, ex);
			throw ex;
	    } catch (WorkflowException ex) {
			String W0000 = ex.getMessage() + argInfo;
			eLog.warn(W0000, ex);
			return null;
	    } catch (Exception ex) {
			// 定義ファイルパス名検索中に例外を捕捉しましたが、処理を継続します。
			String W0004 = StringManager.get("W0004") + argInfo;
			eLog.warn(W0004, ex);
			return null;
	    }
	}
	
	/**
	 * リポジトリローダをアンバインドする。
	 * 
	 * @throws NamingException ネーミング例外が発生した場合
	 */
	public void unbind() throws NamingException
	{
		eLog.info(StringManager.get("I0046"));
	}
	
	/**
 	 * スーパーバイザー用。パッケージ定義を取得する。
	 *
	 * @param pdid プロセス定義ID
	 * @return パッケージ定義
	 * @throws WorkflowException 取得処理中に例外が起こった場合
	 */
	PackageDefinition getPackageDefinition(String pdid)
	throws WorkflowException
	{
		return cache.getPackageDefinition(pdid);
	}
	
	/**
 	 * スーパーバイザー用。プロセス定義を取得する。
	 *
	 * @param pdid プロセス定義ID
	 * @return プロセス定義
	 * @throws WorkflowException 取得処理中に例外が起こった場合
	 */
	ProcessDefinition getProcessDefinition(String pdid)
	throws WorkflowException
	{
		return cache.getProcessDefinition(pdid);
	}
	
	/**
 	 * スーパーバイザー用。プロセス定義状態を取得する。
	 *
	 * @param pdid プロセス定義ID
	 * @return プロセス定義状態
	 * @throws WorkflowException 取得処理中に例外が起こった場合
	 */
	ProcessDefinitionState getProcessDefinitionState(String pdid)
	throws WorkflowException
	{
		try {
			DefinitionBean bean = cache.getDefinitionBean(pdid);
			int stateNum = bean.getState();
			return ProcessDefinitionState.getState(stateNum);
		} catch (Exception ex) {
			// プロセス定義状態の取得に失敗しました。
			String E0187 = StringManager.get("E0187");
			throw new WorkflowException(E0187, ex);
		}
	}

	/**
	 * スーパーバイザー用。プロセス定義状態を設定する。
	 *
	 * @param pdid プロセス定義ID
	 * @param state プロセス定義状態
	 * @throws InvalidProcessDefinitionException 指定されたプロセス定義IDが有効でない場合
	 * @throws InvalidStateException 指定された状態が有効でない場合
	 * @throws TransitionNotAllowedException 状態の変更が許可されていない場合
	 */
	void setProcessDefinitionState(
		String pdid, ProcessDefinitionState state)
	throws InvalidProcessDefinitionException, InvalidStateException,
		TransitionNotAllowedException
	{
		try {
			ProcessDefinitionState currentState = getProcessDefinitionState(pdid);
			ProcessDefinitionState.checkTransition(currentState, state);
	
			WorkflowDAOFactory daoFactory
				= DataAccessManager.getDAOFactory();
			DefinitionDAO dao = daoFactory.getDefinitionDAO();
			dao.updateState(pdid, currentState.toInt(), state.toInt());
		} catch (InvalidProcessDefinitionException ex) {
			throw ex;
		} catch (InvalidStateException ex) {
			throw ex;
		} catch (TransitionNotAllowedException ex) {
			throw ex;
		} catch (Exception ex) {
			// プロセス定義状態の変更に失敗しました。
			String E0186 = StringManager.get("E0186");
			throw new TransitionNotAllowedException(E0186, ex);
		}		
	}
	
	/**
	 * プロセス定義IDのチェックを行う。
	 *
	 * @param pdid プロセス定義ID
	 * @throws InvalidProcessDefinitionException チェックに失敗した場合
	 */
	private void checkProcessDefinitionID(String pdid)
	throws InvalidProcessDefinitionException
	{
		if (pdid == null || pdid.trim().equals("")) {
			// プロセス定義IDが未設定です。
			String E0188 = StringManager.get("E0188");
			throw new InvalidProcessDefinitionException(E0188);
		}
	}
	
	/**
	 * セッションを検証する。
	 *
	 * @param session セッションID
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 */
	private void validateSession(String session)
	throws InvalidSessionException
	{
		if (session == null || session.trim().equals("")) {
			// セッションIDが未設定です。
			String E0170 = StringManager.get("E0170");
			throw new InvalidSessionException(E0170);
		}

		SessionManagerFactory factory = SessionManagerFactory.getInstance();
		SessionManager manager = factory.getSessionManager();
		manager.validateSession(session);
	}
}
