/* $Id: ActivityBehaviorFactory.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.engine;

import java.util.Hashtable;
import java.util.Map;

import jp.co.argo21.nautica.workflow.definition.ActivityDefinition;
import jp.co.argo21.nautica.workflow.omg.WorkflowException;
import jp.co.argo21.nautica.workflow.util.StringManager;

import org.apache.log4j.Logger;

/**
 * アクティビティ動作のファクトリである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
class ActivityBehaviorFactory
{
	/** エンジンログ */
	private static Logger eLog = LogManager.getEngineLogger();
	
	/** ActivityBehaviorFactoryのシングルトン */
	private static ActivityBehaviorFactory singleton;

	/** アクティビティ動作用ハッシュテーブル */
	private Map<String, ActivityBehavior> activityBehaviors = new Hashtable<String, ActivityBehavior>();
	
	/**
	 * ActivityBehaviorFactoryの生成と初期化を行う。
	 *
	 * @throws Exception 任意の例外
	 */
	static void create() throws Exception
	{
		singleton = new ActivityBehaviorFactory();
		try {
			singleton.init();
		} catch (Exception ex) {
			// アクティビティ動作のファクトリの生成に失敗しました。
			String msg = StringManager.get("E0051");
			eLog.error(msg);
			throw new WorkflowException(msg);
		}
	}

	/**
	 * ActivityBehaviorFactoryのインスタンスを返す。
	 *
	 * @return ActivityBehaviorFactory
	 */
	static ActivityBehaviorFactory getInstance()
	{
		return singleton;
	}
	
	/**
	 * ActivityBehaviorFactoryを生成する。
	 */
	private ActivityBehaviorFactory()
	{
	}
	
	/**
	 * ActivityBehaviorFactoryを初期化する。
	 *
	 * @throws Exception 任意の例外
	 */
	private void init() throws Exception
	{
		activityBehaviors.put(ActivityDefinition.START, new StartActivityBehavior());
		activityBehaviors.put(ActivityDefinition.INTERACTIVE, new InteractiveActivityBehavior());
		activityBehaviors.put(ActivityDefinition.AUTOMATIC, new AutomaticActivityBehavior());
		activityBehaviors.put(ActivityDefinition.SUBPROCESS, new SubprocessActivityBehavior());
		activityBehaviors.put(ActivityDefinition.CONCUR_START, new ConcurrentStartActivityBehavior());
		activityBehaviors.put(ActivityDefinition.CONCUR_END, new ConcurrentEndActivityBehavior());
		activityBehaviors.put(ActivityDefinition.END, new EndActivityBehavior());
	}

	/**
	 * 指定された型に対応するアクティビティ動作を返す。
	 *
	 * @param type アクティビティの型
	 * @return アクティビティ動作
	 */
	public ActivityBehavior getActivityBehavior(String type)
	{
		return (ActivityBehavior)activityBehaviors.get(type);
	}
}
